#pragma once

#include "config.h"

#include <drive/telematics/protocol/actions.h>

#include <library/cpp/tvmauth/type.h>

#include <rtline/library/scheduler/registrator.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/network/neh.h>

#include <util/system/spinlock.h>

namespace NRTLine {
    class TNehIndexingClient;
    class TNehSearchClient;
}

namespace NTvmAuth {
    class TTvmClient;
}

namespace NDrive {
    class TTelematicsClient {
    public:
        enum class EStatus {
            Unknown             /* "unknown" */,
            ConnectionNotFound  /* "connection_not_found" */,
            ConnectionDead      /* "connection_dead" */,

            New                 /* "new" */,
            Processing          /* "processing" */,

            Success             /* "success" */,
            Failure             /* "failure" */,
            Incorrect           /* "incorrect" */,

            Lost                /* "lost" */,
            Retry               /* "retry" */,
            Terminated          /* "terminated" */,
            Timeouted           /* "timeouted" */,
            InternalError       /* "internal_error" */,
        };

        struct TEvent {
            NJson::TJsonValue Data;
            TInstant Timestamp;

            inline bool operator <(const TEvent& other) const {
                return Timestamp < other.Timestamp;
            }
        };
        using TEvents = TSet<TEvent>;

        class TContext {
        public:
            TContext(TStringBuf baseId);

            TEvents ExtractEvents() {
                auto guard = Guard(Lock);
                return std::move(Events);
            }
            TEvents GetEvents() const {
                auto guard = Guard(Lock);
                return Events;
            }
            TString GetIMEI() const {
                auto guard = Guard(Lock);
                return IMEI;
            }
            TString GetShard() const {
                auto guard = Guard(Lock);
                return Shard;
            }
            EStatus GetStatus() const {
                return Status;
            }
            TDuration GetWaitConnectionTimeout() const {
                return WaitConnectionTimeout;
            }
            bool GetOptimisticRequestSucceeded() const {
                return OptimisticRequestSucceeded;
            }
            bool GetTerminationProcessed() const {
                return TerminationProcessed;
            }

            void AddEvent(NJson::TJsonValue&& data, TInstant timestamp = TInstant::Zero());
            void SetEvents(TEvents&& value);
            void SetIMEI(TString value);
            void SetShard(TString value);
            void SetStatus(EStatus value);
            void SetWaitConnectionTimeout(TDuration value);
            void SetOptimisticRequestSucceeded(bool value);
            void SetTerminationProcessed(bool value);

        private:
            const TString Id;

            TAdaptiveLock Lock;
            TEvents Events;
            TString IMEI;
            TString Shard;
            EStatus Status = EStatus::Unknown;
            TDuration WaitConnectionTimeout;

            bool OptimisticRequestSucceeded = false;
            bool TerminationProcessed = false;
        };
        using TContextPtr = TAtomicSharedPtr<TContext>;

        class IResponse {
        public:
            TString Shard;

        public:
            virtual ~IResponse() = default;

            virtual EStatus GetStatus() const = 0;
            virtual bool Final() const = 0;
            virtual void FromJson(const NJson::TJsonValue& content) = 0;

        public:
            template <class T>
            static THolder<T> ParseAs(const NJson::TJsonValue& content);
            static THolder<IResponse> Parse(const NJson::TJsonValue& content);
        };
        using TResponsePtr = TAtomicSharedPtr<IResponse>;
        using TAsyncResponse = NThreading::TFuture<TResponsePtr>;
        using TAsyncBroadcast = TMap<TString, TAsyncResponse>;

        using TAsyncPushResult = NThreading::TFuture<void>;

        struct TConnection {
            TString IMEI;
            TString Client;
            TString Server;
            TInstant Created;
            TInstant Blackbox;
            TInstant Heartbeat;
            bool Alive = false;
        };
        using TConnections = TVector<TConnection>;
        using TShardedConnections = TMap<TString, TConnections>;
        struct TConnectionsList {
            TContextPtr Context;
            TShardedConnections ShardedConnections;
            TVector<TString> UnansweredShards;
        };
        using TAsyncConnectionsList = NThreading::TFuture<TConnectionsList>;

        struct TConnectionListResponse: public IResponse {
        public:
            TConnections Connections;

        public:
            virtual EStatus GetStatus() const override {
                return EStatus::Success;
            }
            virtual bool Final() const override {
                return true;
            }
            virtual void FromJson(const NJson::TJsonValue& content) override;
        };

        struct TErrorResponse: public IResponse {
        public:
            TString Message;
            EStatus Status;

        public:
            TErrorResponse(EStatus status, TString message)
                : Message(std::move(message))
                , Status(status)
            {
            }

            virtual EStatus GetStatus() const override {
                return Status;
            }
            virtual bool Final() const override {
                return true;
            }
            virtual void FromJson(const NJson::TJsonValue& content) override {
                Y_UNUSED(content);
            }
        };

        struct TCommonResponse: public IResponse {
        private:
            using TBase = IResponse;

        public:
            TString Id;
            TString IMEI;
            TString Message;
            TEvents Events;
            TInstant Created;
            TInstant Started;
            TInstant Heartbeat;
            TInstant Finished;
            EStatus Status = EStatus::Unknown;

        public:
            virtual EStatus GetStatus() const override {
                return Status;
            }
            virtual bool Final() const override {
                return Status != EStatus::New && Status != EStatus::Processing;
            }
            virtual void FromJson(const NJson::TJsonValue& content) override;
            virtual void Merge(const TCommonResponse& other);

        private:
            auto Tuple() const {
                return std::tie(Id, IMEI, Message, Events, Created, Started, Heartbeat, Finished, Status);
            }
            auto Tuple() {
                return std::tie(Id, IMEI, Message, Events, Created, Started, Heartbeat, Finished, Status);
            }
        };

        struct TCanResponse: public TCommonResponse {
        private:
            using TBase = TCommonResponse;

        public:
            NDrive::NVega::TCanResponse::TFrames Frames;

        public:
            virtual void FromJson(const NJson::TJsonValue& content) override;
        };

        struct TCommandResponse: public TCommonResponse {
        private:
            using TBase = TCommonResponse;

        public:
            NDrive::NVega::ECommandCode Code = NDrive::NVega::ECommandCode::UNKNOWN;
            NDrive::NVega::TCommandResponse::EResult Result = NDrive::NVega::TCommandResponse::UNKNOWN;

        public:
            virtual void FromJson(const NJson::TJsonValue& content) override;
        };
        struct TGetParameterResponse: public TCommandResponse {
        private:
            using TBase = TCommandResponse;

        public:
            TSensor Sensor;

        public:
            virtual void FromJson(const NJson::TJsonValue& content) override;
        };

        struct TDownloadFileResponse: public TCommonResponse {
        private:
            using TBase = TCommonResponse;
            using TResult = NVega::TGetFileResponse::EResult;

        public:
            TResult Result = TResult::UNKNOWN_ERROR;
            TBuffer Data;

        public:
            virtual void FromJson(const NJson::TJsonValue& content) override;
        };

        class THandler;
        using THandlerCallback = std::function<void(const THandler&)>;

        class THandler {
        public:
            THandler() = default;
            THandler(TString id, TContextPtr context, TAsyncResponse&& response)
                : Id(id)
                , Context(context)
                , Response(std::move(response))
            {
            }

            explicit operator bool() const {
                return !Id.empty();
            }

            const TString& GetId() const {
                return Id;
            }
            TEvents GetEvents() const {
                return GetEventsImpl();
            }
            TString GetImei() const {
                return GetImeiImpl();
            }
            TString GetMessage() const {
                return GetMessageImpl();
            }
            TString GetShard() const {
                return GetShardImpl();
            }
            EStatus GetStatus() const {
                return GetStatusImpl();
            }
            NThreading::TFuture<void> GetFuture() const {
                return Response.IgnoreResult();
            }
            bool GetOptimisticRequestSucceeded() const {
                return Context ? Context->GetOptimisticRequestSucceeded() : false;
            }
            bool GetTerminationProcessed() const {
                return Context ? Context->GetTerminationProcessed() : false;
            }

            template <class T>
            TAtomicSharedPtr<const T> GetResponse() const {
                return std::dynamic_pointer_cast<T>(GetResponseImpl());
            }

            const THandler& Subscribe(
                THandlerCallback&& callback,
                TInstant deadline = TInstant::Zero(),
                const NThreading::TFuture<void>& precondition = {}
            ) const;
            const THandler& WaitAndEnsureSuccess(TInstant deadline = TInstant::Max()) const;

        private:
            TResponsePtr GetResponseImpl() const;
            TEvents GetEventsImpl() const;
            TString GetImeiImpl() const;
            TString GetMessageImpl() const;
            TString GetShardImpl() const;
            EStatus GetStatusImpl() const;

        private:
            TString Id;

            TContextPtr Context;
            TAsyncResponse Response;
        };
        using THandlers = TVector<THandler>;

        struct TOptions {
            NSimpleMeta::TConfig MetaConfig = NSimpleMeta::TConfig::ForRequester();
            NTvmAuth::TTvmId DestinationClientId = 0;
            TString ReqIdClass = "API2";

            TString InfoFetchHost;
            ui16 InfoFetchPort = 17000;
            TString InfoFetchService = "drive_cache";
            TString InfoSpMetaSearch;
            NTvmAuth::TTvmId InfoSaasTvmId = 0;

            TString InfoPushHost;
            ui16 InfoPushPort = 80;
            TString InfoPushToken;

            TDuration BroadcastTimeout = TDuration::MilliSeconds(100);
            TDuration RequestTimeout = TDuration::MilliSeconds(250);
            TDuration DefaultCommandTimeout = TDuration::Seconds(100);
            TDuration DefaultWaitConnectionTimeout = TDuration::Zero();
            TDuration DefaultWaitTaskTimeout = TDuration::Seconds(20);
            TDuration InfoLifetime = TDuration::Days(3);
            TDuration ConnectionsShardsUpdatePeriod = TDuration::Seconds(42);
        };

    public:
        TTelematicsClient(
            TAtomicSharedPtr<IServiceDiscovery> sd,
            const TOptions& options = Default<TOptions>(),
            const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm = nullptr
        );
        ~TTelematicsClient();

        THandler Command(
            TStringBuf imei,
            const NDrive::NVega::TCommand& command,
            TDuration timeout = TDuration::Zero(),
            TDuration waitConnectionTimeout = TDuration::Zero(),
            TStringBuf externalId = {}
        ) const;
        THandler Download(
            TStringBuf imei,
            TStringBuf filename,
            TDuration timeout = TDuration::Zero(),
            TDuration waitConnection = TDuration::Zero(),
            TStringBuf externalId = {}
        ) const;
        THandler Upload(
            TStringBuf imei,
            TStringBuf filename,
            TBuffer&& content,
            TDuration timeout = TDuration::Zero(),
            TDuration waitConnectionTimeout = TDuration::Zero(),
            TStringBuf externalId = {}
        ) const;
        THandler FetchInfo(const TString& id) const;
        THandler Restore(const TString& id) const;

        TString GetCachedShard(TStringBuf imei) const;

        TAsyncConnectionsList ListConnections() const;

    private:
        class TConnectionsUpdater;
        using TConnectionsShards = TMap<TString, TString>;
        using TConnectionsShardsPtr = TAtomicSharedPtr<TConnectionsShards>;
        using TConnectionsShardsConstPtr = TAtomicSharedPtr<const TConnectionsShards>;

    private:
        TAsyncBroadcast Parse(const TMap<TString, NThreading::TFuture<NNeh::THttpReply>>& replies, TContextPtr context = nullptr) const;
        TAsyncResponse Parse(const NThreading::TFuture<NNeh::THttpReply>& reply, const TString& shard = {}, TContextPtr context = nullptr) const;
        TResponsePtr Parse(const NNeh::THttpReply& reply, const TString& shard = {}, TContextPtr context = nullptr) const;

        TAsyncResponse FetchInfo(const TString& id, TContextPtr context) const;
        TAsyncPushResult PushInfo(const TString& id, TContextPtr context, const TAsyncResponse& response) const;

        TAsyncResponse Execute(NNeh::THttpRequest&& request, EStatus fallback, TContextPtr context, bool optimistic = true) const;
        TAsyncResponse Wait(TAsyncBroadcast&& broadcast, EStatus fallback, TContextPtr context) const;
        TAsyncResponse Wait(const TString& shard, TStringBuf id, TDuration timeout, TContextPtr context) const;

        NNeh::THttpRequest CreateWaitRequest(TStringBuf id, TDuration timeout, TDuration waitTimeout = TDuration::Zero()) const;
        NNeh::THttpRequest CreateRequest(TString&& uri, TString&& cgi = {}) const;

        THandler Execute(TString id, NNeh::THttpRequest&& request, TContextPtr context, TDuration timeout) const;

        TMap<TString, NThreading::TFuture<NNeh::THttpReply>> BroadcastRequest(
            const NNeh::THttpRequest& request,
            TDuration timeout = TDuration::Zero(),
            TContextPtr context = nullptr
        ) const;
        NThreading::TFuture<NNeh::THttpReply> MakeRequest(
            const TString& shard,
            const NNeh::THttpRequest& request,
            TDuration timeout = TDuration::Zero(),
            TContextPtr context = nullptr
        ) const;
        NThreading::TFuture<NNeh::THttpReply> MakeRequestImpl(
            const TString& shard,
            const NNeh::THttpRequest& request,
            TDuration timeout
        ) const;

        TConnectionsShardsConstPtr GetConnectionsShards() const;
        void SetConnectionsShards(TConnectionsShardsPtr value) const;

        TString GetName() const;

        TVector<TString> GetShards() const;

    private:
        const TOptions Options;
        const TInstant Created;

        TAtomicSharedPtr<NNeh::THttpClient> Client;
        TAtomicSharedPtr<NRTLine::TNehSearchClient> InfoFetchClient;
        TAtomicSharedPtr<NRTLine::TNehIndexingClient> InfoPushClient;
        TAtomicSharedPtr<IServiceDiscovery> ServiceDiscovery;
        TAtomicSharedPtr<NTvmAuth::TTvmClient> Tvm;

        NGlobalScheduler::TOptionalRegistrator GlobalSchedulerRegistrator;

        TAdaptiveLock ConnectionsLock;
        mutable TConnectionsShardsPtr ConnectionsShards;
    };
}
