#include "service_discovery.h"

#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/neh/neh.h>

#include <library/cpp/ipv6_address/ipv6_address_p.h>

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::NK8s::TPod);
DECLARE_FIELDS_JSON_SERIALIZER(NDrive::NK8s::TPodSnapshot);

THolder<NDrive::IServiceDiscovery> NDrive::NK8s::TServiceDiscoveryConfig::Construct() const {
    return MakeHolder<TServiceDiscovery>(Options);
}

void NDrive::NK8s::TServiceDiscoveryConfig::Init(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    Options.Endpoint = directives.Value("Endpoint", Options.Endpoint);
}

void NDrive::NK8s::TServiceDiscoveryConfig::ToString(IOutputStream& os) const {
    os << "Endpoint: " << Options.Endpoint << Endl;
    os << "Type: " << Type() << Endl;
}

NDrive::NK8s::TServiceDiscovery::TServiceDiscovery(const TServiceDiscoveryOptions& options)
    : IAutoActualization("k8s-sd")
    , Options(options)
{
    if (!Start()) {
        throw yexception() << "cannot start " << GetName();
    }
}

NDrive::NK8s::TServiceDiscovery::~TServiceDiscovery() {
    if (!Stop()) {
        ERROR_LOG << "cannot stop " << GetName() << Endl;
    }
}

TVector<TString> NDrive::NK8s::TServiceDiscovery::GetHosts() const {
    auto snapshot = PodSnapshot.AtomicLoad();
    TVector<TString> result;
    for (auto&& pod : Yensured(snapshot)->Pods) {
        result.push_back(pod.Ip);
    }
    return result;
}

bool NDrive::NK8s::TServiceDiscovery::Refresh() {
    auto timestamp = Now();
    auto request = NNeh::Request(Options.Endpoint);
    if (!request) {
        ERROR_LOG << GetName() << ": cannot Request " << Options.Endpoint << Endl;
        return false;
    }
    auto response = request->Wait();
    if (!response) {
        ERROR_LOG << GetName() << ": cannot Wait " << Options.Endpoint << Endl;
        return false;
    }
    if (response->IsError()) {
        ERROR_LOG << GetName() << ": error in response: " << response->GetErrorCode() << " " << static_cast<ui32>(response->GetErrorType()) << " " << response->GetErrorText() << Endl;
        return false;
    }
    NJson::TJsonValue report;
    if (!NJson::ReadJsonFastTree(response->Data, &report)) {
        ERROR_LOG << GetName() << ": cannot ReadJsonFastTree from " << response->Data << Endl;
        return false;
    }
    TPods pods;
    if (!NJson::TryFromJson(report, pods)) {
        ERROR_LOG << GetName() << ": cannot get Pods from " << report.GetStringRobust() << Endl;
        return false;
    }

    for (auto& pod: pods) {
        if (TIpv6Address::Ipv6 == FigureOutType(pod.Ip) && pod.Ip.back() != ']') {
            pod.Ip = '[' + pod.Ip + ']';
        }
    }
    auto snapshot = MakeIntrusive<TPodSnapshot>();
    snapshot->Pods = std::move(pods);
    snapshot->Timestamp = timestamp;
    PodSnapshot.AtomicStore(snapshot);
    INFO_LOG << GetName() << ": refreshed PodSnapshot: " << NJson::ToJson(snapshot).GetStringRobust() << Endl;
    return true;
}

NDrive::IServiceDiscoveryConfig::TFactory::TRegistrator<NDrive::NK8s::TServiceDiscoveryConfig> NDrive::NK8s::TServiceDiscoveryConfig::Registrator(
    NDrive::NK8s::TServiceDiscoveryConfig::Type()
);
