#include "service_discovery.h"

#include <infra/yp_service_discovery/libs/sdlib/client.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/builder.h>

namespace {
    NYP::NServiceDiscovery::TSDConfig MakeServiceDiscoveryConfig(const TServiceDiscoveryOptions& options) {
        auto config = NYP::NServiceDiscovery::TSDConfig();
        config.SetClientName(options.ClientName);
        config.SetCacheDir(options.CacheDir);
        config.SetUpdateFrequency(ToString(options.UpdateFrequency));
        return config;
    }
}

TEndpointSetProvider::TEndpointSetProvider(const NYP::NServiceDiscovery::TEndpointSetKey& key)
    : Key_(key)
{
}

void TEndpointSetProvider::Subscribe(NYP::NServiceDiscovery::TSDClient* sdClient) {
    SubscribeRef_ = sdClient->Subscribe(Key_, *this);
    EndpointSet_ = SubscribeRef_->GetEndpointSet();
    SetActiveEndpointSetInfo(EndpointSet_.Info);
}

void TEndpointSetProvider::Update(const NYP::NServiceDiscovery::TEndpointSetEx& endpointSet) {
    with_lock (Lock_) {
        EndpointSet_ = endpointSet;
    }
    SetActiveEndpointSetInfo(endpointSet.Info);
    INFO_LOG << "Endpoint set: " << Key_.ToString() << " has been updated" << Endl;
}

NYP::NServiceDiscovery::TEndpointSetEx TEndpointSetProvider::GetEndpointSet() const {
    with_lock (Lock_) {
        return EndpointSet_;
    }
}

TServiceDiscoveryManager::TServiceDiscoveryManager(const TServiceDiscoveryOptions& config) {
    Client_ = MakeHolder<NYP::NServiceDiscovery::TSDClient>(MakeServiceDiscoveryConfig(config));
    for (const auto& endpointSetId : config.EndpointSetIds) {
        THolder<TEndpointSetProvider>& provider = Providers_[endpointSetId];
        if (!provider) {
            provider = MakeHolder<TEndpointSetProvider>(endpointSetId);
            provider->Subscribe(Client_.Get());
        }
    }

    Client_->Start();
}

TServiceDiscoveryManager::~TServiceDiscoveryManager() {
    Client_->Stop();
}

TVector<TString> TServiceDiscoveryManager::GetHosts() const {
    TVector<TString> result;
    for (const auto& [key, provider] : Providers_) {
        try {
            Y_VERIFY(provider, "provider disappeared");
            auto endpointSet = provider->GetEndpointSet();
            if (endpointSet.endpoints_size() == 0) {
                INFO_LOG << "No endpoints for set: " << endpointSet.endpoint_set_id() << ", cluster: " << key.cluster_name() << Endl;
                continue;
            }
            for (const auto& endpoint : endpointSet.endpoints()) {
                if (!endpoint.ready()) {
                    INFO_LOG << "Endpoint fqdn: " << endpoint.fqdn() << " is not ready" << Endl;
                    continue;
                }
                const TString shard = TStringBuilder() << endpoint.fqdn() << ":" << endpoint.port();
                result.emplace_back(shard);
            }
        } catch (const yexception& e) {
            ERROR_LOG << "Cant get endpoint " << key.ToString() << ", error:" << e.what() << Endl;
        }
    }
    return result;
}
