#include "car_emulator.h"
#include "handlers.h"

#include <drive/telematics/common/handler.h>

#include <rtline/library/scheduler/global.h>

using namespace NDrive;

class TBlackboxProcess : public TGlobalScheduler::TScheduledItem<TBlackboxProcess> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<TBlackboxProcess>;

public:
    TBlackboxProcess(NDrive::TCarEmulator& emulator, TInstant now)
        : TBase(emulator.Name(), emulator.Name() + ":blackbox", now)
        , Emulator(emulator)
    {
    }

    void Process(void* /*threadSpecificResource*/) override {
        THolder<TBlackboxProcess> cleanup(this);
        auto client = Emulator.GetClient();
        if (!client) {
            WARNING_LOG << Emulator.Name() << " has no client" << Endl;
            return;
        }
        if (client->Alive()) {
            NDrive::NVega::TBlackboxRecords::TRecords records = { CreateBlackboxRecord(Emulator.GetContext()) };
            DEBUG_LOG << Emulator.GetIMEI() << ": sending Blackbox" << Endl;
            client->AddMessageHandler(MakeAtomicShared<NDrive::TBlackboxHandler>(
                std::move(records),
                MessageId++
            ));
        } else {
            WARNING_LOG << Emulator.Name() << " is dead" << Endl;
        }
    }
    THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
        return MakeHolder<TBlackboxProcess>(Emulator, now + Interval);
    }

private:
    NDrive::TCarEmulator& Emulator;

    TDuration Interval = TDuration::Seconds(5);
    std::atomic<ui64> MessageId = 0;
};

class TAdvanceContextProcess : public TGlobalScheduler::TScheduledItem<TAdvanceContextProcess> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<TAdvanceContextProcess>;

public:
    TAdvanceContextProcess(NDrive::TCarEmulator& emulator, TInstant now)
        : TBase(emulator.Name(), emulator.Name() + ":advance_context", now)
        , Emulator(emulator)
    {
    }

    void Process(void* /*threadSpecificResource*/) override {
        THolder<TAdvanceContextProcess> cleanup(this);
        Emulator.GetContext().AdvancePath();
    }
    THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
        return MakeHolder<TAdvanceContextProcess>(Emulator, now + Interval);
    }

private:
    NDrive::TCarEmulator& Emulator;
    TDuration Interval = TDuration::Seconds(1);
};

class TReconnectProcess : public TGlobalScheduler::TScheduledItem<TReconnectProcess> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<TReconnectProcess>;

public:
    TReconnectProcess(NDrive::TCarEmulator& emulator, TInstant now)
        : TBase(emulator.Name(), emulator.Name() + ":reconnect", now)
        , Emulator(emulator)
    {
    }

    void Process(void* /*threadSpecificResource*/) override {
        THolder<TReconnectProcess> cleanup(this);
        auto client = Emulator.GetClient();
        if (!client) {
            WARNING_LOG << Emulator.Name() << " has no client" << Endl;
            return;
        }
        if (client && !client->Alive()) {
            INFO_LOG << Emulator.Name() << " is reconnecting" << Endl;
            client->Connect(Emulator.GetHost(), Emulator.GetPort(), false);
        }
    }
    THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
        return MakeHolder<TReconnectProcess>(Emulator, now + Interval);
    }

private:
    NDrive::TCarEmulator& Emulator;

    TDuration Interval = TDuration::Seconds(10);
};

THolder<NDrive::TTelematicsClientContext> CreateContext(TMaybe<TGeoCoord> position) {
    auto context = MakeHolder<NDrive::TTelematicsClientContext>();
    context->SetCurrentPosition(position.GetOrElse({ 37.587703, 55.733404 }));
    context->TrySetEngineStarted(true);
    return context;
}

void TCarEmulator::AddCommonHandlers(NDrive::TTelematicsTestClient& client) {
    auto blink = MakeAtomicShared<NDrive::TOnBlink>(client, GetContext());
    client.AddHandler(blink);
    auto openDriverDoor = MakeAtomicShared<NDrive::TOnOpenDrDoor>(client, GetContext());
    client.AddHandler(openDriverDoor);
    auto openDoors = MakeAtomicShared<NDrive::TOnOpenDoors>(client, GetContext());
    client.AddHandler(openDoors);
    auto simpleHandler = MakeAtomicShared<NDrive::TSimpleCommandsHandler>(client);
    client.AddHandler(simpleHandler);
    auto closeDoors = MakeAtomicShared<NDrive::TOnCloseDoors>(client, GetContext());
    client.AddHandler(closeDoors);
    auto startOfLease = MakeAtomicShared<NDrive::TOnStartOfLease>(client, GetContext());
    client.AddHandler(startOfLease);
    auto endOfLease = MakeAtomicShared<NDrive::TOnEndOfLease>(client, GetContext());
    client.AddHandler(endOfLease);
    auto forcedEndOfLease = MakeAtomicShared<NDrive::TOnForcedEndOfLease>(client, GetContext());
    client.AddHandler(forcedEndOfLease);
    auto startWarming = MakeAtomicShared<NDrive::TOnStartWarming>(client, GetContext());
    client.AddHandler(startWarming);
    auto stopWarming = MakeAtomicShared<NDrive::TOnStopWarming>(client, GetContext());
    client.AddHandler(stopWarming);
    auto getParameter = MakeAtomicShared<NDrive::TGetParamHandler>(client, GetContext());
    client.AddHandler(getParameter);
    auto setParameter = MakeAtomicShared<NDrive::TSetParamHandler>(client, GetContext());
    client.AddHandler(setParameter);
    auto gsmModemCommand = MakeAtomicShared<NDrive::TOnGsmModemCommand>(client, GetContext());
    client.AddHandler(gsmModemCommand);
    auto getFile = MakeAtomicShared<NDrive::TGetFileHandler>(client);
    client.AddHandler(getFile);
    auto disconnect = MakeAtomicShared<NDrive::TDisconnectCommand>(client);
    client.AddHandler(disconnect);
    auto moveToCoordHandler = MakeAtomicShared<NDrive::TMoveToCoordHandler>(client, GetContext(), Router);
    client.AddHandler(moveToCoordHandler);
    auto setFile = MakeAtomicShared<NDrive::TSetFileHandler>(client);
    client.AddHandler(setFile);
    auto obdForwardConfig = MakeAtomicShared<NDrive::TObdForwardConfigHandler>(GetContext());
    client.AddHandler(obdForwardConfig);
    auto canRequest = MakeAtomicShared<NDrive::TCanRequestHandler>(GetContext());
    client.AddHandler(canRequest);
}

TCarEmulator::TCarEmulator(const TString& imei, const TString& host, const ui16 port,
    TAtomicSharedPtr<NDrive::TTelematicsClientContext> context,
    TAtomicSharedPtr<NGraph::TRouter> router,
    TAtomicSharedPtr<NAsio::TExecutorsPool> executorsPool,
    bool wait
)
    : Client(new NDrive::TTelematicsTestClient(imei, executorsPool, "car_emulator:" + imei))
    , Context(std::move(context))
    , Router(router)
    , Created(Now())
    , IMEI(imei)
    , Host(host)
    , Port(port)
{
    AddCommonHandlers(*Client);
    Client->Connect(Host, Port, false);

    if (wait) {
        auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();
        Client->AddHandler(heartbeat);
        heartbeat->Wait();
        Y_ENSURE(!heartbeat->WasDropped(), "heartbeat was dropped");
    }

    GlobalSchedulerRegistrator.emplace(Name());
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TBlackboxProcess>(*this, Now())));
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TAdvanceContextProcess>(*this, Now())));
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TReconnectProcess>(*this, Now())));
}

TCarEmulator::TCarEmulator(const TString& imei, const TString& host, const ui16 port,
    TAtomicSharedPtr<NGraph::TRouter> router,
    TAtomicSharedPtr<NAsio::TExecutorsPool> executorsPool,
    TMaybe<TGeoCoord> position
)
    : TCarEmulator(imei, host, port, CreateContext(position), router, executorsPool)
{
}

NDrive::TCarEmulator::~TCarEmulator() {
    GlobalSchedulerRegistrator.reset();
    Client.Drop();
}

float NDrive::GetRandomFromRange(float a, float b) {
    return a + RandomNumber<float>() * (b - a);
}

NDrive::NVega::TBlackboxRecords::TRecord& NDrive::AddParameter(NVega::TBlackboxRecords::TRecord& record, TSensorId sensor, TSensorValue value) {
    auto& param = record.Parameters.Mutable().emplace_back();
    param.Id = sensor.Id;
    param.SetValue(value);
    return record;
}

NDrive::NVega::TBlackboxRecords::TRecord NDrive::CreateBlackboxRecord(const TTelematicsClientContext& context) {
    NDrive::NVega::TBlackboxRecords::TRecord record;
    record.Lattitude = context.GetCurrentPosition().Y;
    record.Longitude = context.GetCurrentPosition().X;
    record.Course = 5;
    record.Height = -1;
    record.Satelites = 14;
    record.Speed = context.GetSpeed();
    record.Timestamp = Seconds();
    AddParameter(record, VEGA_MCU_FIRMWARE_VERSION, context.GetMcuFirmwareVersion());
    AddParameter(record, VEGA_HDOP, double(context.GetHDOP()));
    AddParameter(record, VEGA_MCC, ui64(context.GetMCC()));
    AddParameter(record, VEGA_MNC, ui64(context.GetMNC()));
    AddParameter(record, CAN_ENGINE_IS_ON, ui64(context.GetEngineStarted()));
    AddParameter(record, CAN_FUEL_LEVEL_P, ui64(context.GetFuelPercent()));
    AddParameter(record, CAN_ODOMETER_KM, double(context.GetOdometerKm()));
    AddParameter(record, VEGA_SETTING_SERVER_PIN, TBuffer{context.GetServerPin().data(), context.GetServerPin().size()});
    AddParameter(record, VEGA_SETTING_PIN_CODE, TBuffer{context.GetWiredPin().data(), context.GetWiredPin().size()});
    if (context.GetTankerFuelLevel()) {
        AddParameter(record, NDrive::NVega::AuxFuelLevel<1>(), double(context.GetTankerFuelLevel().GetRef()));
    }
    if (context.GetTankerSecondFuelLevel()) {
        AddParameter(record, NDrive::NVega::AuxFuelLevel<2>(), double(context.GetTankerSecondFuelLevel().GetRef()));
    }
    for (const auto& sensor : context.GetCustomSensors()) {
        if (std::holds_alternative<double>(sensor.Value)) {
            AddParameter(record, sensor.Id, sensor.ConvertTo<double>());
        }
    }
    return record;
}

TGeoCoord NDrive::CreateLocation(const TSet<TString>& tags) {
    if (tags.contains("kazan_tag")) {
        return {
            GetRandomFromRange(49.10174, 49.17221),
            GetRandomFromRange(55.75992, 55.80104)
        };
    }
    if (tags.contains("peterburg_tag")) {
        return {
            GetRandomFromRange(30.23712, 30.43712),
            GetRandomFromRange(59.81850, 60.06859)
        };
    }
    {
        return {
            NDrive::GetRandomFromRange(37.479697, 37.744742),
            NDrive::GetRandomFromRange(55.697232, 55.809532)
        };
    }
}
