#include "handlers.h"

#include <rtline/api/graph/router/router.h>
#include <drive/telematics/protocol/mio.h>

#include <util/string/builder.h>

using namespace NDrive;

bool TOnOpenDrDoor::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.TrySetDrDoorOpened(true);
}

bool TOnOpenDoors::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.TrySetDoorsOpened(true);
}

bool TOnCloseDoors::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.TrySetDoorsOpened(false);
}

bool TOnStartEngine::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.TrySetEngineStarted(true);
}

bool TOnStopEngine::ProcessCommand(NVega::IConnection& /*connection*/) {
    Context.TrySetEngineStarted(false);
    return !Context.GetEngineStarted();
}

bool NDrive::TOnBlink::ProcessCommand(NVega::IConnection& /*connection*/) {
    return true;
}

bool TOnStartWarming::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.TrySetEngineStarted(true);
}

bool TOnStopWarming::ProcessCommand(NVega::IConnection& /*connection*/) {
    Context.TrySetEngineStarted(false);
    return !Context.GetEngineStarted();
}

void TCompositeHandler::AddHandler(TTelematicsTestClient::THandlerPtr handler) {
    Handlers.push_back(handler);
}

TTelematicsTestClient::EHandlerStatus TCanRequestHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::CAN_REQUEST && Context.GetEnableCanResponse()) {
        const auto& request = message.As<NDrive::NVega::TCanRequest>();
        auto response = MakeHolder<NVega::TMessage>(NVega::CAN_RESPONSE);
        auto& payload = response->As<NVega::TCanResponse>();
        for (auto&& frame : request.Frames) {
            auto& rframe = payload.Frames.emplace_back();
            rframe.Data.id = frame.Data.id + 1;
            rframe.Data.id_type = frame.Data.id_type;
            rframe.Data.itf_idx = frame.Data.itf_idx;
            rframe.Data.time = 42;
            rframe.Data.dlen = frame.Data.dlen;
            std::memcpy(rframe.Data.data, frame.Data.data, frame.Data.dlen);
        }
        connection.SendMessage(std::move(response));
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TTelematicsTestClient::EHandlerStatus TGetParamHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NVega::TCommandRequest>();
        if (request.Code == NDrive::NVega::ECommandCode::GET_PARAM) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;

            NVega::TCommandResponse::TGetParameter parameter;
            auto arg = request.Argument.Get<NVega::TCommandRequest::TGetParameter>();
            parameter.Id = arg.Id;
            switch (arg.Id) {
                case VEGA_MCU_FIRMWARE_VERSION:
                    parameter.SetValue(Context.GetMcuFirmwareVersion());
                    break;
                case VEGA_LON:
                    parameter.SetValue<double>(Context.GetCurrentPosition().X);
                    break;
                case VEGA_LAT:
                    parameter.SetValue<double>(Context.GetCurrentPosition().Y);
                    break;
                case VEGA_SPEED:
                    parameter.SetValue<double>(Context.GetSpeed());
                    break;
                case CAN_ENGINE_IS_ON:
                    parameter.SetValue<ui8>(Context.GetEngineStarted());
                    break;
                case CAN_FUEL_LEVEL_P:
                    parameter.SetValue<ui8>(Context.GetFuelPercent());
                    break;
                case CAN_DRIVER_DOOR:
                    parameter.SetValue<ui8>(Context.GetDoorsState().DriverDoorOpened);
                    break;
                case CAN_PASS_DOOR:
                    parameter.SetValue<ui32>(Context.GetDoorsState().PassDoorOpened);
                    break;
                case CAN_L_REAR_DOOR:
                    parameter.SetValue<ui32>(Context.GetDoorsState().LRearDoorOpened);
                    break;
                case CAN_R_REAR_DOOR:
                    parameter.SetValue<ui32>(Context.GetDoorsState().RRearDoorOpened);
                    break;
                case CAN_HOOD:
                    parameter.SetValue<ui32>(Context.GetHoodOpened());
                    break;
                case CAN_TRUNK:
                    parameter.SetValue<ui32>(Context.GetTrunkOpened());
                    break;
                case CAN_ODOMETER_KM:
                    parameter.SetValue<double>(Context.GetOdometerKm());
                    break;
                case VEGA_MCC:
                    parameter.SetValue<ui32>(Context.GetMCC());
                    break;
                case VEGA_MNC:
                    parameter.SetValue<ui32>(Context.GetMNC());
                    break;
                case VEGA_LAC:
                    parameter.SetValue<ui32>(Context.GetLAC());
                    break;
                case VEGA_CELLID:
                    parameter.SetValue<ui32>(Context.GetCellID());
                    break;
                case VEGA_GSM_SIGNAL_LEVEL:
                    parameter.SetValue<ui32>(Context.GetVegaGSMSignalLevel());
                    break;
                case VEGA_HDOP:
                    parameter.SetValue<double>(Context.GetHDOP());
                    break;
                case VEGA_INPUT_BUFFER_SIZE:
                    parameter.SetValue<ui16>(1024);
                    break;
                case VEGA_SETTING_SERVER_ADDR:
                    parameter.SetValue(SaveTo<TBuffer>(Context.GetServerSettings()));
                    break;
                case VEGA_SETTING_APN:
                    parameter.SetValue(SaveTo<TBuffer>(Context.GetAPN()));
                    break;
                case VEGA_SETTING_TRANSLATE_SENSORS:
                    parameter.SetValue(SaveTo<TBuffer>(Context.GetSensorTranslation(arg.SubId)));
                    break;
                case VEGA_SETTING_USE_SERVER_PIN:
                    parameter.SetValue<ui8>(Context.UseServerPin());
                    break;
                case VEGA_SETTING_SERVER_PIN: {
                    NDrive::NVega::THardPasswordParameter pin;
                    pin.Value.Set(Context.GetServerPin());
                    parameter.SetValue(SaveTo<TBuffer>(pin));
                    break;
                }
                case VEGA_SETTING_PIN_CODE: {
                    NDrive::NVega::THardPasswordParameter pin;
                    pin.Value.Set(Context.GetWiredPin());
                    parameter.SetValue(SaveTo<TBuffer>(pin));
                    break;
                }
                case VEGA_SETTING_USE_PIN_CODE:
                    parameter.SetValue<ui8>(Context.UseWiredPin());
                    break;
                case VEGA_SETTING_BLE_EXT_BOARD: {
                    Y_ENSURE(arg.SubId == NDrive::NVega::BlePasskey.SubId);
                    NDrive::NVega::TBlePasskeyParameter passkey;
                    passkey.Set(Context.GetBlePasskey());
                    parameter.SetValue(SaveTo<TBuffer>(passkey));
                    break;
                }
                case BLE_EXT_BOARD_MAC: {
                    auto bleMac = Context.GetBleMac();
                    auto value = TBuffer(bleMac.data(), bleMac.size());
                    parameter.SetValue(value);
                    break;
                }
                case NDrive::NVega::BleSessionKey.Id:
                    parameter.SetValue<TString>(Context.GetBleSessionKey());
                    break;
                case BLE_EXT_BOARD_BEACONS_INFO1:
                case BLE_EXT_BOARD_BEACONS_INFO2:
                case BLE_EXT_BOARD_BEACONS_INFO3:
                case BLE_EXT_BOARD_BEACONS_INFO4:
                case BLE_EXT_BOARD_BEACONS_INFO5:
                case BLE_EXT_BOARD_BEACONS_INFO6:
                case BLE_EXT_BOARD_BEACONS_INFO7:
                case BLE_EXT_BOARD_BEACONS_INFO8:
                case BLE_EXT_BOARD_BEACONS_INFO9:
                case BLE_EXT_BOARD_BEACONS_INFO10: {
                    TString res;
                    TStringOutput output(res);
                    const auto& beaconsInfos = Context.GetBeaconsInfos();
                    int offset = arg.Id - BLE_EXT_BOARD_BEACONS_INFO1;
                    beaconsInfos[offset].Save(&output);
                    TBuffer buf(res.data(), res.size());
                    parameter.SetValue(std::move(buf));
                    break;
                }
                case NDrive::NVega::AuxFuelLevel<1>(): {
                    const auto& val = Context.GetTankerFuelLevel();
                    if (val) {
                        parameter.SetValue<double>(*val);
                    } else {
                        response.Result = response.ERROR;
                    }
                    break;
                }
                case NDrive::NVega::AuxFuelLevel<2>(): {
                    const auto& val = Context.GetTankerSecondFuelLevel();
                    if (val) {
                        parameter.SetValue<double>(*val);
                    } else {
                        response.Result = response.ERROR;
                    }
                    break;
                }
                default:
                    response.Result = response.ERROR;
            }

            response.Argument.Set(parameter);
            if (response.Result == response.UNKNOWN) {
                response.Result = response.PROCESSED;
            }

            Client.SendMessage(std::move(ack));
            return TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

template<typename T>
void TSetParamHandler::HandleValue(const NDrive::TSensorRef& value, NVega::TCommandResponse& response, std::function<bool (const NDrive::TSensorRef& value)> setter) {
    if (!std::holds_alternative<T>(value)) {
        response.Result = response.INCORRECT;
    } else try {
        if (setter(value)) {
            response.Result = response.PROCESSED;
        } else {
            response.Result = response.ERROR;
        }
    } catch (const std::exception& e) {
        response.Result = response.ERROR;
        NDrive::NProtocol::TZeroTerminatedString error;
        error.Set(FormatExc(e));
        response.Argument.Set(error);
    }
}

TTelematicsTestClient::EHandlerStatus TSetParamHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NDrive::NVega::TCommandRequest>();
        if (request.Code == NDrive::NVega::ECommandCode::SET_PARAM) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;

            const auto& argument = request.Argument.Get<NDrive::NVega::TCommandRequest::TSetParameter>();
            auto value = argument.GetValue();
            switch (argument.Id) {
                case VEGA_LON:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        TGeoCoord position = Context.GetCurrentPosition();
                        position.X = std::get<double>(value);
                        Context.SetCurrentPosition(position);
                        return true;
                    });
                    break;
                case VEGA_LAT:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        TGeoCoord position = Context.GetCurrentPosition();
                        position.Y = std::get<double>(value);
                        Context.SetCurrentPosition(position);
                        return true;
                    });
                    break;
                case VEGA_SPEED:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        return Context.TrySetSpeed(std::get<double>(value));
                    });
                    break;
                case CAN_ENGINE_IS_ON:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        return Context.TrySetEngineStarted(std::get<ui64>(value));
                    });
                    break;
                case CAN_ODOMETER_KM:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetOdometerKm(std::get<double>(value));
                        return true;
                    });
                    break;
                case CAN_FUEL_LEVEL_P:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetFuelPercent(std::get<ui64>(value));
                        return true;
                    });
                    break;
                case VEGA_MCC:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetMCC(std::get<ui64>(value));
                        return true;
                    });
                    break;
                case VEGA_MNC:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetMNC(std::get<ui64>(value));
                        return true;
                    });
                    break;
                case VEGA_LAC:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetLAC(std::get<ui64>(value));
                        return true;
                    });
                    break;
                case VEGA_CELLID:
                    HandleValue<ui64>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetCellID(std::get<ui64>(value));
                        return true;
                    });
                    break;
                case VEGA_GSM_SIGNAL_LEVEL:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetVegaGSMSignalLevel(std::get<double>(value));
                        return true;
                    });
                    break;
                case VEGA_HDOP:
                    HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetHDOP(std::get<double>(value));
                        return true;
                    });
                    break;
                case VEGA_SETTING_SERVER_ADDR:
                    HandleValue<TConstArrayRef<char>>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetServerSettings(LoadFrom<NVega::TServerSettingsParameter>(std::get<TConstArrayRef<char>>(value)));
                        return true;
                    });
                    break;
                case VEGA_SETTING_APN:
                    HandleValue<TConstArrayRef<char>>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetAPN(LoadFrom<NVega::TAPNParameter>(std::get<TConstArrayRef<char>>(value)));
                        return true;
                    });
                    break;
                case VEGA_SETTING_TRANSLATE_SENSORS:
                    HandleValue<TConstArrayRef<char>>(value, response, [this, argument](const NDrive::TSensorRef& value) {
                        Context.SetSensorTranslation(argument.SubId, LoadFrom<NVega::TSensorTranslation>(std::get<TConstArrayRef<char>>(value)));
                        return true;
                    });
                    break;
                case VEGA_SETTING_USE_SERVER_PIN:
                    HandleValue<ui64>(value, response, [](const NDrive::TSensorRef& value) {
                        Y_UNUSED(value);
                        return true;
                    });
                    break;
                case VEGA_SETTING_SERVER_PIN:
                    HandleValue<TConstArrayRef<char>>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetServerPin(LoadFrom<NVega::THardPasswordParameter>(std::get<TConstArrayRef<char>>(value)).Value.Get());
                        return true;
                    });
                    break;
                case VEGA_SETTING_PIN_CODE:
                    HandleValue<TConstArrayRef<char>>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetWiredPin(LoadFrom<NVega::THardPasswordParameter>(std::get<TConstArrayRef<char>>(value)).Value.Get());
                        return true;
                    });
                    break;
                case VEGA_SETTING_USE_PIN_CODE:
                    HandleValue<ui64>(value, response, [](const NDrive::TSensorRef& value) {
                        Y_UNUSED(value);
                        return true;
                    });
                    break;
                case VEGA_SETTING_BLE_EXT_BOARD:
                    Y_ENSURE(argument.SubId == NDrive::NVega::BlePasskey.SubId);
                    HandleValue<TStringBuf>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetBlePasskey(TString(std::get<TStringBuf>(value)));
                        return true;
                    });
                    break;
                case NDrive::NVega::BleSessionKey.Id:
                    HandleValue<TConstArrayRef<char>>(value, response, [this](const NDrive::TSensorRef& value) {
                        Context.SetBleSessionKey(std::get<TConstArrayRef<char>>(value));
                        return true;
                    });
                    break;
                case BLE_EXT_BOARD_BEACONS_INFO1:
                case BLE_EXT_BOARD_BEACONS_INFO2:
                case BLE_EXT_BOARD_BEACONS_INFO3:
                case BLE_EXT_BOARD_BEACONS_INFO4:
                case BLE_EXT_BOARD_BEACONS_INFO5:
                case BLE_EXT_BOARD_BEACONS_INFO6:
                case BLE_EXT_BOARD_BEACONS_INFO7:
                case BLE_EXT_BOARD_BEACONS_INFO8:
                case BLE_EXT_BOARD_BEACONS_INFO9:
                case BLE_EXT_BOARD_BEACONS_INFO10: {
                    HandleValue<TConstArrayRef<char>>(value, response, [this, argument](const NDrive::TSensorRef& value) {
                        const auto& str = std::get<TConstArrayRef<char>>(value);
                        TMemoryInput input(str.data(), str.size());
                        auto& beaconsInfos = Context.GetBeaconsInfos();
                        int offset = argument.Id - BLE_EXT_BOARD_BEACONS_INFO1;
                        beaconsInfos[offset].Load(&input);
                        return true;
                    });
                    break;
                }
                case NDrive::NVega::AuxFuelLevel<1>(): {
                    if (Context.GetTankerFuelLevel()) {
                        HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                            Context.SetTankerFuelLevel(std::get<ui64>(value));
                            return true;
                        });
                    } else {
                        response.Result = response.ERROR;
                        response.Argument.Set(NVega::TCommandResponse::TInfo(TStringBuilder() << "unsupported sensor " << argument.GetSensorId()));
                    }
                    break;
                }
                case NDrive::NVega::AuxFuelLevel<2>(): {
                    if (Context.GetTankerSecondFuelLevel()) {
                        HandleValue<double>(value, response, [this](const NDrive::TSensorRef& value) {
                            Context.SetTankerSecondFuelLevel(std::get<ui64>(value));
                            return true;
                        });
                    } else {
                        response.Result = response.ERROR;
                        response.Argument.Set(NVega::TCommandResponse::TInfo(TStringBuilder() << "unsupported sensor " << argument.GetSensorId()));
                    }
                    break;
                }
                default:
                    response.Result = response.ERROR;
                    response.Argument.Set(NVega::TCommandResponse::TInfo(TStringBuilder() << "unsupported sensor " << argument.GetSensorId()));
            }
            Client.SendMessage(std::move(ack));
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TTelematicsTestClient::EHandlerStatus TObdForwardConfigHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NDrive::NVega::TCommandRequest>();
        if (request.Code == NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;

            auto argument = request.Argument.TryGet<NDrive::NVega::TCommandRequest::TObdForwardConfig>();
            if (!argument) {
                response.Result = response.ERROR;
                response.Argument.Set(NDrive::NVega::TCommandResponse::TInfo("cannot cast"));
            }
            if (argument && argument->Cans.size() > 0) {
                response.Result = response.PROCESSED;
                Context.SetEnableCanResponse(true);
            } else {
                response.Result = response.ERROR;
                response.Argument.Set(NDrive::NVega::TCommandResponse::TInfo("no Cans"));
            }

            connection.SendMessage(std::move(ack));
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TMoveToCoordHandler::TMoveToCoordHandler(TTelematicsTestClient& client, TTelematicsClientContext& context, TAtomicSharedPtr<NGraph::TRouter> router)
    : Client(client)
    , Context(context)
    , Router(router)
{
}

TMoveToCoordHandler::~TMoveToCoordHandler() {
}

TTelematicsTestClient::EHandlerStatus TMoveToCoordHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NDrive::NVega::TCommandRequest>();
        if (request.Code == NDrive::NVega::ECommandCode::MOVE_TO_COORD) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;
            try {
                const auto& argument = request.Argument.Get<NDrive::NVega::TCommandRequest::TMoveToCoordParameter>();
                MoveTo(argument.Lat, argument.Lon, argument.Speed);
                response.Result = response.PROCESSED;
            } catch (const std::exception& e) {
                NDrive::NVega::TCommandResponse::TInfo error;
                error.Set(FormatExc(e));
                response.Argument.Set(error);
                response.Result = response.ERROR;
            }
            Client.SendMessage(std::move(ack));
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
}

void TMoveToCoordHandler::MoveTo(double latitude, double longitude, double speed) {
    const auto targetPosition = TGeoCoord(longitude, latitude);
    if (std::abs(latitude) < 0.001 && std::abs(longitude) < 0.001) {
        Context.SetCurrentPosition({0, 0});
        return;
    }

    const auto currentPosition = Context.GetCurrentPosition();
    if (std::abs(currentPosition.X) < 0.001 && std::abs(currentPosition.Y) < 0.001) {
        Context.SetCurrentPosition(targetPosition);
        return;
    }

    if (speed < 1) {
        Context.SetCurrentPosition(targetPosition);
        Context.SetPath({});
        return;
    }

    Y_ENSURE(Router);
    auto asyncRoute = Router->GetRouteAsync(currentPosition, targetPosition);
    auto route = asyncRoute.ExtractValueSync();
    Y_ENSURE(route);
    Context.SetPath(GetPolyline(*route));
    Context.SetPathSpeed(speed * 3600 / 1000);
}

bool TOnStartOfLease::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.StartLeasing();
}

bool TOnEndOfLease::ProcessCommand(NVega::IConnection& /*connection*/) {
    return Context.EndLeasing();
}

bool NDrive::TOnForcedEndOfLease::ProcessCommand(NVega::IConnection& /*connection*/) {
    Context.EndLeasing();
    return true;
}

bool NDrive::TOnGsmModemCommand::ProcessCommand(NVega::IConnection& /*connection*/) {
    auto mnc = Context.GetMNC();
    if (mnc == 2) {
        Context.SetMNC(99);
    }
    if (mnc == 99) {
        Context.SetMNC(2);
    }
    return true;
}

bool NDrive::TDisconnectCommand::ProcessCommand(NVega::IConnection& connection) {
    connection.Drop();
    return true;
}

TTelematicsTestClient::EHandlerStatus TGetFileHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::GET_FILE_REQUEST) {
        const auto& request = message.As<NVega::TGetFileRequest>();
        const TString fileName = request.Name.Get();
        if (fileName == "LOG") {
            const TString data = "[11/7/2018][16:11:41]abc\n[11/7/2018][16:19:52]def\n[11/7/2018][16:21:22]123\n[11/7/2018][16:21:56]456\n";
            auto ack = MakeHolder<NVega::TMessage>(NVega::GET_FILE_RESPONSE);
            auto& response = ack->As<NVega::TGetFileResponse>();
            response.Name.Set(fileName);
            response.TotalSize = data.size();

            const TString cutPart = data.substr(request.Offset, request.Size);
            response.Content.Data = TBuffer(cutPart.c_str(), cutPart.size());
            response.Content.Result = response.OK;

            Client.SendMessage(std::move(ack));
            ++Count;
            return TTelematicsTestClient::EHandlerStatus::Continue;
        }
        {
            auto ack = MakeHolder<NVega::TMessage>(NVega::GET_FILE_RESPONSE);
            auto& response = ack->As<NVega::TGetFileResponse>();
            response.Name.Set(fileName);
            response.Content.Result = NVega::TGetFileResponse::FILENAME_ERROR;
            Client.SendMessage(std::move(ack));
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TTelematicsTestClient::EHandlerStatus TSetFileHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::FILE_CHUNK_REQUEST) {
        const auto& request = message.As<NVega::TFileChunkRequest>();
        const TString fileName = request.Name.Get();
        if (fileName == "FIRMWARE") {
            auto ack = MakeHolder<NVega::TMessage>(NVega::FILE_CHUNK_RESPONSE);
            auto& response = ack->As<NVega::TFileChunkResponse>();
            response.Name.Set(fileName);
            response.Chunk = request.Chunk.GetId();
            response.Result = NVega::TFileChunkResponse::OK;
            Client.SendMessage(std::move(ack));
        } else {
            auto ack = MakeHolder<NVega::TMessage>(NVega::FILE_CHUNK_RESPONSE);
            auto& response = ack->As<NVega::TFileChunkResponse>();
            response.Name.Set(fileName);
            response.Result = NVega::TFileChunkResponse::FILENAME_ERROR;
            Client.SendMessage(std::move(ack));
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}
