#include "client.h"

#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/mio.h>
#include <drive/telematics/protocol/wialon.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/neh/asio/asio.h>
#include <library/cpp/neh/asio/executor.h>

#include <util/stream/buffer.h>
#include <util/stream/mem.h>
#include <util/system/event.h>

namespace {
    class TAsioSocketOutput: public IOutputStream {
    public:
        TAsioSocketOutput(NAsio::TTcpSocket& socket, const TString& id)
            : Socket(socket)
            , Id(id)
        {
        }

    protected:
        void DoWrite(const void* buf, size_t len) override {
            Output.Write(buf, len);
        }
        void DoFlush() override {
            Output.Flush();
            Socket.WriteSome(Output.Buffer().Data(), Output.Buffer().Size());
            DEBUG_LOG << Id << " flushed " << Output.Buffer().Size() << " bytes" << Endl;
            Output.Buffer().Clear();
        }

    private:
        NAsio::TTcpSocket& Socket;
        const TString& Id;
        TBufferOutput Output;
    };

    THolder<NDrive::NProtocol::IMessage> CreateMessage(NDrive::NProtocol::EProtocolType protocol) {
        switch (protocol) {
        case NDrive::NProtocol::PT_VEGA:
            return MakeHolder<NDrive::NVega::TMessage>();
        case NDrive::NProtocol::PT_WIALON_IPS:
            return MakeHolder<NDrive::NWialon::TMessage>();
        case NDrive::NProtocol::PT_NAVTELECOM:
            return MakeHolder<NDrive::NNavTelecom::TMessage>(NDrive::NNavTelecom::MT_INCORRECT, NDrive::NNavTelecom::TMessage::ESource::Client);
        default:
            return nullptr;
        };
    }
}

class NDrive::TTelematicsTestClient::TAutoFlushGuard {
public:
    TAutoFlushGuard(NDrive::TTelematicsTestClient& client) noexcept
        : Client(client)
    {
    }
    ~TAutoFlushGuard() noexcept(false) {
        Client.Flush();
    }

private:
    NDrive::TTelematicsTestClient& Client;
};

class NDrive::TTelematicsTestClient::TPeriodicFlush: public TGlobalScheduler::TScheduledItem<TPeriodicFlush> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<TPeriodicFlush>;

public:
    TPeriodicFlush(NDrive::TTelematicsTestClient& client, TInstant now, TDuration period = TDuration::Seconds(1)) noexcept
        : TBase(client.Name(), "PeriodicFlush:" + client.Name(), now + period)
        , Client(client)
        , Period(period)
    {
    }

    void Process(void* /*threadSpecificResource*/) override {
        auto cleanup = Hold(this);
        Client.Flush();
    }
    THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
        return MakeHolder<TPeriodicFlush>(Client, now, Period);
    }

private:
    NDrive::TTelematicsTestClient& Client;
    TDuration Period;
};

class NDrive::TTelematicsTestClient::TScheduledMessage: public TGlobalScheduler::TScheduledItem<TScheduledMessage> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<TScheduledMessage>;

public:
    TScheduledMessage(TTelematicsTestClient& client, THolder<TMessage>&& message, TInstant time)
    : TBase(client.Name(), "Message:" + client.GetIMEI() + ":" + ToString<const void*>(&client), time)
    , Client(client)
    , Message(std::move(message))
    {
    }

    void Process(void* /*threadSpecificResource*/) override {
        THolder<TScheduledMessage> cleanup(this);
        Client.Send(std::move(Message));
        Client.Flush();
    }

private:
    TTelematicsTestClient& Client;
    THolder<TMessage> Message;
};

NDrive::TTelematicsTestClient::TTelematicsTestClient(const TString& imei, TAtomicSharedPtr<NAsio::TExecutorsPool> executorsPool, const TString& id, const TOptions& options)
    : IMEI(imei)
    , OwnedExecutorsPool(executorsPool ? nullptr : MakeAtomicShared<NAsio::TExecutorsPool>(static_cast<size_t>(4)))
    , ExecutorsPool(executorsPool ? executorsPool : OwnedExecutorsPool)
    , Id(id)
    , Options(options)
    , FlushSize(1 << 13)
{
    GlobalSchedulerRegistrator.emplace(Name());
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TPeriodicFlush>(*this, Now())));
}

NDrive::TTelematicsTestClient::~TTelematicsTestClient() {
    GlobalSchedulerRegistrator.reset();
    Drop();
    if (OwnedExecutorsPool) {
        OwnedExecutorsPool->SyncShutdown();
    }
}

bool NDrive::TTelematicsTestClient::Alive() const {
    auto guard = Guard(Lock);
    if (Socket) {
        return Socket->IsOpen();
    } else {
        return false;
    }
}

void NDrive::TTelematicsTestClient::Connect(const TString& host, ui16 port, bool blocking) {
    Drop();
    auto guard = Guard(Lock);
    auto ev = blocking ? MakeAtomicShared<TAutoEvent>() : nullptr;
    CHECK_WITH_LOG(ExecutorsPool);
    CHECK_WITH_LOG(!Socket);
    Socket = MakeHolder<NAsio::TTcpSocket>(ExecutorsPool->GetExecutor().GetIOService());

    TNetworkAddress addr(host, port);
    for (TNetworkAddress::TIterator it = addr.Begin(); it != addr.End(); ++it) {
        TEndpoint ep(new NAddr::TAddrInfo(&*it));
        Socket->AsyncConnect(ep, [this, ev](const NAsio::TErrorCode& ec, NAsio::IHandlingContext&) {
            if (ev) {
                ev->Signal();
            }
            if (!ec) {
                Read();
            } else {
                ERROR_LOG << Id << " cannot async connect: " << ec.Value() << " " << ec.Text() << Endl;
                Drop();
            }
        }, Options.ConnectionTimeout);
        break;
    }
    if (ev) {
        ev->Wait();
    }
}

void NDrive::TTelematicsTestClient::Drop() {
    auto guard = Guard(Lock);
    if (Socket) {
        INFO_LOG << Id << " dropping client " << IMEI << Endl;
        for (auto&& handler : Handlers) {
            CHECK_WITH_LOG(handler);
            handler->OnDrop();
        }

        Authorized = false;

        NAsio::TErrorCode ec;
        Socket->AsyncCancel();
        Socket->Shutdown(NAsio::TTcpSocket::ShutdownBoth, ec);
        if (ec) {
            ERROR_LOG << Id << " cannot shutdown: " << ec.Value() << " " << ec.Text() << Endl;
        }
        Socket.Destroy();
    }
}

void NDrive::TTelematicsTestClient::AddHandler(THandlerPtr handler) {
    CHECK_WITH_LOG(handler);
    auto guard = Guard(Lock);
    Handlers.push_back(std::move(handler));
    Handlers.back()->OnAdd(*this);
}

void NDrive::TTelematicsTestClient::AddMessageHandler(THandlerPtr handler) {
    AddHandler(handler);
}

void NDrive::TTelematicsTestClient::SendMessage(THolder<TMessage>&& message) {
    if (ResponseDelay) {
        ScheduleMessage(std::move(message), Now() + ResponseDelay);
    } else {
        Send(std::move(message));
    }
}


void NDrive::TTelematicsTestClient::Send(std::string_view data) {
    auto guard = Guard(Lock);
    TAsioSocketOutput so(*Socket, Id);
    TFlushBufferedOutputStream output(so, FlushSize);

    ::SavePodArray(&output, data.data(), data.size());
    output.Flush();
}

void NDrive::TTelematicsTestClient::Read() {
    auto guard = Guard(Lock);
    if (Socket) {
        DEBUG_LOG << Id << " setting AsyncPollRead handler" << Endl;
        Socket->AsyncPollRead([this] (const NAsio::TErrorCode& ec, NAsio::IHandlingContext&) {
            if (!ec) {
                auto guard = Guard(Lock);
                if (Socket) {
                    TBuffer buffer(1024 * 1024);
                    NAsio::TErrorCode ec2;
                    buffer.Advance(Socket->ReadSome(buffer.Data(), buffer.Capacity(), ec2));
                    if (!ec2 && !buffer.Empty()) {
                        Read(buffer);
                        Read();
                    } else {
                        ERROR_LOG << Id << " cannot sync read: " << ec.Value() << " " << ec.Text() << Endl;
                        Drop();
                    }
                } else {
                    ERROR_LOG << Id << " cannot sync read: connection dropped" << Endl;
                }
            } else if (ec.Value() == ECANCELED) {
                DEBUG_LOG << Id << " operation cancelled" << Endl;
                Drop();
            } else {
                ERROR_LOG << Id << " cannot async read: " << ec.Value() << " " << ec.Text() << Endl;
                Drop();
            }
        }, Options.ReadTimeout);
    }
}

void NDrive::TTelematicsTestClient::Read(const TBuffer& buffer) {
    TAutoFlushGuard flush(*this);

    const TBuffer* source = nullptr;
    if (Buffer.Empty()) {
        source = &buffer;
    } else {
        Buffer.Append(buffer.Data(), buffer.Size());
        source = &Buffer;
    }
    CHECK_WITH_LOG(source);

    TMemoryInput input(source->Data(), source->Size());
    TMessageInput<NDrive::NProtocol::IMessage> messages(input);
    while (!input.Exhausted()) {
        const auto start = input.Buf();
        try {
            auto message = CreateMessage(Protocol);
            CHECK_WITH_LOG(message);
            messages.Get(*message);
            if (!message->IsValid()) {
                WARNING_LOG << Id << " invalid message" << Endl;
                break;
            }
            Process(*message);
        } catch (const NDrive::TIncompleteDataException& e) {
            DEBUG_LOG << Id << " incomplete data: " << e.what() << Endl;
            Buffer.Assign(start, source->End());
            return;
        } catch (const std::exception& e) {
            ERROR_LOG << Id << " an exception occurred: " << FormatExc(e) << Endl;
        }
    }
    Buffer.Clear();
}

void NDrive::TTelematicsTestClient::Process(const TMessage& message) {
    DEBUG_LOG << Id << " received " << message.DebugString() << Endl;
    if (message.GetProtocolType() == NProtocol::PT_VEGA) {
        switch (message.GetMessageType()) {
        case NVega::AUTH_REQUEST: {
            NVega::TMessage response(NVega::AUTH_RESPONSE);
            Authorized = true;
            response.As<NVega::TAuthorizationResponse>().Status = NVega::TAuthorizationResponse::Success;
            Flush(response);
            break;
        }
        case NVega::PING_REQUEST: {
            NVega::TMessage response(NVega::PING_RESPONSE);
            auto& payload = response.As<NVega::TPingResponse>();
            payload.IMEI.Set(IMEI);
            payload.DeviceType = Options.DeviceType;
            Flush(response);
            break;
        }
        case NVega::COMMAND_REQUEST: {
            if (CommandResponseErrorFlag) {
                const auto& request = message.As<NVega::TCommandRequest>();
                auto response = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
                auto& commandResponse = response->As<NVega::TCommandResponse>();
                commandResponse.Id = request.Id;
                commandResponse.Result = commandResponse.ERROR;
                Send(std::move(response));
                INFO_LOG << Id << " responsed error " << message.DebugString() << Endl;
                return;
            }
            break;
        }
        default:
            break;
        }
    }
    if (auto guard = Guard(Lock)) {
        THandlers survivors;
        survivors.reserve(Handlers.size());
        for (auto&& handler : Handlers) {
            NDrive::TTelematicsTestClient::EHandlerStatus status = NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
            try {
                CHECK_WITH_LOG(handler);
                status = handler->OnMessage(*this, message);
            } catch (const std::exception& e) {
                ERROR_LOG << Id << " handler exception " << FormatExc(e) << Endl;
            }
            if (status == NDrive::TTelematicsTestClient::EHandlerStatus::Continue) {
                survivors.push_back(handler);
            }
        }
        Handlers = std::move(survivors);
    }
    DEBUG_LOG << Id << " processed " << message.DebugString() << Endl;
}

void NDrive::TTelematicsTestClient::Send(THolder<TMessage>&& message) {
    DEBUG_LOG << Id << " queueing " << message->DebugString() << Endl;
    auto guard = Guard(Lock);
    MessageQueue.push_back(std::move(message));
}

void NDrive::TTelematicsTestClient::Flush() {
    auto guard = Guard(Lock);
    while (Alive() && Authorized && !MessageQueue.empty()) {
        Flush(*MessageQueue.front());
        MessageQueue.pop_front();
    }
}

void NDrive::TTelematicsTestClient::Flush(const TMessage& message) {
    DEBUG_LOG << Id << " sending " << message.DebugString() << Endl;

    Y_ENSURE(Socket);
    TAsioSocketOutput so(*Socket, Id);
    TFlushBufferedOutputStream output(so, FlushSize);
    NProtocol::TMessageOutput messages(output);
    messages.Send(message);
    DEBUG_LOG << Id << " sent " << message.DebugString() << Endl;
}

void NDrive::TTelematicsTestClient::SetAuthorized(bool value) {
    Authorized = value;
}

void NDrive::TTelematicsTestClient::SetCommandResponseError(bool value) {
    CommandResponseErrorFlag = value;
}

void NDrive::TTelematicsTestClient::SetResponseDelay(TDuration delay) {
    ResponseDelay = delay;
}

void NDrive::TTelematicsTestClient::SetProtocol(NDrive::NProtocol::EProtocolType protocol) {
    Protocol = protocol;
}

void NDrive::TTelematicsTestClient::ScheduleMessage(THolder<TMessage>&& response, TInstant time) {
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TScheduledMessage>(*this, std::move(response), time)));
}
