#pragma once

#include "context.h"

#include <drive/telematics/common/handler.h>
#include <drive/telematics/protocol/vega.h>
#include <rtline/library/scheduler/global.h>
#include <rtline/library/scheduler/registrator.h>

#include <library/cpp/neh/asio/executor.h>

#include <util/datetime/base.h>
#include <util/generic/buffer.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/system/mutex.h>

namespace NAsio {
    class TExecutorsPool;
    class TTcpSocket;
}

namespace NDrive {
    class TTelematicsTestClient final: public NVega::IConnection {
    public:
        using IHandler = NProtocol::IHandler;
        using THandlerPtr = NProtocol::THandlerPtr;
        using THandlers = TVector<THandlerPtr>;
        using TMessage = NProtocol::IMessage;
        using TMessageQueue = std::deque<THolder<TMessage>>;
        using TProtocolType = NProtocol::TProtocolType;
        using EHandlerStatus = NProtocol::EHandlerStatus;

        using TPtr = TAtomicSharedPtr<TTelematicsTestClient>;

        struct TOptions {
            TDuration ConnectionTimeout = TDuration::Seconds(10);
            TDuration ReadTimeout = TDuration::Seconds(30);
            ui8 DeviceType = 0;
            TAtomicSharedPtr<NAsio::TExecutorsPool> ExecutorsPool;
        };

    public:
        TTelematicsTestClient(const TString& imei, TAtomicSharedPtr<NAsio::TExecutorsPool> executorsPool = Default<TAtomicSharedPtr<NAsio::TExecutorsPool>>(), const TString& id = "TelematicsClient", const TOptions& options = Default<TOptions>());
        TTelematicsTestClient(TTelematicsTestClient&&) = default;
        ~TTelematicsTestClient();

        bool Alive() const;
        void Connect(const TString& host, ui16 port, bool blocking);
        void Drop() override;

        void AddHandler(THandlerPtr handler);
        virtual void AddMessageHandler(THandlerPtr handler) override;
        virtual void SendMessage(THolder<TMessage>&& message) override;

        void Send(std::string_view data);

        void SetAuthorized(bool value);
        void SetCommandResponseError(bool value);
        void SetResponseDelay(TDuration delay);
        void SetProtocol(NDrive::NProtocol::EProtocolType protocol);

        virtual const TString& GetId() const override {
            return GetIMEI();
        }

        const TString& GetIMEI() const {
            return IMEI;
        }

        TString Name() const {
            return "TTelematicsTestClient:" + IMEI + ':' + ToString<const void*>(this);
        }

    private:
        class TAutoFlushGuard;
        class TPeriodicFlush;
        class TScheduledMessage;

    private:
        void Flush();
        void Flush(const TMessage& message);

        void Read();
        void Read(const TBuffer& buffer);

        void Process(const TMessage& message);
        void Send(THolder<TMessage>&& message);
        void ScheduleMessage(THolder<TMessage>&& response, TInstant time);

    private:
        const TString IMEI;
        const TAtomicSharedPtr<NAsio::TExecutorsPool> OwnedExecutorsPool;
        const TAtomicSharedPtr<NAsio::TExecutorsPool> ExecutorsPool;
        const TString Id;
        const TOptions Options;

        THolder<NAsio::TTcpSocket> Socket;

        THandlers Handlers;
        TMessageQueue MessageQueue;
        TMutex Lock;
        TBuffer Buffer;
        size_t FlushSize;
        bool Authorized = false;
        NDrive::NProtocol::EProtocolType Protocol = NDrive::NProtocol::PT_VEGA;

        bool CommandResponseErrorFlag = false;
        TDuration ResponseDelay;

        NGlobalScheduler::TOptionalRegistrator GlobalSchedulerRegistrator;
    };
}
