#include <util/stream/output.h>
#include "context.h"

using namespace NDrive;

bool TTelematicsClientContext::TrySetDrDoorOpened(bool value) {
    auto e = !value;
    return DrDoorOpened.compare_exchange_strong(e, value);
}

bool TTelematicsClientContext::GetDrDoorOpened() const {
    return DrDoorOpened.load();
}

bool TTelematicsClientContext::TrySetDoorsOpened(bool value) {
    auto guard = Guard(Lock);
    DoorsState.DriverDoorOpened = value;
    DoorsState.PassDoorOpened = value;
    DoorsState.LRearDoorOpened = value;
    DoorsState.RRearDoorOpened = value;
    return true;
}

void TTelematicsClientContext::SetTransmissionStatus(ETransmissionStatus status) {
    auto guard = Guard(Lock);
    TransmissionStatus = status;
}

NDrive::TTelematicsClientContext::ETransmissionStatus TTelematicsClientContext::GetTransmissionStatus() const {
    auto guard = Guard(Lock);
    return TransmissionStatus;
}

bool TTelematicsClientContext::TrySetEngineStarted(bool value) {
    auto guard = Guard(Lock);
    if (TransmissionStatus == Parking) {
        EngineStarted = value;
        return true;
    }
    return false;
}

bool TTelematicsClientContext::GetEngineStarted() const {
    auto guard = Guard(Lock);
    return EngineStarted;
}

bool TTelematicsClientContext::TrySetWarming(bool value) {
    auto guard = Guard(Lock);
    if (TransmissionStatus == Parking && EngineStarted != value && Warming != value) {
        EngineStarted = value;
        Warming = value;
        return true;
    }
    return false;
}

bool TTelematicsClientContext::GetWarmingStatus() const {
    auto guard = Guard(Lock);
    return Warming;
}

TTelematicsClientContext::TDoorsState TTelematicsClientContext::GetDoorsState() const {
    auto guard = Guard(Lock);
    return DoorsState;
}

bool TTelematicsClientContext::GetHoodOpened() const {
    auto guard = Guard(Lock);
    return HoodOpened;
}

bool TTelematicsClientContext::GetTrunkOpened() const {
    auto guard = Guard(Lock);
    return TrunkOpened;
}

ui8 TTelematicsClientContext::GetFuelPercent() const {
    auto guard = Guard(Lock);
    return FuelPercent;
}

void TTelematicsClientContext::SetFuelPercent(ui8 value) {
    auto guard = Guard(Lock);
    FuelPercent = value;
}

TMaybe<double> TTelematicsClientContext::GetTankerFuelLevel() const {
    auto guard = Guard(Lock);
    return TankerFuelLevel;
}

void TTelematicsClientContext::SetTankerFuelLevel(double value) {
    auto guard = Guard(Lock);
    TankerFuelLevel = value;
}

TMaybe<double> TTelematicsClientContext::GetTankerSecondFuelLevel() const {
    auto guard = Guard(Lock);
    return TankerSecondFuelLevel;
}

void TTelematicsClientContext::SetTankerSecondFuelLevel(double value) {
    auto guard = Guard(Lock);
    TankerSecondFuelLevel = value;
}

float TTelematicsClientContext::GetOdometerKm() const {
    auto guard = Guard(Lock);
    return OdometerKm;
}

void TTelematicsClientContext::SetOdometerKm(float value) {
    auto guard = Guard(Lock);
    OdometerKm = value;
}

ui32 TTelematicsClientContext::GetMCC() const {
    auto guard = Guard(Lock);
    return MCC;
}

void TTelematicsClientContext::SetMCC(ui32 value) {
    auto guard = Guard(Lock);
    MCC = value;
}

ui32 TTelematicsClientContext::GetMNC() const {
    auto guard = Guard(Lock);
    return MNC;
}

void TTelematicsClientContext::SetMNC(ui32 value) {
    auto guard = Guard(Lock);
    MNC = value;
}

ui32 TTelematicsClientContext::GetLAC() const {
    auto guard = Guard(Lock);
    return LAC;
}

void TTelematicsClientContext::SetLAC(ui32 value) {
    auto guard = Guard(Lock);
    LAC = value;
}

ui32 TTelematicsClientContext::GetCellID() const {
    auto guard = Guard(Lock);
    return CellID;
}

void TTelematicsClientContext::SetCellID(ui32 value) {
    auto guard = Guard(Lock);
    CellID = value;
}

float TTelematicsClientContext::GetVegaGSMSignalLevel() const {
    auto guard = Guard(Lock);
    return VegaGSMSignalLevel;
}

void TTelematicsClientContext::SetVegaGSMSignalLevel(ui32 value) {
    auto guard = Guard(Lock);
    VegaGSMSignalLevel = value;
}

float TTelematicsClientContext::GetHDOP() const {
    auto guard = Guard(Lock);
    return HDOP;
}

void TTelematicsClientContext::SetHDOP(float value) {
    auto guard = Guard(Lock);
    HDOP = value;
}

void TTelematicsClientContext::SetCurrentPosition(const TGeoCoord& value) {
    Y_ENSURE(value.X <= 180);
    Y_ENSURE(value.X >= -180);
    Y_ENSURE(value.Y <= 90);
    Y_ENSURE(value.Y >= -90);
    auto guard = Guard(Lock);
    CurrentPosition = value;
}

TGeoCoord TTelematicsClientContext::GetCurrentPosition() const {
    auto guard = Guard(Lock);
    return CurrentPosition;
}

bool TTelematicsClientContext::TrySetSpeed(double value) {
    auto guard = Guard(Lock);
    if (EngineStarted) {
        Speed = value;
        return true;
    }
    return false;
}

double TTelematicsClientContext::GetSpeed() const {
    auto guard = Guard(Lock);
    return Speed;
}

bool TTelematicsClientContext::StartLeasing() {
    auto guard = Guard(Lock);
    LeasingState = true;
    return true;
}

bool TTelematicsClientContext::EndLeasing() {
    auto guard = Guard(Lock);
    if (Speed == 0) {
        LeasingState = false;
        TrySetDoorsOpened(false);
        return true;
    }
    return false;
}

bool TTelematicsClientContext::GetDoorsOpened() const {
    auto guard = Guard(Lock);
    return DoorsState.DriverDoorOpened && DoorsState.PassDoorOpened && DoorsState.LRearDoorOpened && DoorsState.RRearDoorOpened;
}

TString NDrive::TTelematicsClientContext::GetMcuFirmwareVersion() const {
    auto guard = Guard(Lock);
    return McuFirmwareVersion;
}

NDrive::NVega::TSensorTranslation NDrive::TTelematicsClientContext::GetSensorTranslation(ui16 id) const {
    auto guard = Guard(Lock);
    auto p = SensorTranslationSettings.find(id);
    if (p != SensorTranslationSettings.end()) {
        return p->second;
    } else {
        return {};
    }
}

NDrive::NVega::TAPNParameter NDrive::TTelematicsClientContext::GetAPN() const {
    auto guard = Guard(Lock);
    return APNSettings;
}

NDrive::NVega::TServerSettingsParameter NDrive::TTelematicsClientContext::GetServerSettings() const {
    auto guard = Guard(Lock);
    return ServerSettings;
}

TString NDrive::TTelematicsClientContext::GetServerPin() const {
    auto guard = Guard(Lock);
    return ServerPin;
}

TString NDrive::TTelematicsClientContext::GetWiredPin() const {
    auto guard = Guard(Lock);
    return WiredPin;
}

bool NDrive::TTelematicsClientContext::GetLeasingState() const {
    return LeasingState;
}

bool NDrive::TTelematicsClientContext::UseServerPin() const {
    return UseServerPinFlag;
}

bool NDrive::TTelematicsClientContext::UseWiredPin() const {
    return UseWiredPinFlag;
}

void NDrive::TTelematicsClientContext::SetSensorTranslation(ui16 id, const NVega::TSensorTranslation& value) {
    auto guard = Guard(Lock);
    SensorTranslationSettings[id] = value;
}

void NDrive::TTelematicsClientContext::SetAPN(const NVega::TAPNParameter& value) {
    auto guard = Guard(Lock);
    APNSettings = value;
}

void NDrive::TTelematicsClientContext::SetServerSettings(const NVega::TServerSettingsParameter& value) {
    auto guard = Guard(Lock);
    ServerSettings = value;
}

void NDrive::TTelematicsClientContext::SetServerPin(const TString& value) {
    auto guard = Guard(Lock);
    ServerPin = value;
}

void NDrive::TTelematicsClientContext::SetWiredPin(const TString& value) {
    auto guard = Guard(Lock);
    WiredPin = value;
}

bool NDrive::TTelematicsClientContext::GetEnableCanResponse() const {
    return EnableCanResponse;
}

void NDrive::TTelematicsClientContext::SetEnableCanResponse(bool value) {
    EnableCanResponse = value;
}

TString NDrive::TTelematicsClientContext::GetBleMac() const {
    auto guard = Guard(Lock);
    return BleMac;
}

TString NDrive::TTelematicsClientContext::GetBlePasskey() const {
    auto guard = Guard(Lock);
    return BlePasskey;
}

TString NDrive::TTelematicsClientContext::GetBleSessionKey() const {
    auto guard = Guard(Lock);
    auto raw = BleSessionKey.Raw();
    return { raw.data(), raw.size() };
}

void NDrive::TTelematicsClientContext::SetBlePasskey(const TString& value) {
    Y_ENSURE(value.size() == 6);
    for (auto&& c : value) {
        Y_ENSURE(c >= '0' && c <= '9', "incorrect character in BLE passkey: " << c);
    }
    auto guard = Guard(Lock);
    BlePasskey = value;
}

void NDrive::TTelematicsClientContext::SetBleSessionKey(TConstArrayRef<char> value) {
    Y_ENSURE(value.size() == BleSessionKey.Value.size());
    auto guard = Guard(Lock);
    MemCopy(BleSessionKey.Value.data(), reinterpret_cast<const ui8*>(value.data()), value.size());
}

void NDrive::TTelematicsClientContext::AdvancePath() {
    if (PathPosition >= Path.GetLength()) {
        return;
    }
    auto now = Now();
    auto duration = now - PathUpdated;
    auto distance = PathSpeed / 3600 * 1000 * duration.SecondsFloat();
    PathUpdated = now;
    PathPosition += distance;
    OdometerKm += static_cast<float>(distance * 0.001);
    CurrentPosition = Path.GetCoordByLength(PathPosition);
    auto speed = static_cast<ui32>(distance / duration.Seconds() * 3600 / 1000);
    Y_ENSURE(TrySetEngineStarted(true));
    Y_ENSURE(TrySetSpeed(speed));
    if (PathPosition >= Path.GetLength()) {
        Y_ENSURE(TrySetSpeed(0));
        Y_ENSURE(TrySetEngineStarted(false));
    }
}

void NDrive::TTelematicsClientContext::SetPath(TGeoPolyLine&& value) {
    Path = std::move(value);
    PathPosition = 0;
    PathUpdated = Now();
}

void NDrive::TTelematicsClientContext::SetPathSpeed(double value) {
    PathSpeed = value;
}

void NDrive::TTelematicsClientContext::SetBeaconsInfos(TVector<NVega::TBeaconInfos>&& infos) {
    BeaconsInfos = std::move(infos);
}

const TVector<NDrive::NVega::TBeaconInfos>& NDrive::TTelematicsClientContext::GetBeaconsInfos() const {
    return BeaconsInfos;
}

TVector<NDrive::NVega::TBeaconInfos>& NDrive::TTelematicsClientContext::GetBeaconsInfos() {
    return BeaconsInfos;
}

void NDrive::TTelematicsClientContext::SetCustomSensors(TVector<NDrive::TSensor>&& sensors) {
    auto guard = Guard(Lock);
    CustomSensors = std::move(sensors);
}

const TVector<NDrive::TSensor> NDrive::TTelematicsClientContext::GetCustomSensors() const {
    auto guard = Guard(Lock);
    return CustomSensors;
}
