#pragma once

#include <drive/telematics/common/beacon.h>
#include <drive/telematics/protocol/settings.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/polyline.h>

#include <util/system/mutex.h>

namespace NDrive {
    class TTelematicsClientContext {
    public:
        enum ETransmissionStatus {
            Parking,
            Reverse,
            Neutral,
            Drive,
            Low
        };

        struct TDoorsState {
            bool DriverDoorOpened = false;
            bool PassDoorOpened = false;
            bool LRearDoorOpened = false;
            bool RRearDoorOpened = false;
        };

    public:
        TTelematicsClientContext() = default;

        TString GetMcuFirmwareVersion() const;

        NVega::TSensorTranslation GetSensorTranslation(ui16 id) const;
        NVega::TAPNParameter GetAPN() const;
        NVega::TServerSettingsParameter GetServerSettings() const;
        TString GetServerPin() const;
        TString GetWiredPin() const;
        bool GetLeasingState() const;
        bool UseServerPin() const;
        bool UseWiredPin() const;

        void SetSensorTranslation(ui16 id, const NVega::TSensorTranslation& value);
        void SetAPN(const NVega::TAPNParameter& value);
        void SetServerSettings(const NVega::TServerSettingsParameter& value);
        void SetServerPin(const TString& value);
        void SetWiredPin(const TString& value);

        void SetTransmissionStatus(ETransmissionStatus status);
        ETransmissionStatus GetTransmissionStatus() const;

        bool TrySetEngineStarted(bool value);
        bool GetEngineStarted() const;

        bool TrySetWarming(bool value);
        bool GetWarmingStatus() const;

        bool TrySetDrDoorOpened(bool value);
        bool GetDrDoorOpened() const;

        bool TrySetDoorsOpened(bool value);

        TDoorsState GetDoorsState() const;

        bool GetHoodOpened() const;
        bool GetTrunkOpened() const;

        ui8 GetFuelPercent() const;
        void SetFuelPercent(ui8 value);

        TMaybe<double> GetTankerFuelLevel() const;
        void SetTankerFuelLevel(double value);
        TMaybe<double> GetTankerSecondFuelLevel() const;
        void SetTankerSecondFuelLevel(double value);

        float GetOdometerKm() const;
        void SetOdometerKm(float value);

        ui32 GetMCC() const;
        void SetMCC(ui32 value);

        ui32 GetMNC() const;
        void SetMNC(ui32 value);

        ui32 GetLAC() const;
        void SetLAC(ui32 value);

        ui32 GetCellID() const;
        void SetCellID(ui32 value);

        float GetVegaGSMSignalLevel() const;
        void SetVegaGSMSignalLevel(ui32 value);

        float GetHDOP() const;
        void SetHDOP(float value);

        void SetCurrentPosition(const TGeoCoord& value);
        TGeoCoord GetCurrentPosition() const;

        bool TrySetSpeed(double value);
        double GetSpeed() const;

        bool StartLeasing();
        bool EndLeasing();

        bool GetDoorsOpened() const;

        bool GetEnableCanResponse() const;
        void SetEnableCanResponse(bool value);

        TString GetBleMac() const;
        TString GetBlePasskey() const;
        TString GetBleSessionKey() const;
        void SetBlePasskey(const TString& value);
        void SetBleSessionKey(TConstArrayRef<char> value);

        void AdvancePath();
        void SetPath(TGeoPolyLine&& value);
        void SetPathSpeed(double value);

        void SetBeaconsInfos(TVector<NVega::TBeaconInfos>&& infos);
        const TVector<NDrive::NVega::TBeaconInfos>& GetBeaconsInfos() const;
        TVector<NDrive::NVega::TBeaconInfos>& GetBeaconsInfos();

        void SetCustomSensors(TVector<NDrive::TSensor>&& sensors);
        const TVector<NDrive::TSensor> GetCustomSensors() const;

    private:
        TMutex Lock;

        TString McuFirmwareVersion = "VEGA MT-32K LTE V2 0.10b rc40 'Emulator'";

        TMap<ui16, NVega::TSensorTranslation> SensorTranslationSettings;
        NVega::TAPNParameter APNSettings;
        NVega::TServerSettingsParameter ServerSettings;
        TString ServerPin = "P@ssw0rd";
        TString WiredPin = "WiredPin";
        bool UseServerPinFlag = true;
        bool UseWiredPinFlag = true;

        ETransmissionStatus TransmissionStatus = ETransmissionStatus::Parking;

        bool EngineStarted = false;
        bool Warming = false;

        std::atomic<bool> DrDoorOpened = false;
        TDoorsState DoorsState;

        bool HoodOpened = false;
        bool TrunkOpened = false;

        ui8 FuelPercent = 100;
        float OdometerKm = 10;
        TMaybe<double> TankerFuelLevel;
        TMaybe<double> TankerSecondFuelLevel;

        ui32 MCC = 250;
        ui64 MNC = 99;
        ui32 LAC = 27912;
        ui32 CellID = 13895;
        float VegaGSMSignalLevel = 30;

        float HDOP = 50;

        double Speed = 0;

        TGeoCoord CurrentPosition;
        TGeoPolyLine Path;
        TInstant PathUpdated;
        double PathPosition = 0;
        double PathSpeed = 90;

        bool EnableCanResponse = false;
        bool LeasingState = false;

        TString BleMac = "00224D6A5E81";
        TString BlePasskey = "321123";
        NDrive::NVega::TBleSessionKeyParameter BleSessionKey;
        TVector<NVega::TBeaconInfos> BeaconsInfos{10};
        TVector<NDrive::TSensor> CustomSensors;
    };
}
