#include "handlers.h"

using namespace NDrive;

TTelematicsTestClient::EHandlerStatus TOnHeartbeat::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& /*message*/) {
    Signal();
    return TTelematicsTestClient::EHandlerStatus::Finish;
}

void NDrive::TAuthorizationHandler::OnAdd(NVega::IConnection& connection) {
    if (Password) {
        auto hard = MakeHolder<NVega::THardAuthorizationRequest>();
        hard->Password.Set(Password);
        connection.SendMessage(MakeHolder<NVega::TMessage>(std::move(hard)));
    } else {
        auto request = MakeHolder<NVega::TMessage>(NVega::AUTH_REQUEST);
        connection.SendMessage(std::move(request));
    }
}

TTelematicsTestClient::EHandlerStatus NDrive::TAuthorizationHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NVega::AUTH_RESPONSE) {
        const auto& response = message.As<NVega::TAuthorizationResponse>();
        Status = response.Status;
        Signal();
        return TTelematicsTestClient::EHandlerStatus::Finish;
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

void NDrive::TListDevicesHandler::OnAdd(NVega::IConnection& connection) {
    auto request = MakeHolder<NVega::TMessage>(NVega::LIST_REQUEST);
    connection.SendMessage(std::move(request));
}

TTelematicsTestClient::EHandlerStatus NDrive::TListDevicesHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NVega::LIST_RESPONSE) {
        const auto& response = message.As<NVega::TListResponse>();
        for (auto&& i : response.Devices) {
            IMEIs.push_back(i.IMEI.Get());
        }
        if (response.Devices.size() < 1000) {
            Signal();
            return TTelematicsTestClient::EHandlerStatus::Finish;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

void NDrive::TConnectDeviceHandler::OnAdd(NVega::IConnection& connection) {
    auto message = MakeHolder<NVega::TMessage>(NVega::CONNECT_REQUEST);
    message->As<NDrive::NVega::TConnectRequest>().Device.IMEI.Set(IMEI);
    connection.SendMessage(std::move(message));
}

TTelematicsTestClient::EHandlerStatus NDrive::TConnectDeviceHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NVega::CONNECT_RESPONSE) {
        const auto& response = message.As<NVega::TConnectResponse>();
        Result = static_cast<NVega::TConnectResponse::EResult>(response.Result);
        Signal();
        return TTelematicsTestClient::EHandlerStatus::Finish;
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

void NDrive::TBlackboxHandler::OnAdd(NVega::IConnection& connection) {
    auto message = MakeHolder<NVega::TMessage>(NVega::BLACKBOX_RECORDS);
    auto& payload = message->As<NVega::TBlackboxRecords>();
    payload.Id = BlackboxMessageId;
    payload.Records = std::move(Records);
    connection.SendMessage(std::move(message));
}

TTelematicsTestClient::EHandlerStatus NDrive::TBlackboxHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NVega::BLACKBOX_RECORDS_ACK) {
        const auto& response = message.As<NVega::TBlackboxRecordsAck>();
        if (BlackboxMessageId == response.GetSequenceId()) {
            Result = static_cast<NVega::TBlackboxRecordsAck::EResult>(response.Result);
            Signal();
            return TTelematicsTestClient::EHandlerStatus::Finish;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

void NDrive::TSendCommandHandler::OnAdd(NVega::IConnection& connection) {
    auto message = MakeHolder<NVega::TMessage>(NVega::COMMAND_REQUEST);
    message->As<NVega::TCommandRequest>() = Request;
    connection.SendMessage(std::move(message));
}

TTelematicsTestClient::EHandlerStatus NDrive::TSendCommandHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NVega::COMMAND_RESPONSE && message.GetSequenceId() == Request.Id) {
        const auto& response = message.As<NVega::TCommandResponse>();
        Result = static_cast<NVega::TCommandResponse::EResult>(response.Result);
        if (Result == NVega::TCommandResponse::PROCESSED) {
            switch (Request.Code) {
            case NVega::ECommandCode::GET_PARAM:
                Value = response.Argument.Get<NVega::TCommandResponse::TGetParameter>();
                break;
            default:
                break;
            }
        }
        Signal();
        return TTelematicsTestClient::EHandlerStatus::Finish;
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TTelematicsTestClient::EHandlerStatus THandlerWithoutParams::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NVega::TCommandRequest>();
        if (request.Code == CommandCode) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;

            if (ProcessCommand(connection)) {
                response.Result = response.PROCESSED;
            } else {
                response.Result = response.ERROR;
            }

            Client.SendMessage(std::move(ack));
            Signal();
            return TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TTelematicsTestClient::EHandlerStatus TSimpleCommandsHandler::OnMessage(NVega::IConnection& /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return TTelematicsTestClient::EHandlerStatus::Continue;
    }
    if (message.GetMessageType() == NVega::COMMAND_REQUEST) {
        const auto& request = message.As<NVega::TCommandRequest>();
        if (std::find(CommandCodes.begin(), CommandCodes.end(), request.Code) != CommandCodes.end()) {
            auto ack = MakeHolder<NVega::TMessage>(NVega::COMMAND_RESPONSE);
            auto& response = ack->As<NVega::TCommandResponse>();
            response.Id = request.Id;
            response.Result = response.PROCESSED;

            Client.SendMessage(std::move(ack));
            return TTelematicsTestClient::EHandlerStatus::Continue;
        }
    }
    return TTelematicsTestClient::EHandlerStatus::Continue;
}

TVector<NVega::ECommandCode> TSimpleCommandsHandler::CommandCodes = {
    NDrive::NVega::ECommandCode::TRUNK,
    NDrive::NVega::ECommandCode::HORN,
    NDrive::NVega::ECommandCode::EMULATE_DRIVER_DOOR,
    NDrive::NVega::ECommandCode::HORN_AND_BLINK,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_3S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_7S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_11S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_15S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_19S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_23S,
    NDrive::NVega::ECommandCode::WINDOWS_CLOSING_29S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_3S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_7S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_11S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_15S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_19S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_23S,
    NDrive::NVega::ECommandCode::WINDOWS_OPENING_29S,
    NDrive::NVega::ECommandCode::START_PROD_TESTS,
    NDrive::NVega::ECommandCode::RESET_LORA_SENS_ALARMS,
    NDrive::NVega::ECommandCode::YADRIVE_UNLOCK_HOOD,
    NDrive::NVega::ECommandCode::YADRIVE_LOCK_HOOD,
};

void NDrive::TCompositeHandler::OnAdd(NVega::IConnection& connection) {
    for (auto&& handler : Handlers) {
        handler->OnAdd(connection);
    }
}

TTelematicsTestClient::EHandlerStatus TCompositeHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    TTelematicsTestClient::THandlers survivors;
    survivors.reserve(Handlers.size());
    for (auto&& handler : Handlers) {
        if (handler->OnMessage(connection, message) == NDrive::TTelematicsTestClient::EHandlerStatus::Continue) {
            survivors.push_back(std::move(handler));
        }
    }
    Handlers = std::move(survivors);

    return Handlers ? NDrive::TTelematicsTestClient::EHandlerStatus::Continue : NDrive::TTelematicsTestClient::EHandlerStatus::Finish;
}
