#pragma once

#include "client.h"
#include "context.h"

#include <util/generic/vector.h>

namespace NDrive {
    using TCommonHandler = NProtocol::TCommonHandler;

    class THandlerWithoutParams: public TCommonHandler {
    public:
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) final;

        virtual bool ProcessCommand(NVega::IConnection& /*connection*/) {
            return true;
        };

    protected:
        THandlerWithoutParams(TTelematicsTestClient& client, NVega::ECommandCode commandCode)
            : Client(client)
            , CommandCode(commandCode)
        {
        }

    protected:
        TTelematicsTestClient& Client;
        NVega::ECommandCode CommandCode = NVega::ECommandCode::UNKNOWN;
    };

    class TOnHeartbeat: public TCommonHandler {
    public:
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;
    };

    class TAuthorizationHandler: public TCommonHandler {
    public:
        TAuthorizationHandler(const TString& password)
            : Password(password)
        {
        }

        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        ui8 GetStatus() const {
            return Status;
        }

    private:
        const TString Password;
        ui8 Status = NVega::TAuthorizationResponse::Failure;
    };

    class TListDevicesHandler: public TCommonHandler {
    public:
        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        const TVector<TString>& GetIMEIs() const {
            return IMEIs;
        }

    private:
        TVector<TString> IMEIs;
    };

    class TConnectDeviceHandler: public TCommonHandler {
    public:
        TConnectDeviceHandler(const TString& imei)
            : IMEI(imei)
        {
        }

        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        NVega::TConnectResponse::EResult GetResult() const {
            return Result;
        }

    private:
        const TString IMEI;
        NVega::TConnectResponse::EResult Result = NVega::TConnectResponse::UNKNOWN;
    };

    class TBlackboxHandler: public TCommonHandler {
    public:
        TBlackboxHandler(NProtocol::TArray<NVega::TBlackboxRecords::TRecord, ui16>&& records, ui64 messageId = 0)
            : Records(std::move(records))
            , BlackboxMessageId(messageId)
        {
        }

        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        NVega::TBlackboxRecordsAck::EResult GetResult() const {
            return Result;
        }

    private:
        NProtocol::TArray<NVega::TBlackboxRecords::TRecord, ui16> Records;
        ui64 BlackboxMessageId;
        NVega::TBlackboxRecordsAck::EResult Result = NVega::TBlackboxRecordsAck::Error;
    };

    class TSendCommandHandler: public TCommonHandler {
    public:
        TSendCommandHandler(const NVega::TCommandRequest& request)
            : Request(request)
            , Result(NVega::TCommandResponse::UNKNOWN)
        {
        }

        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        NVega::TCommandResponse::EResult GetResult() const {
            return Result;
        }
        NVega::TIdValue GetValue() const {
            return Value;
        }

    private:
        const NVega::TCommandRequest Request;

        NVega::TCommandResponse::EResult Result;
        NVega::TIdValue Value;
    };

    // handler with permanent "PROCESSED" result for simple command codes
    class TSimpleCommandsHandler: public TCommonHandler {
    public:
        TSimpleCommandsHandler(TTelematicsTestClient& client)
            : Client(client)
        {
        }

        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

    private:
        TTelematicsTestClient& Client;

        static TVector<NVega::ECommandCode> CommandCodes;
    };

    class TCompositeHandler: public TCommonHandler {
    public:
        void AddHandler(TTelematicsTestClient::THandlerPtr handler);

        void OnAdd(NVega::IConnection& connection) override;
        TTelematicsTestClient::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

    private:
        TTelematicsTestClient::THandlers Handlers;
    };
}
