#include "proxy.h"
#include "handlers.h"

#include <drive/telematics/common/file.h>

namespace {
    NDrive::TTelematicsTestClient::TOptions GetClientOptions() {
        NDrive::TTelematicsTestClient::TOptions result;
        result.DeviceType = 254;
        return result;
    }
}

NDrive::TTelematicsProxy::TTelematicsProxy(const TString& host, ui16 port)
    : Host(host)
    , Port(port)
    , Client("", nullptr, "Proxy", GetClientOptions())
    , MessageId(0)
{
    Client.SetAuthorized(true);
}

TVector<TString> NDrive::TTelematicsProxy::List(TDuration timeout /*= TDuration::Seconds(10)*/) {
    Client.Connect(Host, Port, true);
    auto list = NDrive::NProtocol::Handle(Client, MakeAtomicShared<NDrive::TListDevicesHandler>(), timeout);
    Client.Drop();
    return list->GetIMEIs();
}

TBuffer NDrive::TTelematicsProxy::GetFile(const TString& name, ssize_t offset, TDuration timeout /*= TDuration::Seconds(300)*/) {
    Y_ENSURE(Client.Alive(), "connection is dead");
    auto handler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<NDrive::NVega::TGetFileHandler>(name, offset), timeout);
    return std::move(handler->GetData());
}

void NDrive::TTelematicsProxy::SetFile(const TString& name, TBuffer&& content, TDuration timeout /*= TDuration::Seconds(300)*/) {
    Y_ENSURE(Client.Alive(), "connection is dead");
    auto handler = NDrive::NProtocol::Handle(
        Client,
        MakeAtomicShared<NDrive::NVega::TChunkedFileHandler>(name, std::move(content)),
        timeout
    );
    Y_ENSURE(handler->GetResult() == NDrive::NVega::TFileChunkResponse::OK, "cannot stream file: " << handler->GetResult());
}

void NDrive::TTelematicsProxy::Connect(const TString& imei, TConstArrayRef<TString> passwords, TDuration timeout /*= TDuration::Seconds(30)*/) {
    auto heartbeat = MakeAtomicShared<TOnHeartbeat>();
    Client.AddHandler(heartbeat);
    Client.Connect(Host, Port, true);
    heartbeat->Wait(timeout);
    Y_ENSURE(!heartbeat->WasDropped(), "connection was dropped");
    Y_ENSURE(Client.Alive(), "connection is dead");

    auto connectHandler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<TConnectDeviceHandler>(imei), timeout);
    Y_ENSURE(connectHandler->GetResult() == NVega::TConnectResponse::OK, "cannot connect to " << imei << ": " << connectHandler->GetResult());

    auto authorizationStatus = NVega::TAuthorizationResponse::Failure;
    for (auto&& password : passwords) {
        auto authorizeHandler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<TAuthorizationHandler>(password), timeout);
        authorizationStatus = authorizeHandler->GetStatus();
        if (authorizationStatus == NVega::TAuthorizationResponse::Success) {
            break;
        }
    }
    Y_ENSURE(authorizationStatus == NVega::TAuthorizationResponse::Success, "cannot authorize to " << imei);
}

void NDrive::TTelematicsProxy::Disconnect() {
    Client.Drop();
}

void NDrive::TTelematicsProxy::Command(NDrive::NVega::ECommandCode code, TDuration timeout /*= TDuration::Seconds(30)*/) {
    Y_ENSURE(Client.Alive(), "connection is dead");
    NVega::TCommandRequest request;
    request.Id = MessageId++;
    request.Code = code;
    auto handler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<TSendCommandHandler>(request), timeout);
    Y_ENSURE(handler->GetResult() == NVega::TCommandResponse::PROCESSED, "cannot send Command " << code << ": " << handler->GetResult());
}

NDrive::TSensorValue NDrive::TTelematicsProxy::GetParameter(ui16 id, ui16 subid /*= 0*/, TDuration timeout /*= TDuration::Seconds(10)*/) {
    Y_ENSURE(Client.Alive(), "connection is dead");

    NVega::TCommandRequest::TGetParameter parameter;
    parameter.Id = id;
    parameter.SubId = subid;

    NVega::TCommandRequest request;
    request.Id = MessageId++;
    request.Code = NVega::ECommandCode::GET_PARAM;
    request.Argument.Set(parameter);

    auto handler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<TSendCommandHandler>(request), timeout);
    Y_ENSURE(handler->GetResult() == NVega::TCommandResponse::PROCESSED, "cannot GetParameter " << id << ": " << handler->GetResult());
    return NDrive::SensorValueFromRef(handler->GetValue().GetValue());
}

template <class T>
void NDrive::TTelematicsProxy::SetParameter(ui16 id, ui16 subid, T value, TDuration timeout /*= TDuration::Seconds(10)*/) {
    Y_ENSURE(Client.Alive(), "connection is dead");

    NVega::TCommandRequest::TSetParameter parameter;
    parameter.Id = id;
    parameter.SubId = subid;
    parameter.SetValue(value);

    NVega::TCommandRequest request;
    request.Id = MessageId++;
    request.Code = NVega::ECommandCode::SET_PARAM;
    request.Argument.Set(parameter);

    auto handler = NDrive::NProtocol::Handle(Client, MakeAtomicShared<TSendCommandHandler>(request), timeout);
    Y_ENSURE(handler->GetResult() == NVega::TCommandResponse::PROCESSED, "cannot SetParameter " << id << ": " << handler->GetResult());
}

auto SetParameterBuffer = &NDrive::TTelematicsProxy::SetParameter<TBuffer>;
auto SetParameterString = &NDrive::TTelematicsProxy::SetParameter<TString>;
auto SetPatameterInteger = &NDrive::TTelematicsProxy::SetParameter<ui64>;
