#include <drive/telematics/client/car_emulator/car_emulator.h>
#include <drive/telematics/client/library/client.h>
#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/client/library/proxy.h>

#include <drive/telematics/api/client.h>
#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/api/server/client.h>
#include <drive/telematics/protocol/mio.h>
#include <drive/telematics/protocol/nonce.h>
#include <drive/telematics/protocol/settings.h>
#include <drive/telematics/protocol/vega.h>
#include <drive/telematics/server/data/clickhouse/client.h>
#include <drive/telematics/server/location/names.h>
#include <drive/telematics/server/pusher/pusher.h>
#include <drive/telematics/server/saas/pusher.h>
#include <drive/telematics/server/sensors/cache.h>
#include <drive/telematics/server/tasks/lite.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/timeline/requester.h>

#include <library/cpp/clickhouse/client/client.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/neh/asio/executor.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/library/executor/executor.h>
#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/storage/postgres/postgres_storage.h>
#include <rtline/util/algorithm/container.h>

#include <util/digest/fnv.h>
#include <util/network/socket.h>
#include <util/stream/buffer.h>
#include <util/stream/file.h>
#include <util/string/hex.h>
#include <util/system/env.h>
#include <util/system/hostname.h>

#include <fstream>
#include <random>

void AddApiOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption('c', "config", "API configuration file").RequiredArgument("PATH");
    options.AddLongOption("timeout", "API timeout in seconds").RequiredArgument("SECONDS").DefaultValue(ToString(10));
}

void AddApi2Options(NLastGetopt::TOpts& options) {
    options.AddLongOption('s', "shard", "server shards").RequiredArgument("HOST:PORT");
    options.AddLongOption("tvm-destination-client-id", "TVM destination client ID").RequiredArgument("UINT").DefaultValue(ToString(2009819));
    options.AddLongOption("tvm-self-client-id", "TVM self client ID").RequiredArgument("UINT").DefaultValue(ToString(2000615));
    options.AddLongOption("tvm-secret", "TVM secret").RequiredArgument("STRING");
}

void AddCommonOptions(NLastGetopt::TOpts& options) {
    options.AddHelpOption();
    options.AddLongOption('h', "host", "Telematics server host").RequiredArgument("SERVER").DefaultValue("mt.carsharing.yandex.net");
    options.AddLongOption('p', "port", "Telematics server port").RequiredArgument("PORT").DefaultValue(ToString(24690));
    options.SetFreeArgsNum(0);
}

void AddCommandOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption('r', "repeat", "repeat command X times").RequiredArgument("X").Optional();
    options.SetFreeArgsNum(1);
    options.SetFreeArgDefaultTitle("COMMAND", "command to execute");
}

void AddDeviceOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption('i', "IMEI", "Telematics client IMEI").RequiredArgument("IMEI").Optional();
    options.AddLongOption("id", "Yandex.Drive device ID").RequiredArgument("UUID").Optional();
    options.AddLongOption("password", "IMEI password").RequiredArgument("PASSWORD").DefaultValue("");
    options.AddLongOption("drive-endpoint", "Yandex.Drive admin endpoint").RequiredArgument("HOST").DefaultValue("prestable.carsharing.yandex.net");
    options.AddLongOption("drive-token", "Yandex.Drive admin token").RequiredArgument("TOKEN").Optional();
}

void AddGetFileOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption("name", "File name: LOG|FIRMWARE|CANPRO|SETTINGS|BASE").RequiredArgument("NAME").Required();
    options.AddLongOption("offset", "File offset").RequiredArgument("INT64").DefaultValue("0");
    options.AddLongOption("output", "Output file").RequiredArgument("FILE").Optional();
}

void AddSensorApiOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption("sensor-host", "Sensor API host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps-prestable.yandex.net");
    options.AddLongOption("sensor-port", "Sensor API port").RequiredArgument("PORT").DefaultValue(17000);
    options.AddLongOption("sensor-service", "Sensor API token").RequiredArgument("SERVICE").DefaultValue("drive_cache");
    options.AddLongOption("balancer-timeout", "L7 balancer timeout").RequiredArgument("DURATION").Optional();
    options.AddLongOption("timeout", "Request timeout").RequiredArgument("DURATION").Optional();
}

void AddSetServerOptions(NLastGetopt::TOpts& options) {
    options.AddLongOption("index", "Server index: 0-3").RequiredArgument("IDX").DefaultValue(ToString(2));
    options.AddLongOption("server", "Server host:port").RequiredArgument("HOST:PORT").DefaultValue("213.180.204.74:17578");
    options.AddLongOption("protocol", "Server protocol").RequiredArgument("PROTOCOL").DefaultValue(ToString(NDrive::NVega::TServerSettingsParameter::VEGA));
    options.AddLongOption("pin", "Server pin").RequiredArgument("STRING").DefaultValue("");
    options.AddLongOption("force", "Override existing server").Optional().NoArgument();
}

TString GetIMEI(const NLastGetopt::TOptsParseResult& arguments) {
    if (arguments.Has("IMEI")) {
        return arguments.Get("IMEI");
    }
    if (arguments.Has("id")) {
        const TString& endpoint = arguments.Get("drive-endpoint");
        const TString& token = arguments.Get("drive-token");
        const TString& objectId = arguments.Get("id");
        NDrive::TSessionRequester requester(endpoint, "", token);
        NJson::TJsonValue response = requester.Request("/api/staff/car/info?car_id=" + objectId);
        return response["imei"].GetStringSafe();
    }
    ythrow yexception() << "either IMEI or device ID should be specified";
}

TVector<TString> GetPasswords(const NLastGetopt::TOptsParseResult& arguments) {
    TVector<TString> result;
    if (const NLastGetopt::TOptParseResult* parsed = arguments.FindLongOptParseResult("password")) {
        for (auto&& value : parsed->Values()) {
            result.push_back(value);
        }
    }
    if (result.empty()) {
        const TString& endpoint = arguments.Get("drive-endpoint");
        const TString& token = arguments.Get("drive-token");
        const TString& objectId = arguments.Get("id");
        NDrive::TSessionRequester requester(endpoint, "", token);
        NJson::TJsonValue response = requester.Request("/api/staff/car/telematics/password?car_id=" + objectId);
        TString password = response["password"].GetString();
        if (password) {
            result.push_back(password);
        }
    }
    result.push_back("1234");
    return result;
}

NDrive::TTelematicsApi GetApi(const NLastGetopt::TOptsParseResult& arguments) {
    const TString& configFile = arguments.Get("config");
    TAnyYandexConfig config;
    Y_ENSURE(config.Parse(configFile), "cannot parse config from " << configFile);
    TTaskExecutorConfig taskExecutorConfig;
    Y_ENSURE(taskExecutorConfig.Init(config.GetRootSection()), "cannot initialize TaskExecutorConfig from " << configFile);
    NDrive::TTelematicsApi result(taskExecutorConfig);
    return result;
}

NDrive::TTelematicsClient GetApi2(const NLastGetopt::TOptsParseResult& arguments) {
    const auto destination = FromString<NTvmAuth::TTvmId>(arguments.Get("tvm-destination-client-id"));
    const auto source = FromString<NTvmAuth::TTvmId>(arguments.Get("tvm-self-client-id"));
    const auto secret = arguments.Get("tvm-secret");

    NTvmAuth::NTvmApi::TClientSettings::TDstVector destinations;
    destinations.emplace_back(destination);
    NTvmAuth::NTvmApi::TClientSettings settings;
    settings.SetSelfTvmId(source);
    settings.EnableServiceTicketsFetchOptions(secret, std::move(destinations));
    auto tvm = MakeAtomicShared<NTvmAuth::TTvmClient>(settings, NTvmAuth::TDevNullLogger::IAmBrave());

    auto shards = arguments.FindLongOptParseResult("shard");
    Y_ENSURE(shards);

    auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(MakeVector<TString>(shards->Values()));
    NDrive::TTelematicsClient::TOptions clientOptions;
    clientOptions.DestinationClientId = destination;
    return {
        sd,
        clientOptions,
        tvm
    };
}

struct TSensorApiCtx {
    THolder<NRTLine::TNehSearchClient> SearchClient;
};

NDrive::TSensorApi GetSensorApi(const NLastGetopt::TOptsParseResult& res, TSensorApiCtx& ctx) {
    NRTLine::TNehSearchClient::TEndpoint endpoint(res.Get("sensor-host"), FromString(res.Get("sensor-port")));
    endpoint.SetService(res.Get("sensor-service"));
    if (res.Has("balancer-timeout")) {
        auto balancerTimeout = FromString<TDuration>(res.Get("balancer-timeout"));
        NRTLine::TRequestBuilder::TBalancerOptions balancerOptions;
        balancerOptions.BalancerTimeoutTable = { balancerTimeout };
        endpoint.SetBalancerOptions(balancerOptions);
    }
    ctx.SearchClient = MakeHolder<NRTLine::TNehSearchClient>(endpoint);
    NDrive::TSensorApi::TOptions apiOptions;
    if (res.Has("timeout")) {
        auto timeout = FromString<TDuration>(res.Get("timeout"));
        apiOptions.Timeout = timeout;
        apiOptions.HeavyTimeout = timeout;
    }
    NDrive::TSensorApi api("telematics_client", *ctx.SearchClient, apiOptions);
    return api;
}

void Wait(const NDrive::TTelematicsApi& api, NDrive::TTelematicsApi::THandler& handle, TStringBuf description, TDuration timeout) {
    if (handle) {
        Y_ENSURE(api.Wait(handle, timeout), description << '\t' << handle.Id << '\t' << "timeout");
        Y_ENSURE(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success, description << '\t' << handle.Id << '\t' << api.GetStatus(handle));
    }
}

struct TDeviceInfo {
public:
    TString IMEI;
    TMaybe<TGeoCoord> Position;
    TMaybe<ui8> FuelLevel;

public:
    TDeviceInfo() = default;
    TDeviceInfo(const char* imei)
        : IMEI(imei)
    {
    }
    TDeviceInfo(const TString& imei)
        : IMEI(imei)
    {
    }

    static TDeviceInfo Parse(TStringBuf imei) {
        TDeviceInfo result;
        Y_ENSURE(!imei.empty());
        if (imei[0] == '{') {
            NJson::TJsonValue description;
            NJson::ReadJsonFastTree(imei, &description, /*throwOnError=*/true);
            Y_ENSURE(description.Has("imei"));
            result.IMEI = ToString(FromString<ui64>(description["imei"].GetStringRobust()));
            if (description.Has("position")) {
                result.Position = {
                    description["position"]["lon"].GetDoubleRobust(),
                    description["position"]["lat"].GetDoubleRobust()
                };
            }
            if (description.Has("fuel_level")) {
                result.FuelLevel = description["fuel_level"].GetUIntegerRobust();
            }
        } else {
            result.IMEI = ToString(FromString<ui64>(imei));
        }
        return result;
    }
};

int main_device_emulator(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    options.AddLongOption('i', "imei", "IMEI to emulate").RequiredArgument("IMEI").Optional();
    options.AddLongOption("saas-host", "GeoSaaS host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps-prestable.yandex.net");
    options.AddLongOption("saas-port", "GeoSaaS port").RequiredArgument("PORT").DefaultValue("17000");
    options.AddLongOption("saas-service", "GeoSaaS service").RequiredArgument("SERVICE").DefaultValue("drive_router");
    options.AddLongOption("sensor-host", "Sensor API host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps.yandex.net");
    options.AddLongOption("sensor-port", "Sensor API port").RequiredArgument("PORT").DefaultValue("17000");
    options.AddLongOption("sensor-service", "Sensor API service").RequiredArgument("SERVICE").DefaultValue("drive_cache");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));

    TVector<TDeviceInfo> devices;
    if (res.Has("imei")) {
        auto opt = res.FindLongOptParseResult("imei");
        Y_ENSURE(opt);
        devices.insert(devices.end(), opt->Values().cbegin(), opt->Values().cend());
    }
    if (devices.empty()) {
        throw yexception() << "no devices to emulate";
    }

    NRTLine::TNehSearchClient sensorsClient(res.Get("sensor-service"), res.Get("sensor-host"), FromString<ui16>(res.Get("sensor-port")));
    NDrive::TSensorApi sensorsApi("telematics_client", sensorsClient);
    TAtomicSharedPtr<NGraph::TRouter> router = MakeAtomicShared<NGraph::TRouter>(res.Get("saas-host"), FromString<ui16>(res.Get("saas-port")), res.Get("saas-service"));
    TAtomicSharedPtr<NAsio::TExecutorsPool> executorsPool = MakeAtomicShared<NAsio::TExecutorsPool>(static_cast<size_t>(4));
    TVector<TAtomicSharedPtr<NDrive::TCarEmulator>> emulators;
    for (const auto& device : devices) {
        const TString& imei = device.IMEI;
        auto position = device.Position;
        if (!position) {
            auto asyncLocation = sensorsApi.GetLocation(imei);
            if (!asyncLocation.Wait(TDuration::Seconds(4))) {
                ERROR_LOG << imei << ": could not wait location request" << Endl;
            }
            if (asyncLocation.HasValue()) {
                auto location = asyncLocation.ExtractValueSync();
                if (location) {
                    position = {
                        location->Longitude,
                        location->Latitude
                    };
                }
            }
            if (asyncLocation.HasException()) {
                ERROR_LOG << imei << ": an exception occurred during location request - " << NThreading::GetExceptionMessage(asyncLocation) << Endl;
            }
        }
        if (!position) {
            position = NDrive::CreateLocation({});
        }

        Y_ENSURE(imei);
        INFO_LOG << "Creating emulator with imei " << imei << " and position " << position->ToString() << Endl;
        auto emulator = MakeAtomicShared<NDrive::TCarEmulator>(imei, host, port, router, executorsPool, *position);
        if (device.FuelLevel) {
            emulator->GetContext().SetFuelPercent(*device.FuelLevel);
        }
        emulators.push_back(emulator);
    }
    INFO_LOG << "Sleeping for ever" << Endl;
    Sleep(TDuration::Max());
    return EXIT_SUCCESS;
}

namespace {
    struct TPosition: public TGeoCoord {
    };

    struct TSettingsState {
        TString IMEI;
        TMaybe<TPosition> Position;
        TMaybe<ui8> FuelLevel;
    };

    struct TDeviceState {
        TString IMEI;
        TMaybe<NDrive::TLocation> Location;
        TMaybe<bool> EngineOn;
        TMaybe<ui8> FuelLevel;
        TMaybe<float> FuelDistance;
        TMaybe<float> Mileage;
        TMaybe<float> Speed;
        NDrive::TSessionRequester::TTags Tags;
    };
    using TDeviceStates = TVector<TDeviceState>;

    struct TEmulatedDevice {
        TDeviceState State;

        TAtomicSharedPtr<NDrive::TCarEmulator> Emulator;
        TInstant Heartbeat;
        TInstant ResetTimestamp;
    };
    using TEmulatedDevices = TMap<TString, TEmulatedDevice>;

    class TDeviceRequester: public NDrive::TBaseRequester {
    public:
        TDeviceRequester(const TString& endpoint, const TString& token)
            : NDrive::TBaseRequester(
                TString{GetHostAndPort(endpoint)},
                TString{GetPathAndQuery(endpoint).After('?').Before('#')},
                "Authorization: OAuth " + token
            )
        {
        }

        TDeviceStates GetDevices() const {
            auto response = RequestJson("/api/staff/car/list?traits=ReportVIN,ReportIMEI,ReportLocationCourse,ReportLocationDetails,ReportTagDetails&sensors=fuel_level,fuel_distance,engine_on,mileage,speed");
            return NJson::FromJson<TDeviceStates>(response["cars"]);
        }
    };
}

int main_device_emulator2(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    options.AddLongOption("endpoint", "Backend endpoint").RequiredArgument("URL").Required();
    options.AddLongOption("token", "Backend token").RequiredArgument("TOKEN");
    options.AddLongOption("threads", "Internal threads count").RequiredArgument("UINT").DefaultValue(16);
    options.AddLongOption("limit", "Maximum count of emulators").RequiredArgument("UINT").DefaultValue(64);
    options.AddLongOption("lock-ttl", "Lock time to live").RequiredArgument("DURATION").DefaultValue("12h");
    options.AddLongOption("shard-limit", "Maximum count of locked shards").RequiredArgument("UINT").DefaultValue(32);
    options.AddLongOption("shards", "Shards count for emulators").RequiredArgument("UINT").DefaultValue(32);
    options.AddLongOption("router-host", "Router host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps.yandex.net");
    options.AddLongOption("router-port", "Router port").RequiredArgument("PORT").DefaultValue("17000");
    options.AddLongOption("router-service", "Router service").RequiredArgument("SERVICE").DefaultValue("drive_router");
    options.AddLongOption("settings-db", "PostgreSQL database for settings").RequiredArgument("CONNSTRING");
    options.AddLongOption("settings-table", "Table in the database for settings").RequiredArgument("TABLENAME");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));

    const auto endpoints = res.FindLongOptParseResult("endpoint");
    const TString& token = res.Has("token") ? res.Get("token") : GetEnv("DRIVE_TOKEN");
    Y_ENSURE(token, "either --token argument or DRIVE_TOKEN environment variable should be specified");

    auto executorsPool = MakeAtomicShared<NAsio::TExecutorsPool>(FromString<size_t>(res.Get("threads")));
    auto router = MakeAtomicShared<NGraph::TRouter>(res.Get("router-host"), FromString<ui16>(res.Get("router-port")), res.Get("router-service"));
    auto limit = FromString<size_t>(res.Get("limit"));
    auto shards = FromString<size_t>(res.Get("shards"));
    auto shardLimit = FromString<size_t>(res.Get("shard-limit"));
    auto deadTimeout = TDuration::Minutes(5);
    auto lockTtl = FromString<TDuration>(res.Get("lock-ttl"));

    auto settings = NRTLine::IVersionedStorage::TPtr();
    if (res.Has("settings-db")) {
        NRTLine::TPostgresStorageOptions config;
        NRTLine::IVersionedStorage::TOptions optionsStub;
        config.SetHardLimitPoolSize(shardLimit + 1);
        config.SetSimpleConnectionString(res.Get("settings-db"));
        config.SetTableName(res.Get("settings-table"));
        settings = NRTLine::TPostgresStorage::Create(optionsStub, config);
    }

    auto requesters = TVector<TDeviceRequester>();
    for (auto&& endpoint : Yensured(endpoints)->Values()) {
        requesters.emplace_back(TString{endpoint}, token);
    }

    std::atomic_bool active = true;
    TEmulatedDevices emulatedDevices;
    TMap<size_t, NRTLine::TAbstractLock::TPtr> acquiredLocks;
    auto isLocked = [&acquiredLocks, lockTtl, settings, shards, shardLimit] (TStringBuf imei) {
        if (!settings) {
            return true;
        }
        auto now = Now();
        auto hash = FnvHash<ui32>(imei);
        auto shard = hash % shards;
        auto lock = acquiredLocks[shard];
        if (lock && lock->IsLocked() && lock->GetTimestamp() + lockTtl > now) {
            return true;
        } else {
            acquiredLocks.erase(shard);
            lock = nullptr;
        }
        if (acquiredLocks.size() >= shardLimit) {
            return false;
        }
        auto key = "emulators:shard:" + ToString(shard);
        lock = settings->WriteLockNode(key, TDuration::MilliSeconds(10));
        if (lock && lock->IsLocked()) {
            INFO_LOG << "acquired lock " << key << Endl;
            acquiredLocks[shard] = lock;
            return true;
        } else {
            return false;
        }
    };
    while (active) try {
        INFO_LOG << "updating emulators" << Endl;
        auto devices = TDeviceStates();
        for (auto&& requester : requesters) try {
            INFO_LOG << "fetching devices from " << requester.GetDescription() << Endl;
            auto d = requester.GetDevices();
            INFO_LOG << "got " << d.size() << " devices from " << requester.GetDescription() << Endl;
            if (devices.empty()) {
                devices = std::move(d);
            } else {
                devices.insert(devices.end(), d.begin(), d.end());
            }
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot fetch devices from " << requester.GetDescription() << ": " << FormatExc(e) << Endl;
        }
        auto imeis = TSet<TString>();
        for (auto&& device : devices) {
            if (device.IMEI) {
                imeis.insert(device.IMEI);
            }
        }
        auto hosthash = std::hash<TString>()(HostName());
        std::shuffle(devices.begin(), devices.end(), std::default_random_engine(hosthash));

        auto settingsStates = TMap<TString, TSettingsState>();
        if (settings) {
            auto settingsStateString = TString();
            auto settingsStateArray = TVector<TSettingsState>();
            if (!settings->GetValue("emulator:states", settingsStateString)) {
                WARNING_LOG << "cannot fetch emulator states" << Endl;
            }
            if (!NJson::ParseField(NJson::ToJson(NJson::JsonString(settingsStateString)), settingsStateArray)) {
                ERROR_LOG << "cannot parse emulator states: " << settingsStateString << Endl;
            }
            for (auto&& state : settingsStateArray) {
                settingsStates.emplace(state.IMEI, state);
            }
            INFO_LOG << "fetched settings states for " << settingsStates.size() << " devices" << Endl;
        }

        TSet<TString> dead;
        for (auto&& [imei, emulatedDevice] : emulatedDevices) {
            auto now = Now();
            if (imeis.contains(imei)) {
                emulatedDevice.Heartbeat = now;
            } else if (emulatedDevice.Heartbeat + deadTimeout < now) {
                NOTICE_LOG << imei << ": destroying emulator" << Endl;
                dead.insert(imei);
            }
            if (!isLocked(imei)) {
                WARNING_LOG << imei << ": lost lock" << Endl;
                dead.insert(imei);
            }
            if (!emulatedDevice.Emulator || !emulatedDevice.Emulator->GetClient()->Alive()) {
                WARNING_LOG << imei << ": emulator is dead" << Endl;
                dead.insert(imei);
            }
        }
        for (auto&& i : dead) {
            emulatedDevices.erase(i);
        }
        TSet<ui32> failedShards;
        for (auto&& device : devices) {
            const auto& imei = device.IMEI;
            if (!imei) {
                continue;
            }
            if (emulatedDevices.contains(imei)) {
                DEBUG_LOG << imei << ": alright" << Endl;
                continue;
            }
            auto shard = FnvHash<ui32>(imei) % shards;
            if (!failedShards.contains(shard) && isLocked(imei)) {
                INFO_LOG << imei << ": successfully acquired lock" << Endl;
            } else {
                failedShards.insert(shard);
                WARNING_LOG << imei << ": cannot acquire lock " << shard << Endl;
                continue;
            }

            auto settingsState = settingsStates.FindPtr(imei);
            NOTICE_LOG << imei << ": creating emulator" << Endl;

            auto context = MakeHolder<NDrive::TTelematicsClientContext>();
            TMaybe<TGeoCoord> coordinate;
            if (!coordinate && device.Location) {
                coordinate = device.Location->GetCoord();
            }
            if (!coordinate && settingsState && settingsState->Position) {
                coordinate = settingsState->Position;
            }
            if (!coordinate) {
                TSet<TString> tags;
                for (auto&& tag : device.Tags) {
                    tags.insert(tag.Name);
                }
                coordinate = NDrive::CreateLocation(tags);
            }
            context->SetCurrentPosition(*coordinate);
            if (device.EngineOn) {
                context->TrySetEngineStarted(*device.EngineOn);
            }
            if (device.FuelLevel) {
                context->SetFuelPercent(*device.FuelLevel);
            } else if (settingsState && settingsState->FuelLevel) {
                context->SetFuelPercent(*settingsState->FuelLevel);
            }
            if (device.Mileage) {
                context->SetOdometerKm(*device.Mileage);
            }
            if (device.Speed) {
                context->TrySetSpeed(*device.Speed);
            }

            auto now = Now();
            TEmulatedDevice emulatedDevice;
            emulatedDevice.State = device;
            emulatedDevice.Emulator = MakeAtomicShared<NDrive::TCarEmulator>(
                imei,
                host,
                port,
                std::move(context),
                router,
                executorsPool,
                false
            );
            emulatedDevice.Heartbeat = now;
            emulatedDevice.ResetTimestamp = TInstant::Days(now.Days() + 1);
            emulatedDevices.emplace(imei, std::move(emulatedDevice));
            if (emulatedDevices.size() >= limit) {
                INFO_LOG << "limit reached" << Endl;
                break;
            }
        }
        INFO_LOG << "updated emulators" << Endl;
        auto now = Now();
        for (auto&& [imei, emulatedDevice] : emulatedDevices) {
            auto settingsState = settingsStates.FindPtr(imei);
            if (!settingsState) {
                continue;
            }
            if (now > emulatedDevice.ResetTimestamp) {
                INFO_LOG << imei << ": resetting " << emulatedDevice.ResetTimestamp << Endl;
                if (settingsState->Position) {
                    emulatedDevice.Emulator->GetContext().SetCurrentPosition(*settingsState->Position);
                }
                if (settingsState->FuelLevel) {
                    emulatedDevice.Emulator->GetContext().SetFuelPercent(*settingsState->FuelLevel);
                }
                emulatedDevice.ResetTimestamp = TInstant::Days(emulatedDevice.ResetTimestamp.Days() + 1);
                INFO_LOG << imei << ": reset" << Endl;
            } else {
                INFO_LOG << imei << ": resetting at " << emulatedDevice.ResetTimestamp << Endl;
            }
        }
        Sleep(TDuration::Seconds(10));
    } catch (const std::exception& e) {
        ERROR_LOG << "an exception occurred: " << FormatExc(e) << Endl;
        Sleep(TDuration::Seconds(10));
    }

    return EXIT_SUCCESS;
}

namespace {
    class TTelematicsCallback : public IDistributedTask {
    public:
        TTelematicsCallback(const NDrive::TSendCommandDistributedData& data)
            : IDistributedTask({ "TelematicsCallback", "callback-" + data.GetIdentifier() })
        {
        }
        ~TTelematicsCallback() {
            CHECK_WITH_LOG(Ev.WaitT(TDuration::Zero()));
        }
        void Wait() {
            Ev.WaitI();
        }

    protected:
        bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            Ev.Signal();
            return true;
        }
        bool DoOnTimeout(IDistributedTask::TPtr /*self*/) noexcept override {
            Ev.Signal();
            return true;
        }
        bool DoOnIncorrectData(IDistributedTask::TPtr /*self*/) noexcept override {
            Ev.Signal();
            return true;
        }

    private:
        TManualEvent Ev;
    };
}

int main_api_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApiOptions(options);
    AddDeviceOptions(options);
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TDuration timeout = TDuration::Parse(res.Get("timeout"));
    const TString& imei = GetIMEI(res);
    const TString cmd = res.GetFreeArgs().at(0);
    const auto command = NDrive::ParseCommand(NJson::ToJson(NJson::JsonString(cmd)));
    const auto repeat = res.Has("repeat") ? FromString<ui64>(res.Get("repeat")) : 1;

    const NDrive::TTelematicsApi api = GetApi(res);

    TDuration totalDuration;
    for (ui64 i = 0; i < repeat; ++i) {
        auto commandData = api.BuildCommand(imei, command->Code, timeout, timeout, timeout);
        commandData.SetGenericArgument(command->Argument);
        auto callback = MakeAtomicShared<TTelematicsCallback>(commandData);

        TInstant start = Now();
        auto handler = api.Command(commandData, callback);
        callback->Wait();
        TInstant finish = Now();

        TDuration duration = finish - start;
        if (repeat > 1) {
            Cout << imei << '\t' << i << '\t' << duration.MicroSeconds() << '\t';
        }
        if (handler) {
            Cout << imei << '\t' << api.GetStatus(handler) << '\t' << api.GetMessage(handler) << '\t' << handler.GetId() << '\t' << handler.GetError() << Endl;
        } else {
            Cout << imei << '\t' << "bad handle" << Endl;
        }
        totalDuration += duration;
    }
    if (repeat > 1) {
        Cout << imei << "\ttotal\t" << totalDuration.MicroSeconds() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_api_get_file(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApiOptions(options);
    AddDeviceOptions(options);
    AddGetFileOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const NDrive::TTelematicsApi api = GetApi(res);
    const TDuration timeout = TDuration::Parse(res.Get("timeout"));
    const TString& imei = GetIMEI(res);
    const TString& name = res.Get("name");

    auto handle = api.Download(imei, name);
    Wait(api, handle, TStringBuf("GetFile"), timeout);
    const auto& data = api.GetValue<TBuffer>(handle);

    if (res.Has("output")) {
        const TString& outputFile = res.Get("output");
        TOFStream output(outputFile);
        output.Write(data.Data(), data.Size());
    } else {
        Cout << TStringBuf(data.Data(), data.Size());
    }

    return EXIT_SUCCESS;
}

int main_api_get(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("file", main_api_get_file, "Get a file");
    return modChooser.Run(argc, argv);
}

int main_api_set_server(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApiOptions(options);
    AddDeviceOptions(options);
    AddSetServerOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const NDrive::TTelematicsApi api = GetApi(res);
    const TDuration timeout = TDuration::Parse(res.Get("timeout"));
    const TString& imei = GetIMEI(res);

    const auto index = FromString<ui16>(res.Get("index"));
    const TString& server = res.Get("server");
    const TString& pin = res.Get("pin");
    const auto protocol = FromString<NDrive::NVega::TServerSettingsParameter::EProtocol>(res.Get("protocol"));
    const bool force = res.Has("force");
    Y_ENSURE(GetHostAndPort(server) == server, "bad server format: " << server);

    auto currentServerHandle = api.GetParameter(imei, VEGA_SETTING_SERVER_ADDR, index);
    Wait(api, currentServerHandle, TStringBuf("GetCurrentServer"), timeout);

    NDrive::NVega::TServerSettingsParameter serverSettings;
    if (const auto& currentServer = api.GetValue<TBuffer>(currentServerHandle)) {
        TBufferInput input(currentServer);
        serverSettings.Load(&input);
    }
    DEBUG_LOG << imei << " current server: " << serverSettings.HostPort.Get() << Endl;
    if (
        serverSettings.Protocol != serverSettings.DISABLED &&
        serverSettings.HostPort.Get() != server &&
        serverSettings.Protocol != protocol &&
        !force
    ) {
        WARNING_LOG << imei << " server " << index << " is already set to " << serverSettings.HostPort.Get() << Endl;
    } else {
        serverSettings.HostPort.Set(server);
        serverSettings.InteractionPeriod = 0;
        serverSettings.Protocol = protocol;

        TBuffer value;
        TBufferOutput output(value);
        serverSettings.Save(&output);
        auto newServerHandle = api.SetParameter(imei, VEGA_SETTING_SERVER_ADDR, index, std::move(value));
        auto usePinHandle = NDrive::TTelematicsApi::THandler();
        auto pinHandle = NDrive::TTelematicsApi::THandler();
        if (pin) {
            TBuffer b;
            TBufferOutput bo(b);
            NDrive::NVega::THardPasswordParameter p;
            p.Value.Set(pin);
            p.Save(&bo);

            usePinHandle = api.SetParameter(imei, VEGA_SETTING_USE_SERVER_PIN, index, ui64(2));
            pinHandle = api.SetParameter(imei, VEGA_SETTING_SERVER_PIN, index, std::move(b));
        } else {
            usePinHandle = api.SetParameter(imei, VEGA_SETTING_USE_SERVER_PIN, index, ui64(1));
        }
        Wait(api, newServerHandle, TStringBuf("SetServer"), timeout);
        Wait(api, usePinHandle, TStringBuf("SetUsePin"), timeout);
        Wait(api, pinHandle, TStringBuf("SetPin"), timeout);

        INFO_LOG << imei << " successfully set server " << index << " to " << server << Endl;
    }

    return EXIT_SUCCESS;
}

int main_api_set(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("server", main_api_set_server, "Set monitoring server");
    return modChooser.Run(argc, argv);
}

int main_api_DRIVETELEMATICS_79(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApiOptions(options);
    AddDeviceOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const NDrive::TTelematicsApi api = GetApi(res);
    const TDuration timeout = TDuration::Parse(res.Get("timeout"));
    const TString& imei = GetIMEI(res);

    auto warmup = api.Command(imei, NDrive::NVega::ECommandCode::YADRIVE_WARMING);
    api.Wait(warmup);
    Cout << imei << ":warming:" << warmup.Id << ":" << api.GetStatus(warmup) << ' ' << api.GetMessage(warmup) << Endl;

    const ui8 canIndex = 0;
    const auto makeCanRequest = [&](ui32 id, const TVector<unsigned char>& data, TDuration duration = TDuration::Zero(), TDuration interval = TDuration::Zero()) {
        auto buffer = TBuffer(reinterpret_cast<const char*>(data.data()), data.size());
        auto canRequest = api.BuildCanRequest(imei, id, canIndex, std::move(buffer));
        canRequest.SetRepetition(duration, interval);
        auto handler = api.CanRequest(canRequest);
        Wait(api, handler, "CanRequest", timeout);
    };

    const ui32 canId = 0x7A0;
    makeCanRequest(canId, { 0x02, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00 });
    makeCanRequest(canId, { 0x02, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, TDuration::Minutes(3), TDuration::MilliSeconds(500));
    makeCanRequest(canId, { 0x02, 0x22, 0xB0, 0x01, 0x00, 0x00, 0x00, 0x00 });
    for (size_t i = 0; i < 3; ++i) {
        makeCanRequest(canId, { 0x05, 0x2E, 0x12, 0xA0, 0x0A, 0x00, 0x00, 0x00 });
    }
    makeCanRequest(canId, { 0x02, 0x22, 0xB0, 0x01, 0x00, 0x00, 0x00, 0x00 });
    for (size_t i = 0; i < 3; ++i) {
        makeCanRequest(canId, { 0x05, 0x2E, 0x12, 0xA0, 0x09, 0x00, 0x00, 0x00 });
    }
    makeCanRequest(canId, { 0x02, 0x22, 0xB0, 0x01, 0x00, 0x00, 0x00, 0x00 });
    for (size_t i = 0; i < 3; ++i) {
        makeCanRequest(canId, { 0x05, 0x2E, 0x12, 0xA0, 0x02, 0x00, 0x00, 0x00 });
    }

    return EXIT_SUCCESS;
}

int main_api_DRIVEBACK_2494(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApiOptions(options);
    AddDeviceOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const NDrive::TTelematicsApi api = GetApi(res);
    const TDuration timeout = TDuration::Parse(res.Get("timeout"));
    const TString& imei = GetIMEI(res);

    auto warmup = api.Command(imei, NDrive::NVega::ECommandCode::YADRIVE_WARMING);
    api.Wait(warmup);
    Cout << imei << ":warming:" << warmup.Id << ":" << api.GetStatus(warmup) << ' ' << api.GetMessage(warmup) << Endl;

    const ui32 responseCanIndex = 0x77F;

    NDrive::NVega::TCommandRequest::TObdForwardConfig config;
    for (size_t i = 0; i < 3; ++i) {
        auto& settings = config.Cans.emplace_back();
        settings.Data.is_enable = 1;
        settings.Data.id_type = 1;
        settings.Data.value = responseCanIndex;
        settings.Data.mask = 0x7FF;
    }
    config.Duration = 10;

    auto commandData = api.BuildCommand(imei, NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG, timeout, timeout, timeout);
    commandData.SetGenericArgumentFrom(std::move(config));
    auto enableCan = api.Command(commandData);
    Wait(api, enableCan, "enableCan", timeout);

    const ui8 canIndex = 1;
    const auto makeCanRequest = [&](ui32 id, const TVector<unsigned char>& data, TDuration duration = TDuration::Zero(), TDuration interval = TDuration::Zero()) {
        auto buffer = TBuffer(reinterpret_cast<const char*>(data.data()), data.size());
        auto canRequest = api.BuildCanRequest(imei, id, canIndex, std::move(buffer));
        canRequest.SetRepetition(duration, interval);
        auto handler = api.CanRequest(canRequest);
        Wait(api, handler, "CanRequest", timeout);
    };

    const ui32 requestCanId = 0x715;
    makeCanRequest(requestCanId, { 0x03, 0x22, 0x23, 0x64, 0x00, 0x00, 0x00, 0x00 });

    return EXIT_SUCCESS;
}

int main_api(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("command", main_api_command, "Set a command to a device");
    modChooser.AddMode("get", main_api_get, "Get a parameter of a device");
    modChooser.AddMode("set", main_api_set, "Set a parameter of a device");
    modChooser.AddMode("DRIVETELEMATICS-79", main_api_DRIVETELEMATICS_79, "Perform CAN requests");
    modChooser.AddMode("DRIVEBACK-2494", main_api_DRIVEBACK_2494, "Perform CAN requests");
    return modChooser.Run(argc, argv);
}

int main_api2_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApi2Options(options);
    AddDeviceOptions(options);
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& imei = GetIMEI(res);
    const TString cmd = res.GetFreeArgs().at(0);
    const auto command = NDrive::ParseCommand(NJson::ToJson(NJson::JsonString(cmd)));
    const auto repeat = res.Has("repeat") ? FromString<ui64>(res.Get("repeat")) : 1;
    NDrive::TTelematicsClient client = GetApi2(res);

    TDuration totalDuration;
    for (ui64 i = 0; i < repeat; ++i) {
        TInstant start = Now();
        auto handle = client.Command(imei, command.GetValue());
        handle.GetFuture().Wait();
        TInstant finish = Now();
        TDuration duration = finish - start;;
        if (repeat > 1) {
            Cout << imei << '\t' << i << '\t' << duration.MicroSeconds() << '\t';
        }
        if (auto response = handle.GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>()) {
            Cout << imei << '\t' << response->Status << '\t' << response->Sensor.FormatValue().GetStringRobust() << '\t' << response->Id << '\t' << response->Shard << Endl;
        } else if (auto response = handle.GetResponse<NDrive::TTelematicsClient::TCommonResponse>()) {
            Cout << imei << '\t' << response->Status << '\t' << response->Message << '\t' << response->Id << '\t' << response->Shard << Endl;
        } else if (auto response = handle.GetResponse<NDrive::TTelematicsClient::TErrorResponse>()) {
            Cout << imei << '\t' << response->Status << '\t' << response->Message << '\t' << handle.GetId() << '\t' << response->Shard << Endl;
        } else {
            Cout << imei << '\t' << handle.GetStatus() << '\t' << '\t' << handle.GetId() << '\t' << Endl;
        }
        totalDuration += duration;
    }
    if (repeat > 1) {
        Cout << imei << "\ttotal\t" << totalDuration.MicroSeconds() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_api2_download(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApi2Options(options);
    AddDeviceOptions(options);
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& imei = GetIMEI(res);
    const TString& filename = res.GetFreeArgs().at(0);

    NDrive::TTelematicsClient client = GetApi2(res);
    auto handle = client.Download(imei, filename);

    NThreading::WaitAll(handle.GetFuture()).Subscribe([] (const NThreading::TFuture<void>& w) {
        w.GetValue();
        Cout << "wait future complete" << Endl;
    }).Wait();

    auto downloadedFile = handle.GetResponse<NDrive::TTelematicsClient::TDownloadFileResponse>();
    if (!downloadedFile) {
        Cerr << "cannot download file" << Endl;
        return EXIT_FAILURE;
    }

    const auto& data = downloadedFile->Data;
    NDrive::NVega::TSettings settings;
    settings.Parse(data);

    for (const auto& setting : settings) {
        const auto& payload = setting.Payload;
        Cout << "setting result: " << setting.Id << " " << setting.SubId << " " << HexEncode(payload.data(), payload.size()) << Endl;
    }

    return EXIT_SUCCESS;
}

namespace {
    template<class T>
    void ParseSettings(std::string_view name, const NJson::TJsonValue& json, TMap<ui16, T>& storage) {
        if (!NJson::ParseField(json[name], NJson::KeyValue(storage))) {
            Cerr << "cannot parse: " << name << Endl;
        }
    }
}

int main_api2_settings(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApi2Options(options);
    AddDeviceOptions(options);
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& imei = GetIMEI(res);
    const TString file = res.GetFreeArgs().at(0);

    TString jsonString;
    TStringOutput output(jsonString);
    if (file) {
        TIFStream input(file);
        TransferData(&input, &output);
    } else {
        Cerr << "file name is empty" << Endl;
        return EXIT_FAILURE;
    }

    NDrive::TTelematicsClient client = GetApi2(res);

    auto json = NJson::ReadJsonFastTree(jsonString);

    TMap<ui16, NDrive::NVega::TLogicNotify> notifies;
    TMap<ui16, NDrive::NVega::TLogicCheck> checks;
    TMap<ui16, NDrive::NVega::TLogicCommand> commands;
    TMap<ui16, NDrive::NVega::TLogicScript> scripts;
    TMap<ui16, NDrive::NVega::TLogicTrigger> triggers;

    ParseSettings("logic_notifies", json, notifies);
    ParseSettings("logic_checks", json, checks);
    ParseSettings("logic_commands", json, commands);
    ParseSettings("logic_scripts", json, scripts);
    ParseSettings("logic_triggers", json, triggers);

    NDrive::NVega::TSettings settings;
    auto apply = [&] (const auto& storage) {
        for (auto& [subid, object] : storage) {
            NDrive::NVega::TSetting setting;
            setting.Id = object.GetId();
            setting.SubId = subid;

            auto& payload = setting.Payload;
            TBuffer data = object.Dump();

            payload.reserve(data.Size());
            payload.insert(payload.begin(), data.Begin(), data.End());

            Cout << "apply setting: " << setting.Id << " " << setting.SubId << " " << HexEncode(payload.data(), payload.size()) << Endl;
            settings.emplace_back(std::move(setting));
        }
    };

    apply(scripts);
    apply(commands);
    apply(checks);
    apply(triggers);
    apply(notifies);

    auto data = settings.Dump();
    auto handler = client.Upload(imei, "SETTINGS", std::move(data));
    auto waiter = handler.GetFuture();

    auto finalize = [handler = std::move(handler)] (const NThreading::TFuture<void>& w) {
        w.GetValue();
        Cout << "complete wait upload file " << handler.GetStatus() << Endl;
    };

    NThreading::WaitAll(waiter).Subscribe(std::move(finalize)).Wait();
    return EXIT_SUCCESS;
}

int main_api2_parse_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString cmd = res.GetFreeArgs().at(0);
    const auto command = NDrive::ParseCommand(NJson::ToJson(NJson::JsonString(cmd)));
    switch (command->Code) {
    case NDrive::NVega::ECommandCode::SET_PARAM:
    case NDrive::NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM:
    {
        const auto& argument = command->Argument.Get<NDrive::NVega::TCommandRequest::TSetParameter>();
        NJson::TJsonValue result = NJson::TMapBuilder
            ("code", ToString(command->Code))
            ("id", argument.Id)
            ("subid", argument.SubId)
            ("value", NJson::ToJson(argument.GetValue()))
        ;
        Cout << result.GetStringRobust() << Endl;
        break;
    }
    default:
        Cout << NJson::ToJson(*command) << Endl;
    }

    return EXIT_SUCCESS;
}

int main_api2_parse_external_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString cmd = res.GetFreeArgs().at(0);
    auto arg = NJson::ToJson(NJson::JsonString(cmd));
    auto request = NJson::FromJson<NDrive::NVega::TCommandRequestSigned>(arg["request"]);
    auto response = NJson::FromJson<NDrive::NVega::TCommandResponseSigned>(arg["response"]);
    Cout << "request:\t" << NJson::ToJson(request).GetStringRobust() << Endl;
    Cout << "response:\t" << NJson::ToJson(response).GetStringRobust() << Endl;
    auto requestNonce = NDrive::Decode(request.Nonce);
    if (request.Nonce == response.Nonce) {
        Cout << "nonce:\t" << requestNonce.Timestamp.Get() << '\t' << requestNonce.Hash << Endl;
    } else {
        auto responseNonce = NDrive::Decode(response.Nonce);
        Cout << "request_nonce:\t" << requestNonce.Timestamp.Get() << '\t' << requestNonce.Hash << Endl;
        Cout << "response_nonce:\t" << responseNonce.Timestamp.Get() << '\t' << responseNonce.Hash << Endl;
    }
    return EXIT_SUCCESS;
}

int main_api2_set_server(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddApi2Options(options);
    AddDeviceOptions(options);
    AddSetServerOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NDrive::TTelematicsClient client = GetApi2(res);

    const TString& imei = GetIMEI(res);
    const auto index = FromString<ui16>(res.Get("index"));
    const TString& server = res.Get("server");
    const TString& pin = res.Get("pin");
    const auto protocol = FromString<NDrive::NVega::TServerSettingsParameter::EProtocol>(res.Get("protocol"));
    const bool force = res.Has("force");

    auto currentServerHandle = client.Command(imei, NDrive::NVega::TCommand::GetParameter({VEGA_SETTING_SERVER_ADDR, index}));
    auto currentServerResponse = currentServerHandle.WaitAndEnsureSuccess().GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
    auto currentServer = std::get<TBuffer>(Yensured(currentServerResponse)->Sensor.Value);

    NDrive::NVega::TServerSettingsParameter serverSettings;
    if (const auto& value = currentServer) {
        TBufferInput input(value);
        serverSettings.Load(&input);
    }
    DEBUG_LOG << imei << " current server: " << serverSettings.HostPort.Get() << Endl;
    if (
        serverSettings.Protocol != serverSettings.DISABLED &&
        serverSettings.HostPort.Get() != server &&
        serverSettings.Protocol != protocol &&
        !force
    ) {
        Cout << imei << " server " << index << " is already set to " << serverSettings.HostPort.Get() << Endl;
    } else {
        serverSettings.HostPort.Set(server);
        serverSettings.InteractionPeriod = 0;
        serverSettings.Protocol = protocol;

        TBuffer value;
        TBufferOutput output(value);
        serverSettings.Save(&output);
        auto setServerAddr = client.Command(imei, NDrive::NVega::TCommand::SetParameter({VEGA_SETTING_SERVER_ADDR, index}, value));
        setServerAddr.WaitAndEnsureSuccess();

        if (pin) {
            TBuffer b;
            TBufferOutput bo(b);
            NDrive::NVega::THardPasswordParameter p;
            p.Value.Set(pin);
            p.Save(&bo);

            auto setServerPin = client.Command(imei, NDrive::NVega::TCommand::SetParameter({VEGA_SETTING_SERVER_PIN, index}, b));
            setServerPin.WaitAndEnsureSuccess();
            auto setUseServerPin = client.Command(imei, NDrive::NVega::TCommand::SetParameter({VEGA_SETTING_USE_SERVER_PIN, index}, ui64(2)));
            setUseServerPin.WaitAndEnsureSuccess();
        } else {
            auto setUseServerPin = client.Command(imei, NDrive::NVega::TCommand::SetParameter({VEGA_SETTING_USE_SERVER_PIN, index}, ui64(1)));
            setUseServerPin.WaitAndEnsureSuccess();
        }

        Cout << imei << " successfully set server " << index << " to " << server << Endl;
    }

    return EXIT_SUCCESS;
}

int main_api2_set(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("server", main_api2_set_server, "Set server on a device");
    return modChooser.Run(argc, argv);
}

int main_api2(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("command", main_api2_command, "Send a command to a device");
    modChooser.AddMode("parse_command", main_api2_parse_command, "Parse telematics command");
    modChooser.AddMode("parse_external_command", main_api2_parse_external_command, "Parse external command");
    modChooser.AddMode("set", main_api2_set, "Set something on a device");
    modChooser.AddMode("settings", main_api2_settings, "Parse and send settings");
    modChooser.AddMode("download", main_api2_download, "Download file");
    return modChooser.Run(argc, argv);
}

int main_list(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));

    NDrive::TTelematicsProxy proxy(host, port);
    auto imeis = proxy.List();
    for (auto&& imei : imeis) {
        Cout << imei << Endl;
    }
    return EXIT_SUCCESS;
}

int main_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    AddDeviceOptions(options);
    AddCommandOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));
    const TString& imei = GetIMEI(res);
    const auto passwords = GetPasswords(res);

    const TString cmd = res.GetFreeArgs().at(0);
    const auto command = NDrive::ParseCommand(NJson::ToJson(NJson::JsonString(cmd)));

    NDrive::TTelematicsProxy proxy(host, port);
    proxy.Connect(imei, passwords);
    proxy.Command(command->Code);
    proxy.Disconnect();
    return EXIT_SUCCESS;
}

int main_get_file(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    AddDeviceOptions(options);
    AddGetFileOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));
    const TString& imei = GetIMEI(res);
    const TString& name = res.Get("name");
    const auto offset = FromString<ssize_t>(res.Get("offset"));
    const auto passwords = GetPasswords(res);

    NDrive::TTelematicsProxy proxy(host, port);
    proxy.Connect(imei, passwords);

    TBuffer data = proxy.GetFile(name, offset);
    if (res.Has("output")) {
        const TString& outputFile = res.Get("output");
        TOFStream output(outputFile);
        output.Write(data.Data(), data.Size());
    } else {
        Cout << TStringBuf(data.Data(), data.Size());
    }

    proxy.Disconnect();
    return EXIT_SUCCESS;
}

int main_get(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("file", main_get_file, "Get a file");
    return modChooser.Run(argc, argv);
}

int main_set_apn(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    AddDeviceOptions(options);
    options.AddLongOption("index", "SIM card index").RequiredArgument("IDX").DefaultValue(ToString(0));
    options.AddLongOption("expected", "APN name to replace").RequiredArgument("NAME").Required();
    options.AddLongOption("apn-name", "APN name").RequiredArgument("NAME").Required();
    options.AddLongOption("apn-user", "APN user").RequiredArgument("USER").Required();
    options.AddLongOption("apn-password", "APN password").RequiredArgument("PASSWORD").Required();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));
    const TString& imei = GetIMEI(res);
    const auto passwords = GetPasswords(res);
    const auto index = FromString<ui16>(res.Get("index"));
    const TString& expected = res.Get("expected");
    const TString& apnName = res.Get("apn-name");
    const TString& apnUser = res.Get("apn-user");
    const TString& apnPassword = res.Get("apn-password");

    NDrive::TTelematicsProxy proxy(host, port);
    proxy.Connect(imei, passwords);

    auto existing = proxy.GetParameter(VEGA_SETTING_APN, index);

    NDrive::NVega::TAPNParameter apn;
    if (const auto& value = std::get<TBuffer>(existing)) {
        TBufferInput input(value);
        apn.Load(&input);
    }
    if (apn.APN.Get() != expected) {
        WARNING_LOG << imei << " APN " << index << " has unexpected value " << apn.APN.Get() << Endl;
    } else {
        apn.APN.Set(apnName);
        apn.User.Set(apnUser);
        apn.Password.Set(apnPassword);

        TBuffer value;
        TBufferOutput output(value);
        apn.Save(&output);
        proxy.SetParameter(VEGA_SETTING_APN, index, value);
        INFO_LOG << imei << " successfully set APN " << index << " to " << apnName << ":" << apnUser << ":" << apnPassword << Endl;
    }

    proxy.Disconnect();
    return EXIT_SUCCESS;
}

int main_set_file(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    AddDeviceOptions(options);
    options.AddLongOption("name", "File name: LOG|FIRMWARE|CANPRO|SETTINGS|BASE").RequiredArgument("NAME").Required();
    options.AddLongOption("input", "Input file").RequiredArgument("FILE").Optional();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));
    const TString& imei = GetIMEI(res);
    const TString& name = res.Get("name");
    const auto passwords = GetPasswords(res);

    NDrive::TTelematicsProxy proxy(host, port);
    proxy.Connect(imei, passwords);

    TBuffer buffer;
    TBufferOutput output(buffer);
    if (res.Has("input")) {
        const TString& inputFile = res.Get("input");
        TIFStream input(inputFile);
        TransferData(&input, &output);
    } else {
        TransferData(&Cin, &output);
    }
    proxy.SetFile(name, std::move(buffer));

    proxy.Disconnect();
    return EXIT_SUCCESS;
}

int main_set_server(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddCommonOptions(options);
    AddDeviceOptions(options);
    AddSetServerOptions(options);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("host");
    const auto port = FromString<ui16>(res.Get("port"));
    const TString& imei = GetIMEI(res);
    const auto passwords = GetPasswords(res);
    const auto index = FromString<ui16>(res.Get("index"));
    const TString& server = res.Get("server");
    const TString& pin = res.Get("pin");
    const auto protocol = FromString<NDrive::NVega::TServerSettingsParameter::EProtocol>(res.Get("protocol"));
    const bool force = res.Has("force");

    Y_ENSURE(GetHostAndPort(server) == server, "bad server format: " << server);

    NDrive::TTelematicsProxy proxy(host, port);
    proxy.Connect(imei, passwords);

    auto currentServer = proxy.GetParameter(VEGA_SETTING_SERVER_ADDR, index);

    NDrive::NVega::TServerSettingsParameter serverSettings;
    if (const auto& value = std::get<TBuffer>(currentServer)) {
        TBufferInput input(value);
        serverSettings.Load(&input);
    }
    DEBUG_LOG << imei << " current server: " << serverSettings.HostPort.Get() << Endl;
    if (
        serverSettings.Protocol != serverSettings.DISABLED &&
        serverSettings.HostPort.Get() != server &&
        serverSettings.Protocol != protocol &&
        !force
    ) {
        WARNING_LOG << imei << " server " << index << " is already set to " << serverSettings.HostPort.Get() << Endl;
    } else {
        serverSettings.HostPort.Set(server);
        serverSettings.InteractionPeriod = 0;
        serverSettings.Protocol = protocol;

        TBuffer value;
        TBufferOutput output(value);
        serverSettings.Save(&output);
        proxy.SetParameter(VEGA_SETTING_SERVER_ADDR, index, value);

        if (pin) {
            TBuffer b;
            TBufferOutput bo(b);
            NDrive::NVega::THardPasswordParameter p;
            p.Value.Set(pin);
            p.Save(&bo);

            proxy.SetParameter(VEGA_SETTING_SERVER_PIN, index, b);
            proxy.SetParameter(VEGA_SETTING_USE_SERVER_PIN, index, ui64(2));
        } else {
            proxy.SetParameter(VEGA_SETTING_USE_SERVER_PIN, index, ui64(1));
        }

        INFO_LOG << imei << " successfully set server " << index << " to " << server << Endl;
    }

    proxy.Disconnect();
    return EXIT_SUCCESS;
}

int main_set(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("apn", main_set_apn, "Set APN");
    modChooser.AddMode("file", main_set_file, "Set file");
    modChooser.AddMode("server", main_set_server, "Set monitoring server");
    return modChooser.Run(argc, argv);
}

int main_proxy(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("list", main_list, "List connected devices");
    modChooser.AddMode("command", main_command, "Send command to a device");
    modChooser.AddMode("get", main_get, "Get a parameter of a device");
    modChooser.AddMode("set", main_set, "Set a parameter of a device");
    return modChooser.Run(argc, argv);
}

int main_read_base(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("input", "Input file").RequiredArgument("FILE").Optional();
    options.AddLongOption("format", "Format: json|DRIVEBACK-693").RequiredArgument("FORMAT").DefaultValue("json");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    TBuffer buffer;
    TBufferOutput output(buffer);
    if (res.Has("input")) {
        const TString& inputFile = res.Get("input");
        TIFStream input(inputFile);
        TransferData(&input, &output);
    } else {
        TransferData(&Cin, &output);
    }

    const TString& format = res.Get("format");

    NDrive::NVega::TSettings settings;
    settings.Parse({ buffer.Data(), buffer.Size() });

    if (format == "json") {
        NJson::TJsonValue result = settings.ToJson();
        NJson::WriteJson(&Cout, &result, /*formatOutput=*/true, /*sortKeys=*/true);
        Cout << Endl;
        return EXIT_SUCCESS;
    }
    if (format == "DRIVEBACK-693") {
        Cout << "Id\tTrack\tPeriod\tChange\tName" << Endl;
        for (auto&& setting : settings) {
            if (setting.Id != VEGA_SETTING_TRANSLATE_SENSORS) {
                continue;
            }
            auto id = setting.SubId;
            NDrive::NVega::TSensorTranslation parameter;
            parameter.Parse(setting.Payload);
            if (parameter.OnChange + parameter.OnPeriod + parameter.OnTrack == 0) {
                continue;
            }
            Cout << id << '\t' << ui32(parameter.OnTrack) << '\t' << ui32(parameter.OnPeriod) << '\t' << ui32(parameter.OnChange) << '\t' << NDrive::NVega::GetSensorName(id) << Endl;
        }
        return EXIT_SUCCESS;
    }

    Cerr << "unknown format: " << format << Endl;
    return EXIT_FAILURE;
}

int main_read(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("base", main_read_base, "Read BASE file");
    return modChooser.Run(argc, argv);
}

namespace {
    NDrive::TSensorId IdFromJson(const NJson::TJsonValue& value) {
        NDrive::TSensorId result;
        Y_ENSURE(NJson::TryFromJson(value, result) || NJson::TryFromJson(value["id"], result));
        return result;
    }
    TString ImeiFromJson(const NJson::TJsonValue& value) {
        if (value.IsUInteger()) {
            return value.GetStringRobust();
        } else {
            return NJson::FromJson<TString>(value);
        }
    }
    TMaybe<TString> OptionalImeiFromJson(const NJson::TJsonValue& value) {
        if (value.IsUInteger()) {
            return value.GetStringRobust();
        } else {
            return NJson::FromJson<TMaybe<TString>>(value);
        }
    }
}

int main_location_get(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    AddSensorApiOptions(options);
    options.AddLongOption("name", "Location name").RequiredArgument("NAME").DefaultValue(NDrive::RawLocationName);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto name = res.Get("name");

    TSensorApiCtx ctx;
    auto api = GetSensorApi(res, ctx);
    auto asyncLocations = api.GetLocations(name);
    for (auto&& [imei, location] : asyncLocations.GetValueSync()) {
        Cout << imei << ": " << location.ToJson().GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_location_set(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("pusher-host", "Pusher API host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("pusher-port", "Pusher API port").RequiredArgument("PORT").DefaultValue("80");
    options.AddLongOption("pusher-token", "Pusher API token").RequiredArgument("SERVICE").DefaultValue("ac6ac7f7a6544f336202c0b058104374");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("deadline", "Location deadline").RequiredArgument("TIMESTAMP");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto deadline = res.Has("deadline") ? TInstant::Seconds(FromString<ui64>(res.Get("deadline"))) : TInstant::Zero();
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);

    NDrive::TPusherOptions pusherOptions;
    pusherOptions.Host = res.Get("pusher-host");
    pusherOptions.Port = FromString(res.Get("pusher-port"));
    pusherOptions.Token = res.Get("pusher-token");
    NDrive::TPusher pusher(pusherOptions);

    NDrive::TLocation location;
    Y_ENSURE(location.TryFromJson(context) || location.TryFromJson(context["location"]));

    INFO_LOG << imei << ": " << location.ToJson().GetStringRobust() << Endl;
    auto pushResult = pusher.Push(imei, location, deadline);
    pushResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_location_set2(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("lb-host", "Logbroker host").RequiredArgument("HOST").DefaultValue("lbkx.logbroker.yandex.net");
    options.AddLongOption("tvm-destination-id", "TVM destination client ID").RequiredArgument("UINT").DefaultValue(ToString(2009819));
    options.AddLongOption("tvm-source-id", "TVM self client ID").RequiredArgument("UINT").DefaultValue(ToString(2000615));
    options.AddLongOption("tvm-secret", "TVM secret").RequiredArgument("STRING");
    options.AddLongOption("service-name", "Service name").RequiredArgument("STRING").DefaultValue("drive_telematics_pre");
    options.AddLongOption("ctype", "Ctype").RequiredArgument("STRING").DefaultValue("stable_drive");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("deadline", "Location deadline").RequiredArgument("TIMESTAMP");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const auto tvmDestinationId = FromString<NTvmAuth::TTvmId>(res.Get("tvm-destination-id"));
    const auto tvmSourceId = FromString<NTvmAuth::TTvmId>(res.Get("tvm-source-id"));
    const auto secret = res.Get("tvm-secret");
    INFO_LOG << "CTYPE: " <<  res.Get("ctype") << Endl;
    NDrive::TSaasPusherOptions pusherOptions;
    pusherOptions.SetServiceName(res.Get("service-name"));
    pusherOptions.SetTvmSourceId(tvmSourceId);
    pusherOptions.SetTvmToken(secret);
    pusherOptions.SetTvmDestinationId(tvmDestinationId);
    pusherOptions.SetHost(res.Get("lb-host"));
    pusherOptions.SetCodec(NPersQueueCommon::ECodec::LZOP);
    NSaas::TSearchMapInputSettings searchMapInputSettings;
    searchMapInputSettings.Ctype = res.Get("ctype");
    searchMapInputSettings.DMHost = "saas-dm-proxy.n.yandex-team.ru";
    searchMapInputSettings.StaticaHost = "saas-searchmap.s3.mds.yandex.net";
    searchMapInputSettings.StaticaQuery = res.Get("ctype");
    pusherOptions.SetSearchMapInputSettings(searchMapInputSettings);
    pusherOptions.SetCodec(NPersQueueCommon::ECodec::LZOP);
    NDrive::TSaasPusher pusher{pusherOptions};

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto deadline = res.Has("deadline") ? TInstant::Seconds(FromString<ui64>(res.Get("deadline"))) : TInstant::Zero();
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);

    NDrive::TLocation location;
    Y_ENSURE(location.TryFromJson(context) || location.TryFromJson(context["location"]));

    INFO_LOG << imei << ": " << location.ToJson().GetStringRobust() << Endl;
    auto pushResult = pusher.Push(imei, location, deadline);
    pushResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_location_remove(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("pusher-host", "Pusher API host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("pusher-port", "Pusher API port").RequiredArgument("PORT").DefaultValue("80");
    options.AddLongOption("pusher-token", "Pusher API token").RequiredArgument("SERVICE").DefaultValue("ac6ac7f7a6544f336202c0b058104374");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    options.AddLongOption("name", "Location name").RequiredArgument("NAME").DefaultValue(NDrive::RawLocationName);
    options.AddLongOption("timestamp", "Location deadline").RequiredArgument("TIMESTAMP").DefaultValue(Seconds());
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto imei = res.Get("imei");
    auto name = res.Get("name");
    auto timestamp = TInstant::Seconds(FromString<ui64>(res.Get("timestamp")));

    NDrive::TPusherOptions pusherOptions;
    pusherOptions.Host = res.Get("pusher-host");
    pusherOptions.Port = FromString(res.Get("pusher-port"));
    pusherOptions.Token = res.Get("pusher-token");
    NDrive::TPusher pusher(pusherOptions);

    auto location = NDrive::TLocation();
    location.Name = name;
    location.Timestamp = timestamp;
    auto pushResult = pusher.Remove(imei, location);
    pushResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_location(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("get", main_location_get, "Get location");
    modChooser.AddMode("set", main_location_set, "Set location in rtline service");
    modChooser.AddMode("set2", main_location_set2, "Set location in SaaS service");
    modChooser.AddMode("remove", main_location_remove, "Remove location");
    return modChooser.Run(argc, argv);
}

int main_sensor_get(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("sensor-host", "Sensor API host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps-prestable.yandex.net");
    options.AddLongOption("sensor-port", "Sensor API port").RequiredArgument("PORT").DefaultValue(17000);
    options.AddLongOption("sensor-service", "Sensor API token").RequiredArgument("SERVICE").DefaultValue("drive_cache");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto optionalImei = res.Has("imei") ? res.Get("imei") : OptionalImeiFromJson(context["imei"]);
    auto optionalId = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : NJson::FromJson<TMaybe<NDrive::TSensorId>>(context["id"]);
    auto replication = 7;
    Y_ENSURE(optionalImei || optionalId);

    NRTLine::TNehSearchClient searchClient(res.Get("sensor-service"), res.Get("sensor-host"), FromString(res.Get("sensor-port")));
    NDrive::TSensorApi api("telematics_client", searchClient);
    std::multimap<TString, NDrive::TSensor> sensors;
    if (optionalImei && optionalId) {
        auto asyncSensor = api.GetSensor(*optionalImei, *optionalId, {}, replication);
        auto optionalSensor = asyncSensor.ExtractValueSync();
        if (optionalSensor) {
            sensors.emplace(*optionalImei, std::move(*optionalSensor));
        }
    } else if (optionalImei) {
        auto asyncSensors = api.GetSensors(*optionalImei);
        auto multisensors = asyncSensors.ExtractValueSync();
        for (auto&& sensor : multisensors) {
            sensors.emplace(*optionalImei, std::move(sensor));
        }
    } else if (optionalId) {
        auto asyncSensors = api.GetSensor(*optionalId);
        auto imeiSensors = asyncSensors.ExtractValueSync();
        for (auto&& [imei, multisensors] : imeiSensors) {
            for (auto&& sensor : multisensors) {
                sensors.emplace(imei, std::move(sensor));
            }
        }
    }
    for (auto&& [imei, sensor] : sensors) {
        auto serialized = sensor.ToJson();
        serialized["imei"] = imei;
        Cout << serialized.GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_sensor_remove(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("pusher-host", "Pusher API host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("pusher-port", "Pusher API port").RequiredArgument("PORT").DefaultValue("80");
    options.AddLongOption("pusher-token", "Pusher API token").RequiredArgument("SERVICE").DefaultValue("ac6ac7f7a6544f336202c0b058104374");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    options.AddLongOption("timestamp", "Sensor timestamp").RequiredArgument("TIMESTAMP");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);
    auto id = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : NJson::FromJson<NDrive::TSensorId>(context["id"]);
    auto timestamp = res.Has("timestamp")
        ? TInstant::Seconds(FromString<ui64>(res.Get("timestamp")))
        : NJson::FromJson<TMaybe<TInstant>>(context["timestamp"]).GetOrElse(Now());

    NDrive::TPusherOptions pusherOptions;
    pusherOptions.Host = res.Get("pusher-host");
    pusherOptions.Port = FromString(res.Get("pusher-port"));
    pusherOptions.Token = res.Get("pusher-token");
    NDrive::TPusher pusher(pusherOptions);

    INFO_LOG << imei << ": " << id << ' ' << timestamp << Endl;
    auto removeResult = pusher.Remove(imei, id.Id, id.SubId, timestamp);
    removeResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_sensor_set(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("pusher-host", "Pusher API host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("pusher-port", "Pusher API port").RequiredArgument("PORT").DefaultValue("80");
    options.AddLongOption("pusher-token", "Pusher API token").RequiredArgument("SERVICE").DefaultValue("ac6ac7f7a6544f336202c0b058104374");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("deadline", "Sensor deadline").RequiredArgument("TIMESTAMP");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    options.AddLongOption("since", "Sensor since").RequiredArgument("TIMESTAMP");
    options.AddLongOption("timestamp", "Sensor timestamp").RequiredArgument("TIMESTAMP");
    options.AddLongOption("value", "Sensor value").RequiredArgument("VALUE");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto deadline = res.Has("deadline") ? TInstant::Seconds(FromString<ui64>(res.Get("deadline"))) : TInstant::Zero();
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);
    auto id = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : IdFromJson(context);
    auto timestamp = res.Has("timestamp")
        ? TInstant::Seconds(FromString<ui64>(res.Get("timestamp")))
        : NJson::FromJson<TInstant>(context["timestamp"]);
    auto since = res.Has("since")
        ? TInstant::Seconds(FromString<ui64>(res.Get("since")))
        : NJson::FromJson<TMaybe<TInstant>>(context["since"]).GetOrElse(timestamp);
    auto value = res.Has("value")
        ? NDrive::SensorValueFromString(TString{res.Get("value")}, id.Id)
        : NDrive::SensorValueFromJson(context["value"], id.Id);

    NDrive::TPusherOptions pusherOptions;
    pusherOptions.Host = res.Get("pusher-host");
    pusherOptions.Port = FromString(res.Get("pusher-port"));
    pusherOptions.Token = res.Get("pusher-token");
    NDrive::TPusher pusher(pusherOptions);

    NDrive::TSensor sensor(id);
    sensor.Value = value;
    sensor.Timestamp = timestamp;
    sensor.Since = since;

    INFO_LOG << imei << ": " << sensor << Endl;
    auto pushResult = pusher.Push(imei, sensor, deadline);
    pushResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_sensor_set2(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("lb-host", "Logbroker host").RequiredArgument("HOST").DefaultValue("lbkx.logbroker.yandex.net");
    options.AddLongOption("tvm-destination-id", "TVM destination client ID").RequiredArgument("UINT").DefaultValue(ToString(2009819));
    options.AddLongOption("tvm-source-id", "TVM self client ID").RequiredArgument("UINT").DefaultValue(ToString(2000615));
    options.AddLongOption("tvm-secret", "TVM secret").RequiredArgument("STRING");
    options.AddLongOption("service-name", "Service name").RequiredArgument("STRING").DefaultValue("drive_telematics_pre");
    options.AddLongOption("ctype", "Ctype").RequiredArgument("STRING").DefaultValue("stable_drive");
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("deadline", "Sensor deadline").RequiredArgument("TIMESTAMP");
    options.AddLongOption("imei", "Device IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    options.AddLongOption("since", "Sensor since").RequiredArgument("TIMESTAMP");
    options.AddLongOption("timestamp", "Sensor timestamp").RequiredArgument("TIMESTAMP");
    options.AddLongOption("value", "Sensor value").RequiredArgument("VALUE");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const auto tvmDestinationId = FromString<NTvmAuth::TTvmId>(res.Get("tvm-destination-id"));
    const auto tvmSourceId = FromString<NTvmAuth::TTvmId>(res.Get("tvm-source-id"));
    const auto secret = res.Get("tvm-secret");
    INFO_LOG << "CTYPE: " <<  res.Get("ctype") << Endl;
    NDrive::TSaasPusherOptions pusherOptions;
    pusherOptions.SetServiceName(res.Get("service-name"));
    pusherOptions.SetTvmSourceId(tvmSourceId);
    pusherOptions.SetTvmToken(secret);
    pusherOptions.SetTvmDestinationId(tvmDestinationId);
    pusherOptions.SetHost(res.Get("lb-host"));
    pusherOptions.SetCodec(NPersQueueCommon::ECodec::LZOP);
    NSaas::TSearchMapInputSettings searchMapInputSettings;
    searchMapInputSettings.Ctype = res.Get("ctype");
    searchMapInputSettings.DMHost = "saas-dm-proxy.n.yandex-team.ru";
    searchMapInputSettings.StaticaHost = "saas-searchmap.s3.mds.yandex.net";
    searchMapInputSettings.StaticaQuery = res.Get("ctype");
    pusherOptions.SetSearchMapInputSettings(searchMapInputSettings);
    pusherOptions.SetCodec(NPersQueueCommon::ECodec::LZOP);
    NDrive::TSaasPusher pusher{pusherOptions};

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto deadline = res.Has("deadline") ? TInstant::Seconds(FromString<ui64>(res.Get("deadline"))) : TInstant::Zero();
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);
    auto id = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : IdFromJson(context);
    auto timestamp = res.Has("timestamp")
            ? TInstant::Seconds(FromString<ui64>(res.Get("timestamp")))
            : NJson::FromJson<TInstant>(context["timestamp"]);
    auto since = res.Has("since")
            ? TInstant::Seconds(FromString<ui64>(res.Get("since")))
            : NJson::FromJson<TMaybe<TInstant>>(context["since"]).GetOrElse(timestamp);
    auto value = res.Has("value")
            ? NDrive::SensorValueFromString(TString{res.Get("value")}, id.Id)
            : NDrive::SensorValueFromJson(context["value"], id.Id);

    NDrive::TSensor sensor(id);
    sensor.Value = value;
    sensor.Timestamp = timestamp;
    sensor.Since = since;

    INFO_LOG << imei << ": " << sensor << Endl;
    auto pushResult = pusher.Push(imei, sensor, deadline);
    pushResult.GetValueSync();
    return EXIT_SUCCESS;
}

int main_sensor_fix_future(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("sensor-host", "Sensor API host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps-prestable.yandex.net");
    options.AddLongOption("sensor-port", "Sensor API port").RequiredArgument("PORT").DefaultValue(17000);
    options.AddLongOption("sensor-service", "Sensor API token").RequiredArgument("SERVICE").DefaultValue("drive_cache");
    options.AddLongOption("pusher-host", "Pusher API host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("pusher-port", "Pusher API port").RequiredArgument("PORT").DefaultValue("80");
    options.AddLongOption("pusher-token", "Pusher API token").RequiredArgument("SERVICE").DefaultValue("ac6ac7f7a6544f336202c0b058104374");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto id = FromString<NDrive::TSensorId>(res.Get("id"));

    NDrive::TPusherOptions pusherOptions;
    pusherOptions.Host = res.Get("pusher-host");
    pusherOptions.Port = FromString(res.Get("pusher-port"));
    pusherOptions.Token = res.Get("pusher-token");
    NDrive::TPusher pusher(pusherOptions);

    NRTLine::TNehSearchClient searchClient(res.Get("sensor-service"), res.Get("sensor-host"), FromString(res.Get("sensor-port")));
    NDrive::TSensorApi api("telematics_client", searchClient);

    auto threshold = Now() + TDuration::Days(1);
    auto sensors = api.GetSensor(id, {}, 7);
    auto deletes = TMap<TString, NThreading::TFuture<NDrive::IPusher::TPushResult>>();
    for (auto&& [imei, multisensors] : sensors.GetValueSync()) {
        for (auto&& sensor : multisensors) {
            if (sensor.Timestamp > threshold) {
                INFO_LOG << imei << ": " << sensor << Endl;
                auto deletion = pusher.Remove(imei, sensor.Id, sensor.SubId, sensor.Timestamp);
                deletes.emplace(imei, deletion);
            }
        }
    }
    for (auto&& [imei, deletion] : deletes) {
        deletion.Wait();
        if (deletion.HasException()) {
            ERROR_LOG << imei << ": " << NThreading::GetExceptionMessage(deletion) << Endl;
        }
    }
    return EXIT_SUCCESS;
}

int main_sensor_history_add(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("clickhouse-host", "ClickHouse host").RequiredArgument("HOST");
    options.AddLongOption("clickhouse-port", "ClickHouse port").RequiredArgument("PORT").DefaultValue(9440);
    options.AddLongOption("clickhouse-database", "ClickHouse database").RequiredArgument("DATABASE").DefaultValue("production");
    options.AddLongOption("clickhouse-user", "ClickHouse user").RequiredArgument("USER").DefaultValue("carsharing");
    options.AddLongOption("clickhouse-password", "ClickHouse password").RequiredArgument("PASSWORD").Optional();
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("object-id", "ObjectId").RequiredArgument("UUID");
    options.AddLongOption("imei", "IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    options.AddLongOption("timestamp", "Sensor timestamp").RequiredArgument("TIMESTAMP");
    options.AddLongOption("value", "Sensor value").RequiredArgument("VALUE");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NClickHouse::TClientOptions clientOptions;
    clientOptions.SetHost(res.Get("clickhouse-host"));
    clientOptions.SetPort(FromString(res.Get("clickhouse-port")));
    clientOptions.SetDefaultDatabase(res.Get("clickhouse-database"));
    clientOptions.SetUser(res.Get("clickhouse-user"));
    clientOptions.SetPassword(res.Has("clickhouse-password") ? res.Get("clickhouse-password") : GetEnv("CLICKHOUSE_PASSWORD"));
    clientOptions.SetUseSsl(true);

    NDrive::TClickHousePusher::TBalancingOptions balancingOptions;
    NDrive::TClickHousePusher client(clientOptions, balancingOptions);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto objectId = res.Has("object-id") ? res.Get("object-id") : context["object_id"].GetStringRobust();
    auto imei = res.Has("imei") ? res.Get("imei") : ImeiFromJson(context["imei"]);
    auto id = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : IdFromJson(context);
    auto timestamp = res.Has("timestamp")
        ? TInstant::Seconds(FromString<ui64>(res.Get("timestamp")))
        : NJson::FromJson<TInstant>(context["timestamp"]);
    auto value = res.Has("value")
        ? NDrive::SensorValueFromString(TString{res.Get("value")}, id.Id)
        : NDrive::SensorValueFromJson(context["value"], id.Id);
    NDrive::TSensor sensor(id);
    sensor.Value = value;
    sensor.Timestamp = timestamp;

    auto groupstamp = TInstant::Hours(timestamp.Hours());
    auto received = timestamp;

    auto result = client.Push(objectId, imei, sensor, groupstamp, received);
    result.GetValueSync();

    return EXIT_SUCCESS;
}

int main_sensor_history_get(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("clickhouse-endpoints", "ClickHouse endpoints").RequiredArgument("https://HOST:PORT https://HOST:PORT");
    options.AddLongOption("clickhouse-database", "ClickHouse database").RequiredArgument("DATABASE").DefaultValue("production");
    options.AddLongOption("clickhouse-user", "ClickHouse user").RequiredArgument("USER").DefaultValue("carsharing");
    options.AddLongOption("clickhouse-password", "ClickHouse password").RequiredArgument("PASSWORD").Optional();
    options.AddLongOption("context", "Context").RequiredArgument("JSON");
    options.AddLongOption("object-id", "ObjectId").RequiredArgument("UUID");
    options.AddLongOption("imei", "IMEI").RequiredArgument("IMEI");
    options.AddLongOption("id", "SensorId").RequiredArgument("ID[-SUBID]");
    options.AddLongOption("since", "Sensor timestamp since").RequiredArgument("TIMESTAMP");
    options.AddLongOption("until", "Sensor timestamp until").RequiredArgument("TIMESTAMP");
    options.AddLongOption("limit", "Sensor count limit").RequiredArgument("VALUE").DefaultValue(100);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NClickHouse::TAsyncClientOptions clientOptions;
    clientOptions.Endpoints = StringSplitter(res.Get("clickhouse-endpoints")).SplitBySet(" ,").SkipEmpty();
    clientOptions.Database = res.Get("clickhouse-database");
    clientOptions.User = res.Get("clickhouse-user");
    clientOptions.Password = res.Has("clickhouse-password") ? res.Get("clickhouse-password") : GetEnv("CLICKHOUSE_PASSWORD");
    NDrive::TSensorHistoryClient client(clientOptions);

    auto context = res.Has("context") ? NJson::ReadJsonFastTree(res.Get("context")) : NJson::JSON_NULL;
    auto objectId = res.Has("object-id") ? res.Get("object-id") : context["object_id"].GetString();
    auto imei = res.Has("imei") ? res.Get("imei") : OptionalImeiFromJson(context["imei"]).GetOrElse(TString{});
    auto id = res.Has("id") ? FromString<NDrive::TSensorId>(res.Get("id")) : IdFromJson(context);

    auto since = res.Has("since") ? MakeMaybe(FromString<TTimestamp>(res.Get("since"))) : Nothing();
    auto until = res.Has("until") ? MakeMaybe(FromString<TTimestamp>(res.Get("until"))) : Nothing();
    auto limit = FromString<ui64>(res.Get("limit"));

    Y_UNUSED(id);
    Y_UNUSED(limit);

    auto asyncResult = client.Get(objectId, imei, since.GetOrElse(TInstant::Zero()), until.GetOrElse(TInstant::Max()));
    auto result = asyncResult.GetValueSync();
    if (result) {
        NJson::TJsonValue report;
        result->Fill(report);
        Cout << NJson::WriteJson(report, /*formatOutput=*/true);
    }

    return EXIT_SUCCESS;
}

int main_sensor_history_query(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("clickhouse-endpoints", "ClickHouse endpoints").RequiredArgument("https://HOST:PORT https://HOST:PORT");
    options.AddLongOption("clickhouse-database", "ClickHouse database").RequiredArgument("DATABASE").DefaultValue("production");
    options.AddLongOption("clickhouse-user", "ClickHouse user").RequiredArgument("USER").DefaultValue("carsharing");
    options.AddLongOption("clickhouse-password", "ClickHouse password").RequiredArgument("PASSWORD").Optional();
    options.AddLongOption("query", "Query").RequiredArgument("SQL");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NClickHouse::TAsyncClientOptions clientOptions;
    clientOptions.Endpoints = StringSplitter(res.Get("clickhouse-endpoints")).SplitBySet(" ,").SkipEmpty();
    clientOptions.Database = res.Get("clickhouse-database");
    clientOptions.User = res.Get("clickhouse-user");
    clientOptions.Password = res.Has("clickhouse-password") ? res.Get("clickhouse-password") : GetEnv("CLICKHOUSE_PASSWORD");
    NDrive::TSensorHistoryClient client(clientOptions);

    const auto& query = res.Get("query");

    auto asyncResult = client.GetClickHouseClient().SelectIds(query);
    auto result = asyncResult.GetValueSync();
    for (auto&& id : result) {
        Cout << id << Endl;
    }

    return EXIT_SUCCESS;
}

int main_sensor_history(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("add", main_sensor_history_add, "Add sensor history");
    modChooser.AddMode("get", main_sensor_history_get, "Get sensor history");
    modChooser.AddMode("query", main_sensor_history_query, "Select ids from sensor history");
    return modChooser.Run(argc, argv);
}

int main_sensor(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("get", main_sensor_get, "Get sensor");
    modChooser.AddMode("history", main_sensor_history, "Manage sensor history");
    modChooser.AddMode("remove", main_sensor_remove, "Remove sensor");
    modChooser.AddMode("set", main_sensor_set, "Set sensor in rtline service");
    modChooser.AddMode("set2", main_sensor_set2, "Set sensor in SaaS service");
    modChooser.AddMode("fix_future", main_sensor_fix_future, "Set sensor");
    return modChooser.Run(argc, argv);
}

int main_sign(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("input", "Input file").RequiredArgument("FILE").Optional();
    options.AddLongOption("output", "Output file").RequiredArgument("FILE").Optional();
    options.AddLongOption("key", "Key file").RequiredArgument("FILE").Required();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    TBuffer buffer;
    TBufferOutput intermediate(buffer);
    if (res.Has("input")) {
        const TString& inputFile = res.Get("input");
        TIFStream input(inputFile);
        TransferData(&input, &intermediate);
    } else {
        TransferData(&Cin, &intermediate);
    }

    NDrive::NVega::TSettings settings;
    settings.Parse({ buffer.Data(), buffer.Size() });

    const TString& keyFile = res.Get("key");
    std::ifstream keyInput(keyFile);
    std::array<ui32, 4> key;
    for (auto&& i : key) {
        keyInput >> std::hex >> i;
    }

    const auto data = settings.ToSigned(key);
    if (res.Has("output")) {
        const TString& outputFile = res.Get("output");
        TOFStream output(outputFile);
        output.Write(data.data(), data.size());
    } else {
        Cout.Write(data.data(), data.size());
    }

    return EXIT_SUCCESS;
}

int main(int argc, const char** argv) {
    DoInitGlobalLog("cerr", TLOG_INFO, false, false);

    TModChooser modChooser;
    modChooser.AddMode("emulator", main_device_emulator, "Start device emulators");
    modChooser.AddMode("emulator2", main_device_emulator2, "Start device emulators");
    modChooser.AddMode("api", main_api, "API mode");
    modChooser.AddMode("api2", main_api2, "APIv2 mode");
    modChooser.AddMode("location", main_location, "Work with locations");
    modChooser.AddMode("proxy", main_proxy, "Configurator mode");
    modChooser.AddMode("read", main_read, "Read data mode");
    modChooser.AddMode("sensor", main_sensor, "Work with sensors");
    modChooser.AddMode("sign", main_sign, "Sign data mode");
    try {
        return modChooser.Run(argc, argv);
    } catch (const std::exception& e) {
        Cerr << "An exception has occurred: " << FormatExc(e) << Endl;
        return EXIT_FAILURE;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TPosition& result) {
    return NJson::ParseField(value["lat"], result.Y, true) && NJson::ParseField(value["lon"], result.X, true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TSettingsState& result) {
    return NJson::ParseField(value["imei"], result.IMEI) &&
        NJson::ParseField(value["position"], result.Position) &&
        NJson::ParseField(value["fuel_level"], result.FuelLevel);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TDeviceState& result) {
    return NJson::ParseField(value["imei"], result.IMEI) &&
        NJson::ParseField(value["location"], result.Location) &&
        NJson::ParseField(value["tags"], result.Tags) &&
        NJson::ParseField(value["telematics"]["is_engine_on"], result.EngineOn) &&
        NJson::ParseField(value["telematics"]["fuel_level"], result.FuelLevel) &&
        NJson::ParseField(value["telematics"]["fuel_distance"], result.FuelDistance) &&
        NJson::ParseField(value["telematics"]["mileage"], result.Mileage) &&
        NJson::ParseField(value["telematics"]["speed"], result.Speed);
}
