#include <drive/telematics/api/server/client.h>
#include <drive/telematics/client/library/client.h>
#include <drive/telematics/client/car_emulator/car_emulator.h>
#include <drive/telematics/protocol/actions.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TelematicsTestClientSuite) {
    Y_UNIT_TEST(Moving) {
        const TString saasHost = "saas-searchproxy-maps.yandex.net";
        const ui16 saasPort = 17000;
        const TString saasService = "drive_router";
        TAtomicSharedPtr<NGraph::TRouter> router = MakeAtomicShared<NGraph::TRouter>(saasHost, saasPort, saasService);

        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TGeoCoord start(37.590285, 55.749888);
        const TGeoCoord finish(37.593370, 55.747982);

        TString imei = "10001000424242";
        NDrive::TCarEmulator emulator(imei, "localhost", config->GetTelematicsServerOptions().Port, router);
        NDrive::TTelematicsClientContext& context = emulator.GetContext();
        auto haversine = context.GetCurrentPosition().GetLengthTo(finish);
        auto mileageBefore = context.GetOdometerKm();
        context.SetCurrentPosition(start);
        context.SetPathSpeed(300);
        UNIT_ASSERT(haversine > 1000);

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient::TOptions options;
        NDrive::TTelematicsClient client(sd, options);

        NDrive::NVega::TCommandRequest::TMoveToCoordParameter parameter;
        parameter.Lat = finish.Y;
        parameter.Lon = finish.X;
        NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::MOVE_TO_COORD);
        command.Argument.Set(parameter);

        auto move = client.Command(imei, command);
        move.WaitAndEnsureSuccess();

        Sleep(TDuration::Seconds(150));

        UNIT_ASSERT_DOUBLES_EQUAL(context.GetCurrentPosition().GetLengthTo(finish), 0.0, 100);
        auto mileageAfter = context.GetOdometerKm();
        INFO_LOG << haversine << '\t' << mileageAfter << '\t' << mileageBefore << Endl;
        UNIT_ASSERT(mileageAfter - mileageBefore > haversine / 1000);

        parameter.Lat = start.Y;
        parameter.Lon = start.X;
        parameter.Speed = 0;

        NDrive::NVega::TCommand command2(NDrive::NVega::ECommandCode::MOVE_TO_COORD);
        command2.Argument.Set(parameter);

        auto move2 = client.Command(imei, command2);
        move2.WaitAndEnsureSuccess();
        UNIT_ASSERT_DOUBLES_EQUAL(context.GetCurrentPosition().GetLengthTo(start), 0.0, 100);
    }
}
