#include "file.h"

void NDrive::NVega::TGetFileHandler::OnAdd(NVega::IConnection& connection) {
    connection.SendMessage(CreateFileRequest(0, 0));
}

NDrive::NVega::EHandlerStatus NDrive::NVega::TGetFileHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    if (Now() > Deadline) {
        Result = TGetFileResponse::EResult::BUSY;
        Signal();
        return EHandlerStatus::Finish;
    }
    if (message.GetMessageType() == GET_FILE_RESPONSE) {
        const auto& response = message.As<TGetFileResponse>();
        if (response.Name.Get() == Name) {
            Result = static_cast<TGetFileResponse::EResult>(response.Content.Result);
            switch (Result) {
            case TGetFileResponse::OK: {
                Size = response.TotalSize;
                if (Origin > 0 && static_cast<size_t>(Origin) > Size) {
                    Result = TGetFileResponse::EResult::DATA_SIZE_ERROR;
                    Signal();
                    return EHandlerStatus::Finish;
                }
                if (Origin < 0) {
                    Origin = std::max<ssize_t>(Size + Origin, 0);
                }
                Data.Append(response.Content.Data.Data(), response.Content.Data.Size());

                size_t origin = static_cast<size_t>(Origin);
                Y_ENSURE(origin - Origin == 0);
                size_t offset = origin + Data.Size();
                size_t remaining = Size > offset ? Size - offset : 0;
                size_t requesting = std::min(remaining, BufferSize);
                if (requesting) {
                    connection.SendMessage(CreateFileRequest(offset, requesting));
                    return EHandlerStatus::Continue;
                } else {
                    // fallthrough
                }
            }
            default:
                Signal();
                return EHandlerStatus::Finish;
            }
        }
    }
    return EHandlerStatus::Continue;
}

THolder<NDrive::NProtocol::IMessage> NDrive::NVega::TGetFileHandler::CreateFileRequest(size_t offset, size_t size) const {
    Y_ENSURE(offset <= Max<ui32>());
    Y_ENSURE(size <= Max<ui16>());
    auto request = MakeHolder<NDrive::NVega::TGetFileRequest>();
    request->Name.Set(Name);
    request->Offset = offset;
    request->Size = size;
    THolder<NDrive::NProtocol::IMessage> result = MakeHolder<NDrive::NVega::TMessage>(std::move(request));
    return result;
}

NDrive::NVega::TChunkedFileHandler::TChunkedFileHandler(const TString& name, TBuffer&& content, ui16 chunkSize /*= 1000*/)
    : Name(name)
    , Content(std::move(content))
    , ChunkSize(chunkSize)
    , Count(Content.size() / ChunkSize + (Content.size() % ChunkSize > 0 ? 1 : 0))
    , Id(0)
    , Offset(0)
    , Result(TFileChunkResponse::UNKNOWN_ERROR)
{
}

void NDrive::NVega::TChunkedFileHandler::OnAdd(NVega::IConnection& connection) {
    Lock = NNamedLock::TryAcquireLock(connection.GetId() + ':' + Name);
    if (Lock) {
        connection.SendMessage(CreateNextChunkRequest());
    } else {
        Finish(TFileChunkResponse::BUSY_ERROR);
    }
}

NDrive::NVega::EHandlerStatus NDrive::NVega::TChunkedFileHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == FILE_CHUNK_RESPONSE) {
        const auto& response = message.As<TFileChunkResponse>();
        if (response.Name.View() == Name) {
            if (response.Chunk != Id) {
                return Finish(TFileChunkResponse::UNORDERED_ERROR);
            }

            auto result = static_cast<TFileChunkResponse::EResult>(response.Result);
            switch (result) {
            case NDrive::NVega::TFileChunkResponse::OK:
                if (Offset < Content.size()) {
                    connection.SendMessage(CreateNextChunkRequest());
                    break;
                } else {
                    [[fallthrough]];
                }
            default:
                return Finish(result);
            }
        }
    }
    return EHandlerStatus::Continue;
}

THolder<NDrive::NProtocol::IMessage> NDrive::NVega::TChunkedFileHandler::CreateNextChunkRequest() {
    Y_ENSURE(Offset < Content.size());
    Y_ENSURE(Offset % ChunkSize == 0);

    auto request = MakeHolder<NDrive::NVega::TFileChunkRequest>();
    request->Name.Set(Name);

    TBuffer data(Content.data() + Offset, std::min(Content.size() - Offset, ChunkSize));
    Id = Offset / ChunkSize + 1;
    Y_ASSERT(Id <= Count);
    Offset += data.size();
    request->Chunk = NDrive::NVega::TFileChunkRequest::TChunk(Id, Count, std::move(data));
    THolder<NDrive::NProtocol::IMessage> result = MakeHolder<NDrive::NVega::TMessage>(std::move(request));
    return result;
}

NDrive::NVega::EHandlerStatus NDrive::NVega::TChunkedFileHandler::Finish(TFileChunkResponse::EResult result) {
    Result = result;
    Lock.Destroy();
    Signal();
    return EHandlerStatus::Finish;
}

void NDrive::NVega::TStreamFileHandler::OnAdd(NVega::IConnection& connection) {
    connection.SendMessage(CreateFileRequest());
}

NDrive::NVega::EHandlerStatus NDrive::NVega::TStreamFileHandler::OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) {
    Y_UNUSED(connection);
    if (message.GetMessageType() == FILE_STREAM_RESPONSE) {
        const auto& response = message.As<TFileStreamResponse>();
        if (response.Name.Get() == Name) {
            Result = static_cast<TFileStreamResponse::EResult>(response.Result);
            Signal();
            return EHandlerStatus::Finish;
        }
    }
    return EHandlerStatus::Continue;
}


THolder<NDrive::NProtocol::IMessage> NDrive::NVega::TStreamFileHandler::CreateFileRequest() {
    auto request = MakeHolder<TFileStreamRequest>();
    request->Name.Set(Name);
    request->Content.Set(std::move(Content));
    THolder<NDrive::NProtocol::IMessage> result = MakeHolder<NDrive::NVega::TMessage>(std::move(request));
    return result;
}
