#pragma once

#include "handler.h"

#include <drive/telematics/protocol/vega.h>

#include <library/cpp/threading/named_lock/named_lock.h>

namespace NDrive {
    namespace NVega {
        class TGetFileHandler: public NProtocol::TCommonHandler {
        public:
            TGetFileHandler(const TString& name, ssize_t offset = 0, size_t bufferSize = 1024)
                : Name(name)
                , BufferSize(bufferSize)
                , Offset(offset)
                , Deadline(Now() + TDuration::Minutes(5))
                , Origin(offset)
                , Size(0)
                , Result(TGetFileResponse::UNKNOWN_ERROR)
            {
                Y_UNUSED(Offset);
            }

            TInstant GetDeadline() const {
                return Deadline;
            }
            void SetDeadline(TInstant deadline) {
                Deadline = deadline;
            }

            const TBuffer& GetData() const {
                return Data;
            }
            TBuffer& GetData() {
                return Data;
            }
            TGetFileResponse::EResult GetResult() const {
                return Result;
            }
            size_t GetSize() const {
                return Size;
            }

            virtual void OnAdd(NVega::IConnection& connection) override;
            virtual NProtocol::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        private:
            THolder<NProtocol::IMessage> CreateFileRequest(size_t offset, size_t size) const;

        private:
            const TString Name;
            const size_t BufferSize;
            const ssize_t Offset;

            TInstant Deadline;

            TBuffer Data;
            ssize_t Origin;
            size_t Size;
            TGetFileResponse::EResult Result;
        };

        class TChunkedFileHandler : public NProtocol::TCommonHandler {
        public:
            TChunkedFileHandler(const TString& name, TBuffer&& content, ui16 chunkSize = 1000);

            TFileChunkResponse::EResult GetResult() const {
                return Result;
            }

            virtual void OnAdd(NVega::IConnection& connection) override;
            virtual NProtocol::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        private:
            THolder<NProtocol::IMessage> CreateNextChunkRequest();
            NProtocol::EHandlerStatus Finish(TFileChunkResponse::EResult result);

        private:
            const TString Name;
            const TBuffer Content;

            const size_t ChunkSize;
            const size_t Count;
            size_t Id;
            size_t Offset;

            NNamedLock::TNamedLockPtr Lock;
            TFileChunkResponse::EResult Result;
        };

        class TStreamFileHandler: public NProtocol::TCommonHandler {
        public:
            TStreamFileHandler(const TString& name, TBuffer&& content)
                : Name(name)
                , Content(std::move(content))
                , Result(TFileStreamResponse::UNKNOWN_ERROR)
            {
            }

            TFileStreamResponse::EResult GetResult() const {
                return Result;
            }

            virtual void OnAdd(NVega::IConnection& connection) override;
            virtual NProtocol::EHandlerStatus OnMessage(NVega::IConnection& connection, const NProtocol::IMessage& message) override;

        private:
            THolder<NProtocol::IMessage> CreateFileRequest();

        private:
            const TString Name;
            TBuffer Content;

            TFileStreamResponse::EResult Result;
        };
    }
}
