#include "firmware.h"

#include <rtline/library/json/parse.h>

#include <util/charset/utf8.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/string/subst.h>

ui32 NDrive::NVega::TFirmwareInfo::GetCanCount() const {
    switch (Type) {
    case MT32K_3:
    case MT32K_2:
        return 3;
    case MT32K:
    case MT25K:
    case MT32K_MTX:
    case Unknown:
        return 2;
    }
}

bool NDrive::NVega::TFirmwareInfo::operator<(const TFirmwareInfo& other) const {
    return std::tie(Model, Type, Revision, FullName) < std::tie(other.Model, other.Type, other.Revision, other.FullName);
}

bool NDrive::NVega::TFirmwareInfo::IsCompatibleWith(const TFirmwareInfo& other) const {
    return std::tie(Model, Type) == std::tie(other.Model, other.Type);
}

bool NDrive::NVega::TFirmwareInfo::IsFeatureSupported(EFeature feature) const {
    switch (feature) {
    case NDrive::NVega::TFirmwareInfo::Panic:
        return Revision >= 35;
    case NDrive::NVega::TFirmwareInfo::FastData:
        return Revision >= 42;
    case NDrive::NVega::TFirmwareInfo::SwitchSim:
        return Revision >= 40;
    }
}

bool NDrive::NVega::TFirmwareInfo::CanBeTanker() const {
    if (Type == TFirmwareInfo::MT32K && Model.Contains("tanker")) {
        return true;
    }
    return false;
}

bool NDrive::NVega::TFirmwareInfo::CanUpdateFuel() const {
    if (Model.Contains("rapid") || Model.Contains("polo")) {
        return true;
    }
    return false;
}

TString NDrive::NVega::NormalizeFirmwareName(const TString& name) {
    TString result = name;
    SubstGlobal(result, '\'', '_');
    SubstGlobal(result, '"', '_');
    return result;
}

NDrive::NVega::TFirmwareInfo NDrive::NVega::ParseFirmwareInfo(const TString& name) {
    TFirmwareInfo result;
    result.FullName = NormalizeFirmwareName(name);
    TString lowercased = ToLowerUTF8(name);
    for (auto&& i : StringSplitter(lowercased).Split(' ').SkipEmpty()) {
        auto token = i.Token();
        if (token == "vega" || token == "lte" || token == "lp") {
            continue;
        }
        if (token == "mt-25k") {
            result.Type = TFirmwareInfo::MT25K;
            continue;
        }
        if (token == "mt-32k") {
            result.Type = TFirmwareInfo::MT32K;
            continue;
        }
        if (token == "v2") {
            if (result.Type == TFirmwareInfo::MT32K) {
                result.Type = TFirmwareInfo::MT32K_2;
            }
            continue;
        }
        if (token == "v3") {
            if (result.Type == TFirmwareInfo::MT32K) {
                result.Type = TFirmwareInfo::MT32K_3;
            }
            continue;
        }
        if (token == "'mtx'") {
            if (
                result.Type == TFirmwareInfo::MT32K ||
                result.Type == TFirmwareInfo::MT32K_2 ||
                result.Type == TFirmwareInfo::MT32K_3
            ) {
                result.SubType = result.Type;
                result.Type = TFirmwareInfo::MT32K_MTX;
            }
            continue;
        }
        if (token == "0.10b") {
            continue;
        }
        constexpr TStringBuf revisionPrefix = "rc";
        if (token.StartsWith(revisionPrefix)) {
            TryFromString(token.Skip(revisionPrefix.size()), result.Revision);
            continue;
        }
        if (token.StartsWith("'") && token.EndsWith("'")) {
            result.Model = ToString(token.Skip(1).Chop(1));
            continue;
        }
        if (token.StartsWith("_") && token.EndsWith("_")) {
            result.Model = ToString(token.Skip(1).Chop(1));
            continue;
        }
        if (!result.Tag.empty()) {
            result.Tag.append(' ');
        }
        result.Tag.append(token);
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TFirmwareInfo>(const NDrive::NVega::TFirmwareInfo& object) {
    NJson::TJsonValue result;
    result["alias"] = object.Alias;
    result["full_name"] = NDrive::NVega::NormalizeFirmwareName(object.FullName);
    result["model"] = object.Model;
    result["revision"] = object.Revision;
    result["tag"] = object.Tag;
    result["type"] = ToString(object.Type);
    result["percent"] = object.Percent;
    return result;
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TFirmwareInfo>(const NJson::TJsonValue& value, NDrive::NVega::TFirmwareInfo& result) {
    TString type;
    if (!ParseField(value["type"], type) || (!type.empty() && !TryFromString(type, result.Type))) {
        return false;
    }
    TString fullName;
    if (ParseField(value["full_name"], fullName, true)) {
        result.FullName = NDrive::NVega::NormalizeFirmwareName(fullName);
    } else {
        return false;
    }
    return
        ParseField(value["alias"], result.Alias) &&
        ParseField(value["model"], result.Model) &&
        ParseField(value["revision"], result.Revision) &&
        ParseField(value["tag"], result.Tag) &&
        ParseField(value["percent"], result.Percent);
}
