#pragma once

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/system/event.h>

namespace NDrive {
    namespace NProtocol {
        class IHandler;
        class IMessage;
        using THandlerPtr = TAtomicSharedPtr<IHandler>;

        enum class EHandlerStatus {
            Finish,   // remove handler from pool
            Continue  // keep handler in pool
        };

        class IConnection {
        public:
            virtual ~IConnection() = default;

            virtual const TString& GetId() const = 0;

            virtual void AddMessageHandler(THandlerPtr handler) = 0;
            virtual void SendMessage(THolder<IMessage>&& message) = 0;
            virtual void Drop() = 0;
        };

        class IHandler {
        public:
            virtual ~IHandler() = default;

            virtual void OnAdd(IConnection& connection) = 0;
            virtual void OnDrop() = 0;

            virtual EHandlerStatus OnMessage(IConnection& connection, const IMessage& message) = 0;
        };

        class TCallbackHandler: public IHandler {
        public:
            using TCallback = std::function<void(THandlerPtr)>;

        public:
            TCallbackHandler(THandlerPtr handler, TCallback&& callback)
                : Handler(handler)
                , Callback(std::move(callback))
            {
            }

            virtual void OnAdd(IConnection& connection) override;
            virtual void OnDrop() override;
            virtual EHandlerStatus OnMessage(IConnection& connection, const IMessage& message) override;

        private:
            void InvokeCallback();

        private:
            THandlerPtr Handler;
            TCallback Callback;
        };

        class TCommonHandler: public IHandler {
        public:
            bool Wait(TDuration timeout = TDuration::Max()) {
                return Ev.WaitT(timeout);
            }
            bool WasDropped() const {
                return Dropped;
            }

            virtual void OnAdd(IConnection& connection) override;
            virtual void OnDrop() override;

        protected:
            void Signal() {
                Ev.Signal();
            }

        private:
            TManualEvent Ev;
            volatile bool Dropped = false;
        };

        TString GenerateId(TStringBuf imei, TStringBuf type = {});

        void HandleCommon(IConnection& connection, TAtomicSharedPtr<TCommonHandler> handler, TDuration timeout);

        template <class T>
        TAtomicSharedPtr<T> Handle(IConnection& connection, TAtomicSharedPtr<T> handler, TDuration timeout = TDuration::Max()) {
            HandleCommon(connection, handler, timeout);
            return handler;
        }
    }
}
