#include "actions.h"

#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/generic/map.h>
#include <util/generic/serialized_enum.h>
#include <util/string/hex.h>
#include <util/string/split.h>

NDrive::NVega::TCommand NDrive::NVega::TCommand::GetParameter(TSensorId sensor) {
    NDrive::NVega::TCommand result(NDrive::NVega::ECommandCode::GET_PARAM);
    NDrive::NVega::TCommandRequest::TGetParameter parameter;
    parameter.Id = sensor.Id;
    parameter.SubId = sensor.SubId;
    result.Argument.Set(parameter);
    return result;
}

NDrive::NVega::TCommand NDrive::NVega::TCommand::SetParameter(TSensorId sensor, TSensorValue value, bool polite) {
    NDrive::NVega::ECommandCode code = polite
        ? NDrive::NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM
        : NDrive::NVega::ECommandCode::SET_PARAM;
    NDrive::NVega::TCommand result(code);
    NDrive::NVega::TCommandRequest::TSetParameter parameter;
    parameter.Id = sensor.Id;
    parameter.SubId = sensor.SubId;
    parameter.SetValue(value);
    result.Argument.Set(parameter);
    return result;
}

namespace {
    const TMap<TString, NDrive::NVega::ECommandCode> Actions = {
        { "open-doors",         NDrive::NVega::ECommandCode::OPEN_DOORS },
        { "close-doors",        NDrive::NVega::ECommandCode::CLOSE_DOORS },
        { "unlock-hood",        NDrive::NVega::ECommandCode::YADRIVE_UNLOCK_HOOD },
        { "lock-hood",          NDrive::NVega::ECommandCode::YADRIVE_LOCK_HOOD },
        { "unlock-doors-hood",  NDrive::NVega::ECommandCode::SCENARIO_UNLOCK_DOORS_AND_HOOD },
        { "lock-doors-hood",    NDrive::NVega::ECommandCode::SCENARIO_LOCK_DOORS_AND_HOOD },
        { "start-lease",        NDrive::NVega::ECommandCode::YADRIVE_START_OF_LEASE },
        { "end-lease",          NDrive::NVega::ECommandCode::YADRIVE_END_OF_LEASE },
        { "start-heating",      NDrive::NVega::ECommandCode::YADRIVE_WARMING },
        { "stop-heating",       NDrive::NVega::ECommandCode::YADRIVE_STOP_WARMING },
        { "blinker-flush",      NDrive::NVega::ECommandCode::BLINKER_FLASH },
        { "blinker-flash",      NDrive::NVega::ECommandCode::BLINKER_FLASH },
        { "blink-n-horn",       NDrive::NVega::ECommandCode::HORN_AND_BLINK },
        { "horn",               NDrive::NVega::ECommandCode::HORN },
        { "panic",              NDrive::NVega::ECommandCode::YADRIVE_PANIC },
        { "panic_3s",           NDrive::NVega::ECommandCode::YADRIVE_PANIC },
        { "panic_5s",           NDrive::NVega::ECommandCode::YADRIVE_PANIC },
        { "panic_10s",          NDrive::NVega::ECommandCode::YADRIVE_PANIC },
        { "nrf_add_relay",      NDrive::NVega::ECommandCode::NRF_ADD_RELAY },
        { "nrf_remove_relay",   NDrive::NVega::ECommandCode::NRF_REMOVE_RELAY },
        { "nrf_add_mark",       NDrive::NVega::ECommandCode::NRF_ADD_MARK },
        { "nrf_remove_mark",    NDrive::NVega::ECommandCode::NRF_REMOVE_MARK },
        { "unlock-charge-connector", NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND },
        { "unlock-charge-port", NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND },
        { "enable-eco",         NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND },
        { "disable-eco",        NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND },
        { "unlock-fuel-cap",    NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "lock-fuel-cap",      NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "unlock-trunk",       NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "lock-trunk",         NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "start-head-unit",    NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "stop-head-unit",     NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND },
        { "switch-sim",         NDrive::NVega::ECommandCode::GSM_MODEM_COMMAND },
        { "ble-reset",          NDrive::NVega::ECommandCode::SCENARIO_BLE_RESET },
        { "polite-restart",     NDrive::NVega::ECommandCode::SCENARIO_POLITE_RESTART },
        { "restart",            NDrive::NVega::ECommandCode::RESTART },
        { "clear-dtc",          NDrive::NVega::ECommandCode::OBD_COMMAND },
        { "update-dtc",         NDrive::NVega::ECommandCode::OBD_COMMAND },
        { "update-vin",         NDrive::NVega::ECommandCode::OBD_COMMAND },
        { "speed-limit-on",     NDrive::NVega::ECommandCode::WINDOWS_OPENING_19S },
        { "speed-limit-off",    NDrive::NVega::ECommandCode::WINDOWS_OPENING_23S },
        { "digital-output-1-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-1-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-2-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-2-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-3-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-3-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-4-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-4-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-5-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-5-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-6-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-6-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-7-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-7-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-8-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-8-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-9-on",    NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-9-off",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-10-on",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-10-off",  NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-11-on",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-11-off",  NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-12-on",   NDrive::NVega::ECommandCode::SET_PARAM },
        { "digital-output-12-off",  NDrive::NVega::ECommandCode::SET_PARAM },
        { "nrf-relay-on",           NDrive::NVega::ECommandCode::SET_PARAM },
        { "nrf-relay-off",          NDrive::NVega::ECommandCode::SET_PARAM },
        { "dtc-clear",              NDrive::NVega::ECommandCode::YADRIVE_DTC_CLEAR},
        { "emergency-stop",         NDrive::NVega::ECommandCode::YADRIVE_EMERGENCY_STOP},
    };

    const TMap<TString, NDrive::NVega::TCommandRequest::TElectricCarCommand::EType> AuxiallaryCommands = {
        { "unlock-charge-connector", NDrive::NVega::TCommandRequest::TElectricCarCommand::UNBLOCK_CHARGE_CONNECTOR },
        { "unlock-charge-port", NDrive::NVega::TCommandRequest::TElectricCarCommand::UNBLOCK_CHARGE_PORT },
        { "enable-eco",         NDrive::NVega::TCommandRequest::TElectricCarCommand::ENABLE_ECO },
        { "disable-eco",        NDrive::NVega::TCommandRequest::TElectricCarCommand::DISABLE_ECO },
        { "unlock-fuel-cap",    NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::UNLOCK_FUEL_CAP },
        { "lock-fuel-cap",      NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::LOCK_FUEL_CAP },
        { "unlock-trunk",       NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::UNLOCK_TRUNK },
        { "lock-trunk",         NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::LOCK_TRUNK },
    };

    const TMap<TString, NDrive::NVega::TCommandRequest::TObdCommand::EType> ObdCommands = {
        { "clear-dtc",          NDrive::NVega::TCommandRequest::TObdCommand::CLEAR_DTC },
        { "update-dtc",         NDrive::NVega::TCommandRequest::TObdCommand::UPDATE_DTC },
        { "update-vin",         NDrive::NVega::TCommandRequest::TObdCommand::UPDATE_VIN },
    };
}

TVector<TStringBuf> NDrive::ListActions() {
    TVector<TStringBuf> result;
    for (auto&& [action, command] : Actions) {
        result.push_back(action);
    }
    return result;
}

TMaybe<NDrive::NVega::ECommandCode> NDrive::ParseAction(TStringBuf action) {
    auto p = Actions.find(action);
    if (p != Actions.end()) {
        return p->second;
    }

    NDrive::NVega::ECommandCode command;
    if (TryFromString(action, command)) {
        return command;
    }

    return {};
}

TMaybe<NDrive::NProtocol::TArgument> NDrive::ParseArgument(TStringBuf action) {
    auto aux = AuxiallaryCommands.find(action);
    if (aux != AuxiallaryCommands.end()) {
        NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand command;
        command.Type = aux->second;
        NDrive::NProtocol::TArgument result;
        result.Set(command);
        return result;
    }

    auto obd = ObdCommands.find(action);
    if (obd != ObdCommands.end()) {
        NDrive::NVega::TCommandRequest::TObdCommand command;
        command.Type = obd->second;
        NDrive::NProtocol::TArgument result;
        result.Set(command);
        return result;
    }

    if (action == "start-head-unit") {
        NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4 command;
        command.Type = NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4::CONTROL_HEAD_UNIT;
        command.TurnOn = 1;
        NDrive::NProtocol::TArgument result;
        result.Set(command);
        return result;
    }

    if (action == "stop-head-unit") {
        NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4 command;
        command.Type = NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4::CONTROL_HEAD_UNIT;
        command.TurnOn = 0;
        NDrive::NProtocol::TArgument result;
        result.Set(command);
        return result;
    }

    if (action == "switch-sim") {
        NDrive::NVega::TCommandRequest::TSwitchSim switchSim;
        NDrive::NProtocol::TArgument result;
        result.Set(switchSim);
        return result;
    }

    constexpr TStringBuf panicPrefix = "panic_";
    if (action.StartsWith(panicPrefix)) {
        NDrive::NVega::TCommandRequest::TPanic panic;
        panic.Type = NDrive::NVega::TCommandRequest::TPanic::HORN_AND_BLINK;
        panic.Time = FromStringWithDefault<TDuration>(action.Skip(panicPrefix.size()), TDuration::Seconds(3)).Seconds();
        NDrive::NProtocol::TArgument result;
        result.Set(panic);
        return result;
    }

    constexpr TStringBuf digitalOutputPrefix = "digital-output-";

    if (action.StartsWith(digitalOutputPrefix)) {
        const size_t actionSplitSize = 2;
        TVector<TStringBuf> actionSplit = StringSplitter(action.Skip(digitalOutputPrefix.size())).Split('-').SkipEmpty();

        Y_ENSURE(actionSplit.size() == actionSplitSize, "incorrect format for digital output command");

        auto indexStr = actionSplit[0];
        auto stateStr = actionSplit[1];

        bool state = IsTrue(stateStr);
        ui16 index = 1;

        if (TryFromString(indexStr, index)) {
            NDrive::NVega::TCommandRequest::TSetParameter setParameter;
            setParameter.Id = NDrive::NVega::DigitalOutputId(index);
            setParameter.SubId = 0;
            setParameter.SetValue<ui8>(state);

            NDrive::NProtocol::TArgument result;
            result.Set(setParameter);
            return result;
        }
    }

    constexpr TStringBuf nrfRelayPrefix = "nrf-relay-";
    if (action.StartsWith(nrfRelayPrefix)) {
        TStringBuf stateStr = action.Skip(nrfRelayPrefix.size());

        if (stateStr) {
            bool state = IsTrue(stateStr);

            NDrive::NVega::TCommandRequest::TSetParameter setParameter;
            setParameter.Id = VEGA_NRF_DESIRED_RELAY_STATE;
            setParameter.SubId = 0;
            setParameter.SetValue<ui8>(state);

            NDrive::NProtocol::TArgument result;
            result.Set(setParameter);
            return result;
        }
    }

    return {};
}

NDrive::NVega::TCommand ParseCommandImpl(const NJson::TJsonValue& value) {
    TMaybe<TString> action;
    if (value.IsString()) {
        action = value.GetStringSafe();
    }
    if (!action) {
        action = NJson::TryFromJson<TString>(value["action"]);
    }
    if (!action) {
        action = NJson::TryFromJson<TString>(value["command"]);
    }
    Y_ENSURE(action, "either action or command field should be present");

    TMaybe<NDrive::NVega::ECommandCode> code = NDrive::ParseAction(*action);
    Y_ENSURE(code, "cannot parse action from " << *action);

    NDrive::NProtocol::TArgument arg = NDrive::ParseArgument(*action).GetOrElse({});
    NDrive::NVega::TCommand result(*code, arg);
    if (!arg.IsNull()) {
        return result;
    }

    if (auto argument = NJson::FromJson<TMaybe<TString>>(value["argument"])) {
        auto decoded = HexDecode(*argument);
        {
            TStringInput si(decoded);
            result.Argument.Load(&si);
        }
    } else switch (result.Code) {
        case NDrive::NVega::ECommandCode::GET_PARAM:
        {
            NDrive::NVega::TCommandRequest::TGetParameter parameter;
            parameter.Id = NJson::FromJson<ui16>(value["id"]);
            parameter.SubId = NJson::FromJson<TMaybe<ui16>>(value["subid"]).GetOrElse(0);
            result.Argument.Set(parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM:
        case NDrive::NVega::ECommandCode::SET_PARAM:
        {
            NDrive::NVega::TCommandRequest::TSetParameter parameter;
            parameter.Id = NJson::FromJson<ui16>(value["id"]);
            parameter.SubId = NJson::FromJson<TMaybe<ui16>>(value["subid"]).GetOrElse(0);
            parameter.SetValue(NDrive::SensorValueFromJson(value["value"], parameter.Id));
            result.Argument.Set(parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::BLINKER_FLASH:
        {
            NDrive::NVega::TCommandRequest::TBlinkerFlash parameter;

            auto timeParameter = NJson::FromJson<TMaybe<ui16>>(value["time"]);
            auto samplingParameter = NJson::FromJson<TMaybe<ui16>>(value["sampling"]);

            if (timeParameter) {
                parameter.Time = *timeParameter;
            }

            if (samplingParameter) {
                parameter.Sampling = *samplingParameter;
            }

            if (timeParameter || samplingParameter) {
                result.Argument.Set(parameter);
            }
            break;
        }
        case NDrive::NVega::ECommandCode::YADRIVE_PANIC:
        {
            NDrive::NVega::TCommandRequest::TPanic panic;
            panic.Time = NJson::FromJson<ui8>(value["time"]);
            panic.Type = NJson::FromJson<ui8>(value["type"]);
            result.Argument.Set(panic);
            break;
        }
        case NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND:
        {
            const auto& type = value["type"];
            auto type1 = NJson::TryFromJson<NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::EType>(type);
            auto type4 = NJson::TryFromJson<NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4::EType>(type);
            if (type1) {
                NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand command;
                command.Type = *type1;
                result.Argument.Set(command);
            } else if (type4) {
                NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4 command;
                command.Type = *type4;
                switch (command.Type) {
                case NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4::CONTROL_HEAD_UNIT:
                    command.TurnOn = NJson::FromJson<TMaybe<bool>>(value["turn_on"]).GetOrElse(false) ? 1 : 0;
                    command.Duration = NJson::FromJson<TMaybe<ui16>>(value["duration"]).GetOrElse(command.Duration);
                    break;
                }
                result.Argument.Set(command);
            } else {
                ythrow yexception() << "cannot parse type of AUXILIARY_CAR_COMMAND: " << type.GetStringRobust();
            }
            break;
        }
        case NDrive::NVega::ECommandCode::YADRIVE_WARMING:
        {
            auto time = NJson::FromJson<TMaybe<ui8>>(value["time"]);
            if (time) {
                NDrive::NVega::TCommandRequest::TWarming warming;
                warming.Time = *time;
                result.Argument.Set(warming);
            }
            break;
        }
        case NDrive::NVega::ECommandCode::YADRIVE_AUDIO_COMMAND:
        {
            auto type = NJson::FromJson<ui8>(value["type"]);
            if (type == 0) {
                NDrive::NVega::TCommandRequest::TAudioFile file;
                file.FileId = NJson::FromJson<ui16>(value["id"]);
                result.Argument.Set(file);
            }
            if (type == 1) {
                NDrive::NVega::TCommandRequest::TAudioScenario scenario;
                scenario.ScenarioId = NJson::FromJson<ui8>(value["id"]);
                result.Argument.Set(scenario);
            }
            break;
        }
        case NDrive::NVega::ECommandCode::NRF_ADD_MARK:
        case NDrive::NVega::ECommandCode::NRF_REMOVE_MARK:
        {
            NDrive::NVega::TCommandRequest::TMark mark;
            if (auto id = NJson::FromJson<TMaybe<ui8>>(value["id"])) {
                mark.Id = *id;
            }
            result.Argument.Set(mark);
            break;
        }
        case NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG:
        {
            NDrive::NVega::TCommandRequest::TCustomObdForwardConfig config;
            auto duration = NJson::FromJson<TMaybe<TDuration>>(value["duration"]).GetOrElse(TDuration::Seconds(10));
            Y_ENSURE(duration.Seconds() < Max<decltype(config.Duration)>());
            config.Duration = duration.Seconds();
            NJson::ReadField(value["mask"], config.Mask);
            NJson::ReadField(value["value"], config.Value);
            result.Argument.Set(config);
            break;
        }
        case NDrive::NVega::ECommandCode::FAST_DATA_CONFIG:
        {
            auto discretization = NJson::FromJson<TMaybe<TDuration>>(value["discretization"]).GetOrElse(TDuration::MilliSeconds(100));
            auto duration = NJson::FromJson<TMaybe<TDuration>>(value["duration"]).GetOrElse(TDuration::Seconds(100));
            NDrive::NVega::TCommandRequest::TFastDataConfig config;
            config.SetDiscretization(discretization);
            config.SetDuration(duration);
            result.Argument.Set(config);
            break;
        }
        case NDrive::NVega::ECommandCode::MOVE_TO_COORD:
        {
            auto latitude = NJson::FromJson<double>(value["latitude"]);
            auto longitude = NJson::FromJson<double>(value["longitude"]);
            auto speed = NJson::FromJson<TMaybe<double>>(value["speed"]);
            NDrive::NVega::TCommandRequest::TMoveToCoordParameter parameter;
            parameter.Lat = latitude;
            parameter.Lon = longitude;
            if (speed) {
                parameter.Speed = *speed;
            }
            result.Argument.Set(parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::SCENARIO_QUERY_FUEL_LEVEL:
        {
            auto ignitionDuration = NJson::FromJson<TMaybe<TDuration>>(value["ignition_duration"]).GetOrElse(TDuration::Seconds(30));
            NDrive::NVega::TCommandRequest::TQueryFuelLevelArgs parameter;
            parameter.SetIgnitionDuration(ignitionDuration);
            result.Argument.Set(parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::SCENARIO_OBD_REQUEST:
        {
            NDrive::NVega::TCommandRequest::TObdRequest request;
            request.Id = NJson::FromJson<ui32>(value["id"]);
            request.Index = NJson::FromJson<ui8>(value["index"]);
            auto data = NJson::FromJson<TVector<ui8>>(value["data"]);
            request.Data = { data.begin(), data.end() };
            auto responseId = NJson::FromJson<TMaybe<ui32>>(value["response_id"]);
            if (responseId) {
                request.ResponseId = *responseId;
            }
            result.Argument.Set(request);
            break;
        }
        case NDrive::NVega::ECommandCode::CAN_SCRIPT:
        {
            NDrive::NVega::TCommandRequest::TCanScript request;
            request.Id = NJson::FromJson<ui8>(value["id"]);
            result.Argument.Set(request);
            break;
        }
        default:
            break;
    }
    return result;
}

TExpected<NDrive::NVega::TCommand, yexception> NDrive::ParseCommand(const NJson::TJsonValue& value) {
    return WrapUnexpected<yexception>(ParseCommandImpl, value);
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TCommand>(const NDrive::NVega::TCommand& object) {
    NJson::TJsonValue result;
    result["command"] = ToString(object.Code);
    if (!object.Argument.IsNull()) {
        TStringStream ss;
        object.Argument.Save(&ss);
        result["argument"] = HexEncode(ss.Str());
    }
    return result;
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TCommand>(const NJson::TJsonValue& value, NDrive::NVega::TCommand& result) {
    auto expectedResult = NDrive::ParseCommand(value);
    if (expectedResult) {
        result = expectedResult.ExtractValue();
        return true;
    } else {
        return false;
    }
}
