#include "calibration.h"

#include <rtline/library/json/cast.h>

#include <library/cpp/json/json_reader.h>

double NDrive::NVega::TSensorCalibration::GetMaxCalibrated() const {
    auto maxElement = std::max_element(Table.begin(), Table.end());
    Y_ENSURE(maxElement != Table.end());
    return maxElement->Calibrated;
}

double NDrive::NVega::TSensorCalibration::Get(const TSensor& original) const {
    return Get(original.ConvertTo<double>());
}

double NDrive::NVega::TSensorCalibration::Get(double original) const {
    Y_ASSERT(std::is_sorted(Table.begin(), Table.end()));
    if (Table.empty()) {
        return 0;
    }
    const TElement& last = Table.back();
    if (original >= last.Original) {
        return last.Calibrated + last.Slope * (original - last.Original);
    }
    const TElement& first = Table.front();
    if (original <= first.Original) {
        return first.Calibrated + first.Slope * (original - first.Original);
    }
    auto p = std::lower_bound(Table.begin(), Table.end(), original);
    Y_ASSERT(p != Table.begin());
    Y_ASSERT(p != Table.end());
    p--;
    return p->Calibrated + p->Slope * (original - p->Original);
}

void NDrive::NVega::TSensorCalibration::SetTable(TTable&& value) {
    Table = std::move(value);
    Finalize();
}

void NDrive::NVega::TSensorCalibration::Finalize() {
    std::sort(Table.begin(), Table.end());
    for (size_t i = 0; (i + 1) < Table.size(); ++i) {
        TElement& current = Table[i];
        const TElement& next = Table[i + 1];
        current.Slope = (next.Calibrated - current.Calibrated) / std::max(next.Original - current.Original, Precision);
    }
    if (Table.size() >= 2) {
        TElement& current = Table.back();
        const TElement& previous = Table[Table.size() - 2];
        current.Slope = previous.Slope;
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TSensorCalibration::TElement& object) {
    NJson::TJsonValue result;
    result["original"] = object.Original;
    result["calibrated"] = object.Calibrated;
    return result;
}

template <>
bool NJson::TryFromJson(const TJsonValue& value, NDrive::NVega::TSensorCalibration::TElement& result) {
    return
        TryFromJson(value["original"], result.Original) &&
        TryFromJson(value["calibrated"], result.Calibrated);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TSensorCalibration& object) {
    NJson::TJsonValue result;
    result["table"] = ToJson(object.GetTable());
    return result;
}

template <>
bool NJson::TryFromJson(const TJsonValue& value, NDrive::NVega::TSensorCalibration& result) {
    NDrive::NVega::TSensorCalibration::TTable table;
    if (!TryFromJson(value["table"], table)) {
        return false;
    }
    result.SetTable(std::move(table));
    return true;
}

template <>
bool TryFromStringImpl(const char* data, size_t size, NDrive::NVega::TSensorCalibration& result) {
    NJson::TJsonValue value;
    TStringBuf s(data, size);
    return NJson::ReadJsonFastTree(s, &value) && NJson::TryFromJson(value, result);
}
