#pragma once

#include <util/generic/vector.h>
#include <util/ysaveload.h>

#include <numeric>
#include <tuple>

namespace NPrivate {
    template <class T, class H>
    class TVectorWithHeader: public ::TVector<T> {
    public:
        using TBase = TVector<T>;
        using TBase::TBase;
    };

    template <class T, class S>
    class TVectorWithTypedSize: public TVectorWithHeader<T, S> {
    public:
        using TBase = TVectorWithHeader<T, S>;
        using TBase::TBase;
    };

    template <class T, class L>
    class TVectorWithTypedContentLength: public TVectorWithHeader<T, L> {
    public:
        using TBase = TVectorWithHeader<T, L>;
        using TBase::TBase;
    };

    template <class T>
    class TSizer {
    private:
        template <class C>
        static typename std::enable_if<std::is_pod<C>::value, size_t>::type SizeOfImpl(const C& v) {
            return sizeof(v);
        }

        template <class C>
        static typename std::enable_if<!std::is_pod<C>::value, size_t>::type SizeOfImpl(const C& v) {
            return v.CalcSize();
        }

    public:
        static size_t SizeOf(const T& v) {
            return SizeOfImpl(v);
        }
    };
    template <class T, class H>
    class TSizer<TVectorWithHeader<T, H>> {
    public:
        static size_t SizeOf(const TVectorWithHeader<T, H>& v) {
            const size_t header = sizeof(H);
            const size_t data = std::accumulate(v.begin(), v.end(), size_t(0), [](size_t size, const T& element) {
                return size + TSizer<T>::SizeOf(element);
            });
            return header + data;
        }
    };
    template <class T, class S>
    class TSizer<TVectorWithTypedSize<T, S>>: public TSizer<TVectorWithHeader<T, S>> {
    };
    template <class T, class L>
    class TSizer<TVectorWithTypedContentLength<T, L>>: public TSizer<TVectorWithHeader<T, L>> {
    };

    template <class T>
    size_t SizeOf(const T& v) {
        return TSizer<T>::SizeOf(v);
    }
    template <class T>
    size_t SizeMany(const T& v) {
        return SizeOf(v);
    }
    template <class T, class... TArgs>
    size_t SizeMany(const T& v, const TArgs&... args) {
        return SizeOf(v) + SizeMany(args...);
    }

#define DEFINE_FIELDS(...) \
    inline size_t CalcSize() const { return ::NPrivate::SizeMany(__VA_ARGS__); } \
    inline auto Tuple() const { return std::tie(__VA_ARGS__); } \
    inline auto Tuple() { return std::tie(__VA_ARGS__); } \
    template <class TOther> inline bool operator==(const TOther& other) const { return Tuple() == other.Tuple(); } \
    Y_PASS_VA_ARGS(Y_SAVELOAD_DEFINE(__VA_ARGS__));

#define DEFINE_MESSAGE_FIELDS(...) \
    virtual size_t GetSize() const override { return CalcSize(); } \
    virtual void Load(IInputStream& stream) override { Load(&stream); } \
    virtual void Save(IOutputStream& stream) const override { Save(&stream); } \
    Y_PASS_VA_ARGS(DEFINE_FIELDS(__VA_ARGS__));

#define NO_MESSAGE_FIELDS() \
    virtual size_t GetSize() const override { return 0; } \
    virtual void Load(IInputStream&) override {} \
    virtual void Save(IOutputStream&) const override {} \
    inline auto Tuple() const { return std::tuple<>{}; } \
    inline auto Tuple() { return std::tuple<>{}; } \
    template <class TOther> inline bool operator==(const TOther& other) const { return Tuple() == other.Tuple(); }
}

template <class T, class H>
class TSerializer<NPrivate::TVectorWithTypedSize<T, H>> {
public:
    using TType = NPrivate::TVectorWithTypedSize<T, H>;

public:
    static inline void Save(IOutputStream* stream, const TType& value) {
        const auto size = static_cast<H>(value.size());
        Y_ENSURE(size == value.size());
        ::Save(stream, size);
        ::SaveRange(stream, value.begin(), value.end());
    }
    static inline void Load(IInputStream* stream, TType& value) {
        H size = 0;
        ::Load(stream, size);
        value.resize(size);
        ::LoadRange(stream, value.begin(), value.end());
    }
};

template <class T, class L>
class TSerializer<NPrivate::TVectorWithTypedContentLength<T, L>> {
public:
    using TType = NPrivate::TVectorWithTypedContentLength<T, L>;

public:
    static inline void Save(IOutputStream* stream, const TType& value) {
        const auto content = NPrivate::SizeOf(value) - sizeof(L);
        const auto length = static_cast<L>(content);
        Y_ENSURE(length == content);
        ::Save(stream, length);
        ::SaveRange(stream, value.begin(), value.end());
    }
    static inline void Load(IInputStream* stream, TType& value) {
        L length = 0;
        ::Load(stream, length);
        for (size_t read = 0; read < length; read += NPrivate::SizeOf(value.back())) {
            ::Load(stream, value.emplace_back());
        }
    }
};
