#include "errors.h"

#include <util/generic/map.h>
#include <util/string/cast.h>

namespace {
    constexpr TStringBuf breakage = "break, ";

    const TMap<TString, NDrive::ETelematicsNotification> Errors = {
        { "driver door is open", NDrive::ETelematicsNotification::DriverDoorOpen },
        { "roof is open", NDrive::ETelematicsNotification::RoofIsOpen },
        { "roof is open (sensor 1)", NDrive::ETelematicsNotification::RoofIsOpen2 },
        { "roof is open (sensor 2)", NDrive::ETelematicsNotification::RoofIsOpen2 },
        { "left roof lock is open", NDrive::ETelematicsNotification::LeftRoofLockIsOpen },
        { "right roof lock is open", NDrive::ETelematicsNotification::RightRoofLockIsOpen },
        { "driver's window not closed", NDrive::ETelematicsNotification::DriverWindowOpen },
        { "passenger's window not closed", NDrive::ETelematicsNotification::PassengerWindowOpen },
        { "driver's window leaf not closed", NDrive::ETelematicsNotification::DriverWindowLeafOpen },
        { "passenger's window leaf not closed", NDrive::ETelematicsNotification::PassengerWindowLeafOpen },
        { "no fuel cap", NDrive::ETelematicsNotification::NoFuelCap },
        { "external board err", NDrive::ETelematicsNotification::ExternalBoardError },
        { "pass door is open", NDrive::ETelematicsNotification::PassengerDoorOpen },
        { "left rear door is open", NDrive::ETelematicsNotification::LeftRearDoorOpen },
        { "right rear door is open", NDrive::ETelematicsNotification::RightRearDoorOpen },
        { "hood door is open", NDrive::ETelematicsNotification::HoodOpen },
        { "trunk door is open", NDrive::ETelematicsNotification::TrunkOpen },
        { "wrong key position", NDrive::ETelematicsNotification::WrongKeyPosition },
        { "wrong selector position", NDrive::ETelematicsNotification::WrongSelectorPosition },
        { "light is on", NDrive::ETelematicsNotification::LightsOn },
        { "engine run fail", NDrive::ETelematicsNotification::EngineRunFailure },
        { "wrong light switch position", NDrive::ETelematicsNotification::WrongLightSwitchPosition },
        { "windows not closed", NDrive::ETelematicsNotification::WindowsNotClosed },
        { "door not locked", NDrive::ETelematicsNotification::DoorsNotLocked },
        { "doors not locked", NDrive::ETelematicsNotification::DoorsNotLocked },
        { "handbreak is't applied", NDrive::ETelematicsNotification::WrongHandbrakePosition },
        { "left rear window not closed", NDrive::ETelematicsNotification::LeftRearWindowOpen },
        { "right rear window not closed", NDrive::ETelematicsNotification::RightRearWindowOpen },
        { "broadcast finished: connection_not_found", NDrive::ETelematicsNotification::ConnectionNotFound },
        { "null response", NDrive::ETelematicsNotification::WaitTimeout },
        { "wrong position of the transfer case selector", NDrive::ETelematicsNotification::WrongTransferCaseSelectorPosition },
        { "key not in acc", NDrive::ETelematicsNotification::KeyIsNotInIgnition },
    };

    TString GetUserErrorDescription_ENG(NDrive::ETelematicsNotification value) {
        switch (value) {
        case NDrive::ETelematicsNotification::ConnectionNotFound:
        case NDrive::ETelematicsNotification::WaitTimeout:
        case NDrive::ETelematicsNotification::Unknown:
            return "Unknown server error";
        case NDrive::ETelematicsNotification::DriverDoorOpen:
            return "Driver door is open";
        case NDrive::ETelematicsNotification::PassengerDoorOpen:
            return "Passenger door is open";
        case NDrive::ETelematicsNotification::LeftRearDoorOpen:
            return "Left rear door is open";
        case NDrive::ETelematicsNotification::RightRearDoorOpen:
            return "Right rear door is open";
        case NDrive::ETelematicsNotification::HoodOpen:
            return "The bonnet is open";
        case NDrive::ETelematicsNotification::TrunkOpen:
            return "The trunk is open";
        case NDrive::ETelematicsNotification::EngineOn:
            return "The engine is on";
        case NDrive::ETelematicsNotification::WrongKeyPosition:
            return "The key is in wrong position";
        case NDrive::ETelematicsNotification::WrongSelectorPosition:
            return "The transmission selector is not in parking";
        case NDrive::ETelematicsNotification::IgnitionOn:
            return "The ignition is on";
        case NDrive::ETelematicsNotification::LightsOn:
            return "Headlights are on";
        case NDrive::ETelematicsNotification::EngineRunning:
            return "The engine is running";
        case NDrive::ETelematicsNotification::EngineRunFailure:
            return "The engine has failed to start";
        case NDrive::ETelematicsNotification::WrongLightSwitchPosition:
            return "Lights switch is on";
        case NDrive::ETelematicsNotification::WindowsNotClosed:
            return "Windows are not closed";
        case NDrive::ETelematicsNotification::DoorsNotLocked:
            return "Doors are not locked";
        case NDrive::ETelematicsNotification::WrongHandbrakePosition:
            return "Handbrake is not applied";
        case NDrive::ETelematicsNotification::DriverWindowOpen:
            return "Driver's window not closed";
        case NDrive::ETelematicsNotification::PassengerWindowOpen:
            return "Passenger's window not closed";
        case NDrive::ETelematicsNotification::DriverWindowLeafOpen:
            return "Driver's window leaf not closed";
        case NDrive::ETelematicsNotification::PassengerWindowLeafOpen:
            return "Passenger's window leaf not closed";
        case NDrive::ETelematicsNotification::NoFuelCap:
            return "No fuel cap";
        case NDrive::ETelematicsNotification::ExternalBoardError:
            return "External board error";
        case NDrive::ETelematicsNotification::RoofIsOpen:
        case NDrive::ETelematicsNotification::RoofIsOpen2:
            return "Roof is open";
        case NDrive::ETelematicsNotification::LeftRoofLockIsOpen:
            return "Left roof lock is open";
        case NDrive::ETelematicsNotification::RightRoofLockIsOpen:
            return "Right roof lock is open";
        case NDrive::ETelematicsNotification::LeftRearWindowOpen:
            return "Left rear window is open";
        case NDrive::ETelematicsNotification::RightRearWindowOpen:
            return "Right rear window is open";
        case NDrive::ETelematicsNotification::WrongTransferCaseSelectorPosition:
            return "WrongTransferCaseSelectorPosition";
        case NDrive::ETelematicsNotification::KeyIsNotInIgnition:
            return "Key is not in ingnition";
        }
    }
    TString GetUserErrorDescription_RUS(NDrive::ETelematicsNotification value) {
        switch (value) {
        case NDrive::ETelematicsNotification::ConnectionNotFound:
            return "Машина не отвечает\nПозвоните нам в поддержку, разберёмся";
        case NDrive::ETelematicsNotification::WaitTimeout:
            return "Машина не отвечает\nПопробуйте ещё раз через минуту. Если и тогда нет — звоните нам, разберёмся";
        case NDrive::ETelematicsNotification::Unknown:
            return "Неопределенные проблемы с автомобилем";
        case NDrive::ETelematicsNotification::DriverDoorOpen:
            return "Не закрыта дверь водителя";
        case NDrive::ETelematicsNotification::PassengerDoorOpen:
            return "Не закрыта передняя правая дверь";
        case NDrive::ETelematicsNotification::LeftRearDoorOpen:
            return "Не закрыта задняя левая дверь";
        case NDrive::ETelematicsNotification::RightRearDoorOpen:
            return "Не закрыта задняя правая дверь";
        case NDrive::ETelematicsNotification::HoodOpen:
            return "Капот не закрыт";
        case NDrive::ETelematicsNotification::TrunkOpen:
            return "Багажник не закрыт";
        case NDrive::ETelematicsNotification::EngineOn:
            return "Двигатель не заглушен";
        case NDrive::ETelematicsNotification::WrongKeyPosition:
            return "Ключ зажигания в неправильной позиции";
        case NDrive::ETelematicsNotification::WrongSelectorPosition:
            return "Селектор трансмиссии не в положении \"P\"";
        case NDrive::ETelematicsNotification::IgnitionOn:
            return "Зажигание на выключено ";
        case NDrive::ETelematicsNotification::LightsOn:
            return "Фары не выключены";
        case NDrive::ETelematicsNotification::EngineRunning:
            return "Двигатель запущен";
        case NDrive::ETelematicsNotification::EngineRunFailure:
            return "Не удалось запустить двигатель";
        case NDrive::ETelematicsNotification::WrongLightSwitchPosition:
            return "Для блокировки машины переведите фары в режим «Auto»";
        case NDrive::ETelematicsNotification::WindowsNotClosed:
            return "Не удалось поднять стекла";
        case NDrive::ETelematicsNotification::DoorsNotLocked:
            return "Не удалось заблокировать двери";
        case NDrive::ETelematicsNotification::WrongHandbrakePosition:
            return "Ручник не поднят";
        case NDrive::ETelematicsNotification::DriverWindowOpen:
            return "Не закрыто окно водителя";
        case NDrive::ETelematicsNotification::PassengerWindowOpen:
            return "Не закрыто переднее правое окно";
        case NDrive::ETelematicsNotification::DriverWindowLeafOpen:
            return "Форточка водителя не закрыта";
        case NDrive::ETelematicsNotification::PassengerWindowLeafOpen:
            return "Форточка пассажира не закрыта";
        case NDrive::ETelematicsNotification::NoFuelCap:
            return "Нет крышки бензобака";
        case NDrive::ETelematicsNotification::ExternalBoardError:
            return "Нет связи с блоком расширения, невозможно опросить входы (неисправность)";
        case NDrive::ETelematicsNotification::RoofIsOpen:
        case NDrive::ETelematicsNotification::RoofIsOpen2:
            return "Крыша не закрыта";
        case NDrive::ETelematicsNotification::LeftRoofLockIsOpen:
            return "Левый замок крыши не закрыт";
        case NDrive::ETelematicsNotification::RightRoofLockIsOpen:
            return "Правый замок крыши не закрыт";
        case NDrive::ETelematicsNotification::LeftRearWindowOpen:
            return "Не закрыто заднее левое окно";
        case NDrive::ETelematicsNotification::RightRearWindowOpen:
            return "Не закрыто заднее правое окно";
        case NDrive::ETelematicsNotification::WrongTransferCaseSelectorPosition:
            return "Неправильное положение дифференциала";
        case NDrive::ETelematicsNotification::KeyIsNotInIgnition:
            return "Ключ не в замке зажигания";
        }
    }
}

bool NDrive::IsError(TStringBuf text) {
    switch (ParseError(text)) {
        case ETelematicsNotification::DriverDoorOpen:
        case ETelematicsNotification::PassengerDoorOpen:
        case ETelematicsNotification::LeftRearDoorOpen:
        case ETelematicsNotification::RightRearDoorOpen:
        case ETelematicsNotification::HoodOpen:
        case ETelematicsNotification::TrunkOpen:
        case ETelematicsNotification::EngineOn:
        case ETelematicsNotification::WrongKeyPosition:
        case ETelematicsNotification::WrongSelectorPosition:
        case ETelematicsNotification::IgnitionOn:
        case ETelematicsNotification::LightsOn:
        case ETelematicsNotification::EngineRunFailure:
        case ETelematicsNotification::WrongLightSwitchPosition:
        case ETelematicsNotification::WindowsNotClosed:
        case ETelematicsNotification::DoorsNotLocked:
        case ETelematicsNotification::WrongHandbrakePosition:
        case ETelematicsNotification::DriverWindowOpen:
        case ETelematicsNotification::PassengerWindowOpen:
        case ETelematicsNotification::DriverWindowLeafOpen:
        case ETelematicsNotification::PassengerWindowLeafOpen:
        case ETelematicsNotification::NoFuelCap:
        case ETelematicsNotification::ExternalBoardError:
        case ETelematicsNotification::RoofIsOpen:
        case ETelematicsNotification::RoofIsOpen2:
        case ETelematicsNotification::LeftRoofLockIsOpen:
        case ETelematicsNotification::RightRoofLockIsOpen:
        case ETelematicsNotification::LeftRearWindowOpen:
        case ETelematicsNotification::RightRearWindowOpen:
        case ETelematicsNotification::ConnectionNotFound:
        case ETelematicsNotification::WaitTimeout:
        case ETelematicsNotification::WrongTransferCaseSelectorPosition:
        case ETelematicsNotification::KeyIsNotInIgnition:
            return true;
        case ETelematicsNotification::EngineRunning:
            return false;
        case ETelematicsNotification::Unknown:
            return text.StartsWith(breakage);
    }
}

NDrive::ETelematicsNotification NDrive::ParseError(TStringBuf text) {
    bool interruption = text.StartsWith(breakage);
    if (interruption) {
        text.Skip(breakage.size());
    }
    if (auto p = Errors.find(text); p != Errors.end()) {
        return p->second;
    }
    if (text == "engine is on") {
        if (interruption) {
            return ETelematicsNotification::EngineOn;
        } else {
            return ETelematicsNotification::EngineRunning;
        }
    }
    return ETelematicsNotification::Unknown;
}

TString NDrive::GetErrorDescription(ETelematicsNotification value) {
    return ToString(value);
}

TString NDrive::GetUserErrorDescription(ETelematicsNotification value, ELanguage language) {
    switch (language) {
    case LANG_RUS:
    case LANG_BASIC_RUS:
    case LANG_UNK:
        return GetUserErrorDescription_RUS(value);
    default:
        return GetUserErrorDescription_ENG(value);
    }
}
