#pragma once

#include <util/stream/buffer.h>
#include <util/stream/mem.h>
#include <util/stream/output.h>
#include <util/stream/str.h>

class IInputStream;

template <class TMessage>
class TMessageInput {
public:
    TMessageInput(IInputStream& input)
        : Input(input)
    {
    }

    void Get(TMessage& message) {
        message.Load(Input);
    }

private:
    IInputStream& Input;
};

template <class TMessage>
class TMessageOutput {
public:
    TMessageOutput(IOutputStream& output)
        : Output(output)
    {
    }

    void Send(const TMessage& message) {
        message.Save(Output);
        Output.Flush();
    }

private:
    IOutputStream& Output;
};

class TFlushBufferedOutputStream: public IOutputStream {
public:
    TFlushBufferedOutputStream(IOutputStream& output, size_t flushSize);

protected:
    virtual void DoWrite(const void* buf, size_t len) override;
    virtual void DoFlush() override;

private:
    IOutputStream& Output;
    const size_t FlushSize;
};

template <class T, class TData>
void LoadFrom(const TData& data, T& result) {
    TMemoryInput input(data.data(), data.size());
    result.Load(&input);
}
template <class T, class TData>
T LoadFrom(const TData& data) {
    T result;
    LoadFrom(data, result);
    return result;
}

template <class T>
void SaveTo(const T& object, TBuffer& result) {
    TBufferOutput output(result);
    object.Save(&output);
}
template <class T>
void SaveTo(const T& object, TString& result) {
    TStringOutput output(result);
    object.Save(&output);
}
template <class TData, class T>
TData SaveTo(const T& object) {
    TData result;
    SaveTo(object, result);
    return result;
}
