#include "crc.h"
#include "navtelecom.h"
#include "vega.h"

#include <rtline/library/json/cast.h>

#include <util/generic/algorithm.h>
#include <util/generic/serialized_enum.h>
#include <util/string/hex.h>

namespace {
    class TFlexInfo {
    public:
        using TMessageType = NDrive::NNavTelecom::EMessageType;

    public:
        TFlexInfo() = default;
        TFlexInfo(TMessageType request, TMessageType answer)
            : Request(request)
            , Answer(answer)
        {
        }

        NDrive::NNavTelecom::EMessageType GetRequest() const {
            return Request;
        }
        NDrive::NNavTelecom::EMessageType GetAnswer() const {
            return Answer;
        }

    private:
        TMessageType Request = TMessageType::MT_INCORRECT;
        TMessageType Answer = TMessageType::MT_INCORRECT;
    };

    class TPreambleParser {
    public:
        using TItem = std::pair<TStringBuf, NDrive::NNavTelecom::EMessageType>;
        using TReverseItem = std::pair<NDrive::NNavTelecom::EMessageType, TStringBuf>;

    public:
        TPreambleParser() {
            Reverse();
        }

        NDrive::NNavTelecom::EMessageType Parse(TStringBuf preamble) const {
            Y_ENSURE(!preamble.Empty());

            auto comparator = [&preamble] (const TItem& item) {
                return item.first.StartsWith(preamble) && !item.first.Empty();
            };
            auto iter = FindIf(Value.begin(), Value.end(), comparator);

            if (iter != Value.end()) {
                return iter->second;
            }
            return NDrive::NNavTelecom::MT_INCORRECT;
        }

        size_t GetSize(NDrive::NNavTelecom::EMessageType messageType) {
            auto comparator = [&messageType] (const TReverseItem& item) {
                return item.first == messageType;
            };
            auto iter = FindIf(ReverseValue.begin(), ReverseValue.end(), comparator);

            if (iter != ReverseValue.end()) {
                return iter->second.Size();
            }
            return 0;
        }

        TStringBuf GetPreamble(NDrive::NNavTelecom::EMessageType messageType) {
            auto comparator = [&messageType] (const TReverseItem& item) {
                return item.first == messageType;
            };
            auto iter = FindIf(ReverseValue.begin(), ReverseValue.end(), comparator);

            if (iter != ReverseValue.end()) {
                return iter->second;
            }
            return nullptr;
        }

        size_t GetPreambleSize(NDrive::NNavTelecom::EPreambleType preambleType) {
            switch (preambleType) {
            case NDrive::NNavTelecom::PT_NTCB:
                return NTCBPreamble.Size();
            case NDrive::NNavTelecom::PT_FLEX:
                return 2;
            case NDrive::NNavTelecom::PT_PING:
                return 1;
            default:
                return 0;
            }
        }

        const TString& GetNTCBPreamble() const {
            return NTCBPreamble;
        }

        TFlexInfo GetFlexInfo(std::string_view preamble) const {
            return FlexInfo.Value(preamble, TFlexInfo());
        }

    private:
        TVector<TReverseItem> ReverseValue;
        const TVector<TItem> Value = {
            { "", NDrive::NNavTelecom::MT_INCORRECT },
            { "*>S:", NDrive::NNavTelecom::MT_HANDSHAKE_REQUEST },
            { "*<S", NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER },
            { "*>FLEX", NDrive::NNavTelecom::MT_PROTOCOL_SETTING_REQUEST },
            { "*<FLEX", NDrive::NNavTelecom::MT_PROTOCOL_SETTING_ANSWER },
            { "~A", NDrive::NNavTelecom::MT_BLACKBOX_REQUEST },
            { "~A", NDrive::NNavTelecom::MT_BLACKBOX_ANSWER },
            { "~E", NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST },
            { "~E", NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_ANSWER },
            { "~T", NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST },
            { "~T", NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_ANSWER },
            { "~X", NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST },
            { "~X", NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER },
            { "~C", NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST },
            { "~C", NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_ANSWER },
            { "*!", NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_COMMAND },
            { "@C", NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_ANSWER },
            { "*?ICCID", NDrive::NNavTelecom::MT_ICCID_REQUEST },
            { "*#ICCID:", NDrive::NNavTelecom::MT_ICCID_ANSWER },
        };

        const TString NTCBPreamble = "@NTC";

        const TMap<std::string_view, TFlexInfo> FlexInfo = {
            { "~A", { NDrive::NNavTelecom::MT_BLACKBOX_REQUEST, NDrive::NNavTelecom::MT_BLACKBOX_ANSWER } },
            { "~E", { NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST, NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_ANSWER } },
            { "~T", { NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST, NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_ANSWER } },
            { "~X", { NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST, NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER } },
            { "~C", { NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST, NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_ANSWER } },
        };

    private:
        void Reverse() {
            for (auto&& item : Value) {
                ReverseValue.push_back({item.second, item.first});
            }
        }
    };

    THolder<NDrive::NProtocol::IPayload> CreatePayload(NDrive::NNavTelecom::EMessageType messageType, const NDrive::NNavTelecom::TBitField& bitField) {
        using namespace NDrive::NProtocol;

        switch(messageType) {
        case NDrive::NNavTelecom::MT_HANDSHAKE_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::THandShakeRequest>();
        case NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::THandShakeAnswer>();
        case NDrive::NNavTelecom::MT_PROTOCOL_SETTING_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TProtocolSettingRequest>();
        case NDrive::NNavTelecom::MT_PROTOCOL_SETTING_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TProtocolSettingAnswer>();
        case NDrive::NNavTelecom::MT_BLACKBOX_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TBlackBoxRequest>(bitField);
        case NDrive::NNavTelecom::MT_BLACKBOX_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TBlackBoxAnswer>();
        case NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TAdditionalBlackBoxRequest>();
        case NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TAdditionalBlackBoxAnswer>();
        case NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TSingleBlackBoxRequest>(bitField);
        case NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TSingleBlackBoxAnswer>();
        case NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TAdditionalSingleBlackBoxRequest>();
        case NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TAdditionalSingleBlackBoxAnswer>();
        case NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TOnlineBlackBoxRequest>(bitField);
        case NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TOnlineBlackBoxAnswer>();
        case NDrive::NNavTelecom::MT_PING:
            return IPayload::Create<NDrive::NNavTelecom::TPing>();
        case NDrive::NNavTelecom::MT_ICCID_REQUEST:
            return IPayload::Create<NDrive::NNavTelecom::TICCIDRequest>();
        case NDrive::NNavTelecom::MT_ICCID_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TICCIDAnswer>();
        case NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_COMMAND:
            return IPayload::Create<NDrive::NNavTelecom::TDigitalOutputCommandRequest>();
        case NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_ANSWER:
            return IPayload::Create<NDrive::NNavTelecom::TDigitalOutputCommandAnswer>(bitField);
        default:
            return nullptr;
        };
    }

    NDrive::NNavTelecom::EPreambleType GetPreambleType(NDrive::NNavTelecom::EMessageType messageType) {
        switch (messageType) {
        case NDrive::NNavTelecom::MT_HANDSHAKE_REQUEST:
        case NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER:
        case NDrive::NNavTelecom::MT_PROTOCOL_SETTING_REQUEST:
        case NDrive::NNavTelecom::MT_PROTOCOL_SETTING_ANSWER:
        case NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_ANSWER:
        case NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_COMMAND:
        case NDrive::NNavTelecom::MT_ICCID_REQUEST:
        case NDrive::NNavTelecom::MT_ICCID_ANSWER:
            return NDrive::NNavTelecom::PT_NTCB;
        case NDrive::NNavTelecom::MT_BLACKBOX_REQUEST:
        case NDrive::NNavTelecom::MT_BLACKBOX_ANSWER:
        case NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST:
        case NDrive::NNavTelecom::MT_ADDITIONAL_BLACKBOX_ANSWER:
        case NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST:
        case NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_ANSWER:
        case NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST:
        case NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER:
        case NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST:
        case NDrive::NNavTelecom::MT_ONLINE_BLACKBOX_ANSWER:
            return NDrive::NNavTelecom::PT_FLEX;
        default:
            return NDrive::NNavTelecom::PT_INCORRECT;
        }
    }


    ui8 Reverse(ui8 value) {
        value = (value & 0xF0) >> 4 | (value & 0x0F) << 4;
        value = (value & 0xCC) >> 2 | (value & 0x33) << 2;
        value = (value & 0xAA) >> 1 | (value & 0x55) << 1;
        return value;
    }

    NDrive::NNavTelecom::TParameter::TValue GetParameterValue(IInputStream *input, NDrive::NNavTelecom::EParameterType type, size_t size = 0) {
        Y_ENSURE(input);

        NDrive::NNavTelecom::TParameter::TValue value;

        switch (type) {
        case NDrive::NNavTelecom::EParameterType::Char:
        case NDrive::NNavTelecom::EParameterType::I8: {
            i8 result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = i64(result);
            break;
        }
        case NDrive::NNavTelecom::EParameterType::I16: {
            i16 result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = i64(result);
        }
        case NDrive::NNavTelecom::EParameterType::I32: {
            i32 result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = i64(result);
            break;
        }
        case NDrive::NNavTelecom::EParameterType::I64: {
            i64 result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = result;
            break;
        }
        case NDrive::NNavTelecom::EParameterType::U8:
        case NDrive::NNavTelecom::EParameterType::U16:
        case NDrive::NNavTelecom::EParameterType::U32:
        case NDrive::NNavTelecom::EParameterType::U64: {
            ui64 result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = result;
            break;
        }
        case NDrive::NNavTelecom::EParameterType::Float: {
            float result = 0;
            input->Load(&result, NPrivate::SizeOf(type));
            value = double(result);
            break;
        }
        case NDrive::NNavTelecom::EParameterType::Array: {
            if (size == 0) {
                value = TNull();
                break;
            }
            TBuffer result(size);
            result.Advance(input->Load(result.End(), size));
            value = std::move(result);
        }
        default: {
            value = TNull();
            break;
        }
            };

        return value;
    }

    constexpr size_t GetTypeSize(NDrive::NNavTelecom::EParameterType type) {
        switch (type) {
        case NDrive::NNavTelecom::EParameterType::U8:
        case NDrive::NNavTelecom::EParameterType::I8:
        case NDrive::NNavTelecom::EParameterType::Char:
            return 1;
        case NDrive::NNavTelecom::EParameterType::U16:
        case NDrive::NNavTelecom::EParameterType::I16:
            return 2;
        case NDrive::NNavTelecom::EParameterType::U32:
        case NDrive::NNavTelecom::EParameterType::I32:
        case NDrive::NNavTelecom::EParameterType::Float:
            return 4;
        case NDrive::NNavTelecom::EParameterType::U64:
        case NDrive::NNavTelecom::EParameterType::I64:
            return 8;
        default:
            return 0;
        }
    }

    void ProtectFlexChecksum(NDrive::NNavTelecom::EMessageType type, ui8 received, const TBuffer& buffer, TStringBuf message = "checksum mismatch") {
        ui8 calculated = NDrive::NavTelecomFlexCrc8(buffer.Begin(), buffer.End());

        Y_ENSURE(
            calculated == received,
            ToString(type) << " " + ToString(message) + " " << ToString(calculated) << " " << ToString(received) << " " << HexEncode(buffer.Data(), buffer.Size())
        );
    }

    using TParameterMetaValue = NDrive::NNavTelecom::EParameterType;
    using TParameterMetaKey = std::pair<ui64, ui64>;
    using TParamterSize = TMap<TParameterMetaKey, TParameterMetaValue>;

    const TParamterSize ParameterMeta = {
        { { NDrive::NNavTelecom::PI_ID, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_EVENT_ID, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_TIMESTAMP, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_DEVICE_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_MODULE_STATUS_1, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_MODULE_STATUS_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_GSM_LEVEL, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_GPS_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_GPS_VALID_TIMESTAMP, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_LATITUDE, 0 }, NDrive::NNavTelecom::EParameterType::I32 },
        { { NDrive::NNavTelecom::PI_LONGITUDE, 0 }, NDrive::NNavTelecom::EParameterType::I32 },
        { { NDrive::NNavTelecom::PI_HEIGHT, 0 }, NDrive::NNavTelecom::EParameterType::I32 },
        { { NDrive::NNavTelecom::PI_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_COURSE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CURRENT_MILEAGE, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_MILEAGE_BETWEEN_CURRENT_AND_LAST, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_TIMESTAMP_BETWEEN_CURRENT_AND_LAST, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_DIFFERENCE_TIMESTAMP_MILEAGE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_MAIN_VOLTAGE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_VOLTAGE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_6, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_7, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ANALOG_INPUT_8, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_DIGITAL_INPUT_CHUNK_1, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DIGITAL_INPUT_CHUNK_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DIGITAL_OUTPUT_CHUNK_1, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DIGITAL_OUTPUT_CHUNK_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_IMPULSE_COUNTER_1, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_COUNTER_2, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ENGINE_HOUR, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_6, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_RS232, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_1, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_2, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_3, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_4, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_5, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_6, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_7, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPERATURE_SENSOR_8, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_CAN_FUEL, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_ENGINE_RPM, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_CAN_FUEL_DISTANCE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_COOLANT_TEMPERATURE, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_CAN_TOTAL_DISTANCE, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_CAN_AXLE_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_AXLE_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_AXLE_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_AXLE_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_AXLE_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_ACCELERATOR, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CAN_BREAK, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CAN_ENGINE_LOAD, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CAN_DIESEL_FILTER_LEVEL, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_ENGINE_UPTIME, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CAN_DISTANCE_TO_MAINTENANCE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_CAN_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_NAVIGATION_INFO, 7 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_HDOP_PDOP, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_HDOP_PDOP, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_GPS_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_LATITUDE_LONGITUDE, 0 }, NDrive::NNavTelecom::EParameterType::I64 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_LATITUDE_LONGITUDE, 1 }, NDrive::NNavTelecom::EParameterType::I64 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_HEIGHT, 0 }, NDrive::NNavTelecom::EParameterType::I32 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_COURSE, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ADDITIONAL_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::Float },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 1 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 2 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 3 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 5 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 6 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 7 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 8 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 9 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 10 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 11 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 12 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 13 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 14 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_LBS_INFO, 15 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_1, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_2, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_3, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_4, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_5, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_6, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_7, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_7, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_8, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_8, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_9, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_9, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_10, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_10, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_11, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_11, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_12, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_12, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_13, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_13, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_14, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_14, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_15, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_15, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_16, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_LEVEL_AND_TEMPERATURE_16, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_1_TO_2, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 7 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 8 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 9 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 10 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_3_TO_6, 11 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 7 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 8 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 9 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 10 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 11 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 12 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 13 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 14 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 15 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 16 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 17 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 18 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 19 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 20 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 21 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 22 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_7_TO_14, 23 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 7 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 8 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 9 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 10 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 11 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 12 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 13 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 14 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 15 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 16 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 17 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 18 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 19 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 20 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 21 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 22 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 23 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 24 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 25 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 26 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 27 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 28 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 29 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 30 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 31 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 32 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 33 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 34 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 35 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 36 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 37 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 38 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 39 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 40 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 41 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 42 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 43 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 44 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 45 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 46 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PRESSURE_SENSOR_FROM_15_TO_30, 47 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_TAHO_INFO, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_TAHO_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_TAHO_FLAGS, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_TAHO_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_TAHO_ODOMETER, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_TAHO_TIME, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_DRIVER_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DISPLAY_MESSAGE_ID, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_DIFFERENCE_TIME, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_AXIS_X_Y_Z, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_AXIS_X_Y_Z, 1 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_AXIS_X_Y_Z, 2 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_VECTOR_MODULE_ACCELERETION, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_MAXIMUM_MINIMUM_ACCELERATION, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_MAXIMUM_MINIMUM_ACCELERATION, 1 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_MAXIMUM_MINIMUM_ACCELERATION, 2 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_1_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_1_2, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_3_4, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_3_4, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_5_6, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_5_6, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_7_8, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_7_8, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_9_10, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_9_10, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_11_12, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_11_12, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_13_14, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_13_14, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_15_16, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_PASSENGER_COUNTER_15_16, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_INFORMATOR_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_GEOZONE_LAST_ID, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_LAST_STOP_ID, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_ROUTE_ID, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAMERA_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DEVICE_STATUS_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_FUNCTIONAL_MODULE_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CONNECTION_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_DIGITAL_SENSORS_VALUE, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_3, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_4, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_5, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_6, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_7, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_IMPULSE_SENSOR_VALUE_8, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_6, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_7, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FREQUENCY_SENSOR_8, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_VIRTUAL_ACCELEROMETER, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_INTERNAL_TILT_LOCAL_VERTICAL, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_INTERNAL_TILT_SHEER_VERTICAL, 0 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_INTERNAL_TILT_SHEER_VERTICAL, 1 }, NDrive::NNavTelecom::EParameterType::I8 },
        { { NDrive::NNavTelecom::PI_EXTERNAL_TILT, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_EXTERNAL_TILT, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_EXTERNAL_TILT, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_MAXIMUM_ACCELERATION, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_MAXIMUM_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 5 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 6 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_SPEED, 7 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_ACCELERATION, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_ACCELERATION, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ECO_DRIVING_STEP_ACCELERATION, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_6, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_7, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_8, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_9, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_10, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_11, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_12, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_13, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_14, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_15, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FREQUENCY_OUTPUT_16, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_1, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_2, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_3, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_4, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_1, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_3, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_4, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_ERROR, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_TOTAL, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_BY_SESSION, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_CURRENT, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_TOTAL_VOLUME, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_SPEED, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_TEMPERATURE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_TOTAL_VOLUME_REVERSE, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_SPEED_REVERSE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_FUEL_FLOW_SENSOR_TEMPERATURE_REVERSE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_STATUS, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_TEMPERATURE_1, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_TEMPERATURE_2, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_TEMPERATURE_3, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_DIRECT_TEMPERATURE_1, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_DIRECT_TEMPERATURE_2, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_DIRECT_TEMPERATURE_3, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_OUTSIDE_TEMPERATURE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_TEMPERATURE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_VOLTAGE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_AMPERAGE, 0 }, NDrive::NNavTelecom::EParameterType::I16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ENGINE_HOUR, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_HOUR, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_COUNT, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_COUNT, 1 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_2, 1 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_4, 1 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ERROR_4, 2 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ENGINE_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ENGINE_STATUS, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_ENGINE_STATUS, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_REFRIGERATOR_COMPRESSOR_SETTING, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_GEOZONE_STATUS, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_FLAGS, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CAN_EVENT, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CAN_EMERGENCY_INDICATION, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_ERROR, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ERROR, 1 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ERROR, 2 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ERROR, 3 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_ERROR, 4 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_ENGINE_HOUR, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_DIAGNOSTIC_CODE, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_1, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_2, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_3, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_4, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_5, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_6, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_7, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_8, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_9, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_10, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_11, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_12, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_13, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_14, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_15, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_1_16, 0 }, NDrive::NNavTelecom::EParameterType::U8 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_1, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_2, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_3, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_4, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_5, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_6, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_7, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_8, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_9, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_10, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_11, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_12, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_13, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_14, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_2_15, 0 }, NDrive::NNavTelecom::EParameterType::U16 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_1, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_2, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_3, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_4, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_5, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_6, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_7, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_8, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_9, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_10, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_11, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_12, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_13, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_14, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_4_15, 0 }, NDrive::NNavTelecom::EParameterType::U32 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_8_1, 0 }, NDrive::NNavTelecom::EParameterType::U64 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_8_2, 0 }, NDrive::NNavTelecom::EParameterType::U64 },
        { { NDrive::NNavTelecom::PI_CUSTOMER_PARAMETER_8_3, 0 }, NDrive::NNavTelecom::EParameterType::U64 },
    };

    TVector<NDrive::NNavTelecom::EParameterType> GetTypeById(NDrive::NNavTelecom::EParameterId id) {
        TVector<NDrive::NNavTelecom::EParameterType> result;

        for (auto& meta : ParameterMeta) {
            if (meta.first.first == id) {
                result.push_back(meta.second);
            }
        }

        return result;
    }

    NDrive::NNavTelecom::EParameterType GetTypeById(ui64 id, ui64 subId) {
        auto realId = std::make_pair(id, subId);
        return ParameterMeta.Value(realId, NDrive::NNavTelecom::EParameterType::None);
    }

    size_t GetPacketSize(const NDrive::NNavTelecom::TBitField& bitField) {
        size_t result = 0;

        for (size_t i = 0; i < (bitField.GetChunkCount() * 8); ++i) {
            auto parameterId = static_cast<NDrive::NNavTelecom::EParameterId>(i);
            bool isPresent = bitField.Get(i);

            if (!isPresent) {
                continue;
            }

            auto types = GetTypeById(parameterId);

            auto summator = [] (size_t counter, NDrive::NNavTelecom::EParameterType type) {
                return NPrivate::SizeOf(type) + counter;
            };

            result += Accumulate(types.begin(), types.end(), size_t(0), summator);
        }

        return result;
    }
}

NDrive::NNavTelecom::TMessage::TMessage(EMessageType type, ESource source, TBitField bitField)
    : BitField(bitField)
    , Source(source)
{
    Payload = CreatePayload(type, BitField);
}

NDrive::NNavTelecom::TMessage::TMessage(THolder<NProtocol::IPayload>&& payload, TBitField&& bitField)
    : BitField(bitField)
{
    Payload = std::move(payload);
}

TString NDrive::NNavTelecom::THeader::DebugString() const {
    TString result;

    result += "Header ";
    result += ToString(ReceiverId) + " ";
    result += ToString(TransmitterId) + " ";
    result += ToString(DataSize) + " ";
    result += ToString(DataChecksum);

    return result;
}

void NDrive::NNavTelecom::TMessage::Load(IInputStream &input) {
    EPreambleType preambleType;
    SaveLoad(&input, preambleType);

    try {
        Y_ENSURE(GetEnumNames<EPreambleType>().contains(preambleType), "bad type " << preambleType);
        Y_ENSURE(preambleType != PT_INCORRECT, "bad preamble type " << preambleType);

        size_t realPreambleSize = Singleton<TPreambleParser>()->GetPreambleSize(preambleType);
        Y_ENSURE(realPreambleSize > 0, "bad real preamble size " << realPreambleSize);

        if (preambleType == PT_NTCB) {
            input.Skip(realPreambleSize - sizeof(EPreambleType));
            LoadNTCB(input);
        } else if (preambleType == PT_FLEX) {
            LoadFlex(input);
        } else if (preambleType == PT_PING) {
            Payload = CreatePayload(MT_PING, BitField);
            Y_ENSURE(Payload);
        }

        Y_ENSURE(Payload);
    } catch (const TIncompleteDataException&) {
        throw;
    } catch (const std::exception& e) {
        ythrow yexception() << "cannot deserialize "
            << preambleType << " "
            << FormatExc(e);
    }

    Payload->PostLoad(input);
}

namespace {
    NDrive::NNavTelecom::THeader LoadHeader(IInputStream& input) {
        NDrive::NNavTelecom::THeader result;
        size_t size = result.CalcSize() + sizeof(ui8);
        TBuffer buffer(size);

        buffer.Advance(input.Read(buffer.End(), size));

        if (buffer.size() < size) {
            throw NDrive::TIncompleteDataException(size, buffer.size());
        }

        TBufferInput stream(buffer);
        ui8 checksum = 0;

        ::Load(&stream, result);
        ::Load(&stream, checksum);

        TBuffer full;
        TBufferOutput fullStream(full);

        auto preamble = Singleton<TPreambleParser>()->GetNTCBPreamble();
        ::SavePodArray(&fullStream, preamble.data(), preamble.size());
        ::SavePodArray(&fullStream, buffer.data(), buffer.size() - sizeof(ui8)); // without checksum

        auto current = NDrive::NavTelecomCrc8(full.Begin(), full.End());

        Y_ENSURE(
            current == checksum,
            result.DebugString() << " header checksum mismatch " << current<< " " << HexEncode(full.Data(), full.Size())
        );

        return result;
    }

    char TryLoadChar(IInputStream& input) {
        char ch = 0;
        if (!input.ReadChar(ch)) {
            throw NDrive::TIncompleteDataException(sizeof(ch), 0);
        }
        return ch;
    }
}

void NDrive::NNavTelecom::TMessage::LoadNTCB(IInputStream& input) {
    Header = LoadHeader(input);

    TBuffer buffer(Header.DataSize);
    TBufferInput stream(buffer);

    buffer.Advance(input.Load(buffer.End(), Header.DataSize));

    if (buffer.Size() < Header.DataSize) {
        throw TIncompleteDataException(Header.DataSize, buffer.Size());
    }

    ui8 current = NDrive::NavTelecomCrc8(buffer.Begin(), buffer.End());
    Y_ENSURE(
        current  == Header.DataChecksum,
        Header.DebugString() << " payload checksum mismatch " << current << " " << HexEncode(buffer.Data(), buffer.Size())
    );

    TString preamble;
    EMessageType messageType = MT_INCORRECT;
    preamble.push_back(TryLoadChar(stream));
    preamble.push_back(TryLoadChar(stream));

    if (preamble == "*!") {
        messageType = MT_DIGITAL_OUTPUT_COMMAND;
    } else if (preamble == "@C") {
        messageType = MT_DIGITAL_OUTPUT_ANSWER;
    } else {
        preamble.push_back(TryLoadChar(stream));
    }

    if (messageType == MT_INCORRECT) {
        messageType = Singleton<TPreambleParser>()->Parse(preamble);
    }

    Y_ENSURE(GetEnumNames<EMessageType>().contains(messageType), "bad type " << preamble);

    size_t realPreambleSize = Singleton<TPreambleParser>()->GetSize(messageType);
    stream.Skip(realPreambleSize - preamble.size());

    Payload = CreatePayload(messageType, BitField);
    Y_ENSURE(Payload);
    Payload->Load(stream);
}

namespace {

    void ControlChecksum(std::string_view preamble, IInputStream& input, const NDrive::NProtocol::IPayload& payload) {
        ui8 checksum = 0;
        if (!input.Read(&checksum, sizeof(checksum))) {
            throw NDrive::TIncompleteDataException(sizeof(checksum), 0);
        }

        TBuffer data(preamble.data(), preamble.size());
        TBufferOutput stream(data);
        payload.Save(stream);

        ui8 current = NDrive::NavTelecomFlexCrc8(data.Begin(), data.End());

        Y_ENSURE(
            current == checksum,
            payload.DebugString() << " header checksum mismatch " << current<< " " << HexEncode(data.Data(), data.Size())
        );
    }
}

void NDrive::NNavTelecom::TMessage::LoadFlex(IInputStream& input) {
    TString preamble = "~";
    char c = 0;

    if (!input.ReadChar(c)) {
        throw TIncompleteDataException(sizeof(c), 0);
    }

    preamble += c;
    auto messageType = Singleton<TPreambleParser>()->Parse(preamble);
    Y_ENSURE(GetEnumNames<EMessageType>().contains(messageType), "bad type " << ToString(preamble));

    auto info = Singleton<TPreambleParser>()->GetFlexInfo(preamble);
    Y_ENSURE(info.GetAnswer() != MT_INCORRECT && info.GetRequest() != MT_INCORRECT);

    auto request = CreatePayload(info.GetRequest(), BitField);
    auto answer = CreatePayload(info.GetAnswer(), BitField);

    Y_ENSURE(request && answer);

    if (Source == ESource::Client) {
        answer->Load(input);
        Payload = std::move(answer);
        ControlChecksum(preamble, input, *Payload);
    } else {
        request->Load(input);
        Payload = std::move(request);
    }
}

void NDrive::NNavTelecom::TMessage::Save(IOutputStream& output) const {
    auto messageType = static_cast<EMessageType>(GetMessageType());
    auto preambleType = GetPreambleType(messageType);

    if (preambleType == PT_NTCB) {
        SaveNTCB(output);
    } else if (preambleType == PT_FLEX) {
        SaveFlex(output);
    }
}

void NDrive::NNavTelecom::TMessage::SaveNTCB(IOutputStream& output) const {
    Y_ENSURE(IsValid());

    auto messageType = static_cast<EMessageType>(GetMessageType());
    auto payloadPreamble = Singleton<TPreambleParser>()->GetPreamble(messageType);

    size_t payloadSize = Payload->GetSize() + payloadPreamble.Size();

    TBuffer payloadBuffer(payloadSize);
    TBufferOutput payloadStream(payloadBuffer);

    for (auto symbol : payloadPreamble) {
        SaveLoad(&payloadStream, symbol);
    }
    Payload->Save(payloadStream);

    ui8 payloadChecksum = NDrive::NavTelecomCrc8(payloadBuffer.Begin(), payloadBuffer.End());

    THeader header = Header;

    header.DataSize = payloadSize;
    header.DataChecksum = payloadChecksum;

    const auto& headerPreamble = Singleton<TPreambleParser>()->GetNTCBPreamble();

    size_t headerSize = header.CalcSize() + headerPreamble.Size();
    TBuffer headerBuffer(headerSize);
    TBufferOutput headerStream(headerBuffer);

    for (auto symbol : headerPreamble) {
        SaveLoad(&headerStream, symbol);
    }
    SaveLoad(&headerStream, header);

    ui8 headerChecksum = NDrive::NavTelecomCrc8(headerBuffer.Begin(), headerBuffer.End());

    output.Write(headerBuffer.Data(), headerBuffer.Size());
    SaveLoad(&output, headerChecksum);
    output.Write(payloadBuffer.Data(), payloadBuffer.Size());
}

void NDrive::NNavTelecom::TMessage::SaveFlex(IOutputStream& output) const {
    auto messageType = static_cast<EMessageType>(GetMessageType());
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(messageType);

    Y_ENSURE(Payload);

    TBuffer buffer(preamble.Size() + Payload->GetSize());
    TBufferOutput bufferStream(buffer);
    bufferStream.Write(preamble.Data(), preamble.Size());
    Payload->Save(bufferStream);

    ui8 checksum = NDrive::NavTelecomFlexCrc8(buffer.Begin(), buffer.End());

    output.Write(buffer.Data(), buffer.Size());
    output.Write(static_cast<char>(checksum));
}

size_t NDrive::NNavTelecom::TProtocolSettingRequest::GetSize() const {
    size_t result = 0;
    result += NPrivate::SizeOf(Protocol);
    result += NPrivate::SizeOf(ProtocolVersion);
    result += NPrivate::SizeOf(StructVersion);
    result += NPrivate::SizeOf(DataSize);
    result += GetBitFieldSize();
    return result;
}

void NDrive::NNavTelecom::TProtocolSettingRequest::Load(IInputStream& input) {
    ::Load(&input, Protocol);
    ::Load(&input, ProtocolVersion);
    ::Load(&input, StructVersion);
    ::Load(&input, DataSize);

    ui8 chunkSize = sizeof(ui8);
    size_t bitFieldSize = CalculateBitFieldSize(DataSize);
    ui64 bitSize = DataSize;

    size_t realBitSize = sizeof(bitSize);
    size_t bufferSize = bitFieldSize + chunkSize + realBitSize;

    TBuffer buffer(bufferSize);
    TBufferInput bitFieldInputStream(buffer);
    TBufferOutput bitFieldOutputStream(buffer);

    ::Save(&bitFieldOutputStream, chunkSize);
    ::Save(&bitFieldOutputStream, bitSize);

    for (size_t i = 0; i < bitFieldSize; ++i) {
        ui8 data = 0;
        if (input.Load(&data, sizeof(data)) == 0) {
            throw TIncompleteDataException(bitFieldSize, buffer.Size());
        }
        data = Reverse(data);
        buffer.Append(static_cast<i8>(data));
    }

    BitField.Load(&bitFieldInputStream);
}

void NDrive::NNavTelecom::TProtocolSettingRequest::Save(IOutputStream& output) const {
    ::Save(&output, Protocol);
    ::Save(&output, ProtocolVersion);
    ::Save(&output, StructVersion);
    ::Save(&output, DataSize);

    const ui8* chunks = BitField.GetChunks();
    size_t size = BitField.GetChunkCount();
    for (size_t i = 0; i < size; ++i) {
        ui8 data = chunks[i];
        data = Reverse(data);
        ::Save(&output, data);
    }
}

TString NDrive::NNavTelecom::TProtocolSettingRequest::DebugString() const {
    TString result = ToString(GetMessageTypeAs<EMessageType>());

    result += " ";
    result += ToString(Protocol) + " ";
    result += ToString(ProtocolVersion) + " ";
    result += ToString(StructVersion) + " ";
    result += ToString(DataSize) + " ";
    result += GetBitFieldDebugString();

    return result;
}

TString NDrive::NNavTelecom::TProtocolSettingRequest::GetBitFieldDebugString() const {
    TString result;
    size_t bitsCount = BitField.GetChunkCount() * 8;
    for (size_t i = 0; i < bitsCount; ++i) {
        result += ToString(bool(BitField[i]));
    }
    return result;
}

TString NDrive::NNavTelecom::TProtocolSettingAnswer::DebugString() const {
    TString result = ToString(GetMessageTypeAs<EMessageType>());

    result += " ";
    result += ToString(Protocol) + " ";
    result += ToString(ProtocolVersion) + " ";
    result += ToString(StructVersion) + " ";

    return result;
}

size_t NDrive::NNavTelecom::TBlackBoxRequest::GetSize() const {
    size_t packetSize = BitField.GetChunkCount();

    size_t result = sizeof(Count);
    result += packetSize * Count;
    return result;
}

void NDrive::NNavTelecom::TBlackBoxRequest::Load(IInputStream& input) {
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(GetMessageTypeAs<EMessageType>());

    ::Load(&input, Count);

    size_t packetSize = ::GetPacketSize(BitField);
    size_t packetsSize = Count * packetSize;
    size_t totalSize = packetsSize + preamble.Size() + sizeof(Count);

    TBuffer buffer(totalSize);
    TBufferInput stream(buffer);

    buffer.Append(preamble.Data(), preamble.Size());
    buffer.Append(static_cast<char>(Count));

    buffer.Advance(input.Load(buffer.End(), packetsSize));
    if (buffer.Size() < totalSize) {
        throw TIncompleteDataException(totalSize, buffer.Size());
    }

    ui8 checksum = 0;
    ::Load(&input, checksum);
    ProtectFlexChecksum(GetMessageTypeAs<EMessageType>(), checksum, buffer);

    stream.Skip(preamble.Size() + sizeof(Count));

    for (i16 i = 0; i < Count; ++i) {
        auto& packet = Records.emplace_back();
        packet.Load(&stream, BitField);
    }
}

void NDrive::NNavTelecom::TBlackBoxRequest::Save(IOutputStream& output) const {
    TBuffer result;
    TBufferOutput stream(result);

    ui8 count = Records.size();

    ::Save(&output, count);
    for (const auto& record : Records) {
        ::Save(&stream, record);
    }

    ::SavePodArray(&output, result.Data(), result.Size());
}

TString NDrive::NNavTelecom::TBlackBoxRequest::DebugString() const {
    TString result;

    result += ToString(Count);
    result += " ";
    for (auto& record : Records) {
        result += "{" + record.DebugString() + "}";
    }

    return result;
}

size_t NDrive::NNavTelecom::TSingleBlackBoxRequest::GetSize() const {
    size_t packetSize = ::GetPacketSize(BitField);
    return packetSize + sizeof(EventIndex);
}

void NDrive::NNavTelecom::TSingleBlackBoxRequest::Load(IInputStream& input) {
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(GetMessageTypeAs<EMessageType>());

    SaveLoad(&input, EventIndex);

    size_t packetSize = ::GetPacketSize(BitField);
    size_t totalSize = preamble.Size() + packetSize + sizeof(EventIndex);

    TBuffer buffer(totalSize);
    TBufferInput stream(buffer);
    buffer.Append(preamble.Data(), preamble.Size());
    buffer.Append(reinterpret_cast<const char*>(&EventIndex), sizeof(EventIndex));
    buffer.Advance(input.Load(buffer.End(), packetSize));

    if (buffer.Size() < totalSize) {
        throw TIncompleteDataException(totalSize, buffer.Size());
    }

    ui8 checksum = 0;
    SaveLoad(&input, checksum);
    ProtectFlexChecksum(GetMessageTypeAs<EMessageType>(), checksum, buffer);

    stream.Skip(preamble.Size());
    stream.Skip(sizeof(EventIndex));
    Record.Load(&stream, BitField);
}

void NDrive::NNavTelecom::TSingleBlackBoxRequest::Save(IOutputStream& output) const {
    Y_UNUSED(output);
}

TString NDrive::NNavTelecom::TSingleBlackBoxRequest::DebugString() const {
    return ToString(GetMessageTypeAs<EMessageType>()) + " " + Record.DebugString() + " " + ToString(EventIndex);
}

size_t NDrive::NNavTelecom::TRecord::GetSize() const {
    size_t result = 0;

    for (auto& parameter : Parameters) {
        result += parameter.GetSize();
    }

    return result;
}

NDrive::NNavTelecom::TParameter::TValue NDrive::NNavTelecom::TRecord::Get(EParameterId id, ui64 subId) const {
    auto comparator = [&id, &subId] (const TParameter& parameter) {
        return parameter.GetId() == id && parameter.GetSubId() == subId;
    };
    auto iter = FindIf(Parameters.begin(), Parameters.end(), comparator);

    if (iter != Parameters.end()) {
        return iter->GetValue();
    }
    return TNull();
}

void NDrive::NNavTelecom::TRecord::Load(IInputStream* input, const TBitField& bitField) {
    size_t packetSize = GetPacketSize(bitField);
    TBuffer buffer(packetSize);
    TBufferInput stream(buffer);

    buffer.Advance(input->Load(buffer.End(), packetSize));

    if (buffer.Size() < packetSize) {
        throw TIncompleteDataException(packetSize, buffer.Size());
    }

    size_t currentCount = 0;
    size_t totalCount = bitField.Count();
    size_t lastPosition = currentCount;

    while (currentCount < totalCount) {
        bool isPresent = bitField.Get(lastPosition);

        if (!isPresent) {
            continue;
        }

        auto id = static_cast<NDrive::NNavTelecom::EParameterId>(lastPosition);
        auto types = GetTypeById(id);
        Y_ENSURE(!types.empty());

        for (size_t i = 0; i < types.size(); ++i) {
            auto value = GetParameterValue(&stream, types[i]);
            Parameters.emplace_back(id, i, std::move(value));
        }

        lastPosition = bitField.NextNonZeroBit(lastPosition);
        currentCount++;
    }
}

void NDrive::NNavTelecom::TRecord::Save(IOutputStream* output) const {
    for (const auto& parameter : Parameters) {
        const size_t size = GetTypeSize(GetTypeById(parameter.GetId(), parameter.GetSubId()));

        std::visit([output, size] (const auto& value) {
            if constexpr (std::is_same_v<decltype(value), TBuffer>) {
                ::SavePodArray(output, value.Data(), value.Size());
            } else {
                ::SavePodArray(output, reinterpret_cast<const ui8*>(&value), size);
            }
        }, parameter.GetValue());
    }
}

TString NDrive::NNavTelecom::TRecord::DebugString() const {
    TString result;
    for (auto& parameter : Parameters) {
        result += parameter.DebugString() + ",";
    }
    return result;
}

size_t NDrive::NNavTelecom::TAdditional::GetSize() const {
    size_t result = 0;
    result += sizeof(TotalLength);
    result += sizeof(StructVersion);
    result += sizeof(DataLength);
    result += GetHeadSize();

    for (auto&& item : Fields) {
        auto& value = item.GetValue();
        if (std::holds_alternative<TBuffer>(value)) {
            result += std::get<TBuffer>(value).Size();
        }
    }
    return result;
}

size_t NDrive::NNavTelecom::TAdditional::GetHeadSize() const {
    size_t result = 0;
    result += sizeof(Number);
    result += sizeof(EventCode);
    result += sizeof(Timestamp);
    result += sizeof(GpsStatus);
    result += sizeof(GpsTimestamp);
    result += sizeof(Latitude);
    result += sizeof(Longitude);
    result += sizeof(Height);
    result += sizeof(Speed);
    result += sizeof(Course);
    result += sizeof(Mileage);
    return result;
}

void NDrive::NNavTelecom::TAdditional::Load(IInputStream* input) {
    Y_ENSURE(input);

    SaveLoad(input, TotalLength);
    SaveLoad(input, StructVersion);
    SaveLoad(input, DataLength);
    SaveLoad(input, Number);
    SaveLoad(input, EventCode);
    SaveLoad(input, Timestamp);
    SaveLoad(input, GpsStatus);
    SaveLoad(input, GpsTimestamp);
    SaveLoad(input, Latitude);
    SaveLoad(input, Longitude);
    SaveLoad(input, Height);
    SaveLoad(input, Speed);
    SaveLoad(input, Course);
    SaveLoad(input, Mileage);

    i32 currentLength = static_cast<i32>(TotalLength - sizeof(StructVersion) - sizeof(DataLength) - DataLength);

    while (currentLength > 0) {
        ui8 type = 0;
        ui8 length = 0;

        SaveLoad(input, type);
        SaveLoad(input, length);

        currentLength -= sizeof(type);
        currentLength -= sizeof(length);

        TBuffer value(length);
        value.Advance(input->Load(value.End(), length));
        Fields.emplace_back(ui64(type), ui64(EParameterType::Array), std::move(value));

        currentLength -= length;
    }
}

void NDrive::NNavTelecom::TAdditional::Save(IOutputStream* output) {
    Y_UNUSED(output);
}

TString NDrive::NNavTelecom::TAdditional::DebugString() const {
    TString result;

    result += ToString(TotalLength) + " ";
    result += ToString(StructVersion) + " ";
    result += ToString(DataLength) + " ";
    result += ToString(Number) + " ";
    result += ToString(Timestamp) + " ";
    result += ToString(GpsStatus) + " ";
    result += ToString(GpsTimestamp) + " ";
    result += ToString(Latitude) + " ";
    result += ToString(Longitude) + " ";
    result += ToString(Height) + " ";
    result += ToString(Speed) + " ";
    result += ToString(Course) + " ";
    result += ToString(Mileage) + " ";

    for (auto&& field : Fields) {
        result += "[" + ToString(field.GetId()) + "]=" + field.DebugString() + " ";
    }

    return result;
}

NDrive::NNavTelecom::TParameter::TValue NDrive::NNavTelecom::TAdditional::Get(ui64 id) const {
    auto comparator = [&id] (const TParameter& parameter) {
        return parameter.GetId() == id;
    };

    auto iter = FindIf(Fields.begin(), Fields.end(), comparator);

    if (iter != Fields.end()) {
        return iter->GetValue();
    }
    return TNull();
}

size_t NDrive::NNavTelecom::TParameter::GetSize() const {
    auto parameterType = GetTypeById(Id, SubId);
    if (parameterType == EParameterType::Array) {
        if (std::holds_alternative<TBuffer>(Value)) {
            return std::get<TBuffer>(Value).Size();
        }
        return 0;
    }
    return NPrivate::SizeOf(parameterType);
}

TString NDrive::NNavTelecom::TParameter::DebugString() const {
    TString result;

    result += "[" + ToString(Id) + "][" + ToString(SubId) + "]=";
    if (std::holds_alternative<i64>(Value)) {
        result += ToString(std::get<i64>(Value));
    } else if (std::holds_alternative<ui64>(Value)) {
        result += ToString(std::get<ui64>(Value));
    } else if (std::holds_alternative<double>(Value)) {
        result += ToString(std::get<double>(Value));
    } else if (std::holds_alternative<TBuffer>(Value)) {
        auto& value = std::get<TBuffer>(Value);
        result += HexEncode(value.Data(), value.Size());
    }

    return result;
}

size_t NDrive::NNavTelecom::TAdditionalBlackBoxRequest::GetSize() const {
    size_t result = 0;
    result += sizeof(Count);
    for (auto& record : Records) {
        result += record.GetSize();
    }
    return result;
}

void NDrive::NNavTelecom::TAdditionalBlackBoxRequest::Load(IInputStream& input) {
    SaveLoad(&input, Count);
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(GetMessageTypeAs<EMessageType>());

    TBuffer buffer;
    TBufferInput stream(buffer);
    buffer.Append(preamble.Data(), preamble.Size());
    buffer.Append(static_cast<char>(Count));

    for (size_t i = 0; i < Count; ++i) {
        ui16 packetSize = 0;
        SaveLoad(&input, packetSize);

        buffer.Append(reinterpret_cast<const char*>(&packetSize), sizeof(packetSize));

        TBuffer currentPacket(packetSize);
        currentPacket.Advance(input.Load(currentPacket.End(), packetSize));

        if (currentPacket.Size() < packetSize) {
            throw TIncompleteDataException(packetSize, currentPacket.Size());
        }

        buffer.Append(currentPacket.Data(), currentPacket.Size());
    }

    ui8 checksum = 0;
    SaveLoad(&input, checksum);

    ProtectFlexChecksum(GetMessageTypeAs<EMessageType>(), checksum, buffer);

    stream.Skip(preamble.Size());
    stream.Skip(sizeof(Count));

    for (size_t i = 0; i < Count; ++i) {
        auto& record = Records.emplace_back();
        record.Load(&stream);
    }
}

void NDrive::NNavTelecom::TAdditionalBlackBoxRequest::Save(IOutputStream& output) const {
    Y_UNUSED(output);
}

TString NDrive::NNavTelecom::TAdditionalBlackBoxRequest::DebugString() const {
    TString result = ToString(GetMessageTypeAs<EMessageType>()) + " ";

    result += ToString(Count);
    for (const auto& record : Records) {
        result += "{" + record.DebugString() + "}";
    }

    return result;
}

size_t NDrive::NNavTelecom::TAdditionalSingleBlackBoxRequest::GetSize() const {
    return Record.GetSize() + sizeof(EventIndex);
}

void NDrive::NNavTelecom::TAdditionalSingleBlackBoxRequest::Load(IInputStream& input) {
    SaveLoad(&input, EventIndex);
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(GetMessageTypeAs<EMessageType>());

    TBuffer buffer;
    TBufferInput stream(buffer);
    buffer.Append(preamble.Data(), preamble.Size());
    buffer.Append(reinterpret_cast<const char*>(&EventIndex), sizeof(EventIndex));

    ui16 packetSize = 0;
    SaveLoad(&input, packetSize);

    buffer.Append(reinterpret_cast<const char*>(&packetSize), sizeof(packetSize));

    TBuffer currentPacket(packetSize);
    currentPacket.Advance(input.Load(currentPacket.End(), packetSize));

    if (currentPacket.Size() < packetSize) {
        throw TIncompleteDataException(packetSize, currentPacket.Size());
    }

    buffer.Append(currentPacket.Data(), currentPacket.Size());

    ui8 checksum = 0;
    SaveLoad(&input, checksum);

    ProtectFlexChecksum(GetMessageTypeAs<EMessageType>(), checksum, buffer);

    stream.Skip(preamble.Size());
    stream.Skip(sizeof(EventIndex));

    Record.Load(&stream);
}

void NDrive::NNavTelecom::TAdditionalSingleBlackBoxRequest::Save(IOutputStream& output) const {
    Y_UNUSED(output);
}

TString NDrive::NNavTelecom::TAdditionalSingleBlackBoxRequest::DebugString() const {
    TString result = ToString(GetMessageTypeAs<EMessageType>()) + " ";

    result += Record.DebugString() + " ";
    result += ToString(EventIndex);

    return result;
}

size_t NDrive::NNavTelecom::TOnlineBlackBoxRequest::GetSize() const {
    return ::GetPacketSize(BitField);
}

void NDrive::NNavTelecom::TOnlineBlackBoxRequest::Load(IInputStream& input) {
    auto preamble = Singleton<TPreambleParser>()->GetPreamble(GetMessageTypeAs<EMessageType>());
    size_t packetSize = ::GetPacketSize(BitField);
    size_t totalSize = preamble.Size() + packetSize;

    TBuffer buffer(totalSize);
    TBufferInput stream(buffer);
    buffer.Append(preamble.Data(), preamble.Size());
    buffer.Advance(input.Load(buffer.End(), packetSize));

    if (buffer.Size() < totalSize) {
        throw TIncompleteDataException(totalSize, buffer.Size());
    }

    ui8 checksum = 0;
    SaveLoad(&input, checksum);
    ProtectFlexChecksum(GetMessageTypeAs<EMessageType>(), checksum, buffer);

    stream.Skip(preamble.Size());
    Record.Load(&stream, BitField);
}

void NDrive::NNavTelecom::TOnlineBlackBoxRequest::Save(IOutputStream& output) const {
    Y_UNUSED(output);
}

TString NDrive::NNavTelecom::TOnlineBlackBoxRequest::DebugString() const {
    return ToString(GetMessageTypeAs<EMessageType>()) + " " + Record.DebugString();
}

double NDrive::NNavTelecom::CoordinateConvert(i64 coordinate) {
    const ui64 scale = 10000;
    const double minuteCoefficient = 60.;

    bool isSigned = coordinate < 0;

    if (isSigned) {
        coordinate *= -1;
    }

    auto seconds = double(coordinate % scale);
    auto degrees = ui64((double(coordinate) / scale) / minuteCoefficient);
    auto minutes = ui64((double(coordinate) / scale) - (double(degrees) * minuteCoefficient));

    double result = (double(minutes) + (seconds / double(scale))) / minuteCoefficient;
    result += double(degrees);

    if (isSigned) {
        result *= -1.0;
    }

    return result;
}

void NDrive::NNavTelecom::TDigitalOutputCommandRequest::Load(IInputStream& input) {
    char number = '0';
    char state = 'N';

    ::Load(&input, number);
    ::Load(&input, state);

    Y_ENSURE(state == 'N' || state == 'Y', "ivalid digital output state " << state);
    Y_ENSURE('0' <= number && number <= '9', "invalid digital output number " << number);

    Number = FromString<ui8>(&number, sizeof(number));
    State = state == 'Y';
}

void NDrive::NNavTelecom::TDigitalOutputCommandRequest::Save(IOutputStream& output) const {
    output.Write(ToString(Number));
    output.Write(State ? 'Y' : 'N');
}

size_t NDrive::NNavTelecom::TDigitalOutputCommandAnswer::GetSize() const {
    return Record.GetSize();
}

void NDrive::NNavTelecom::TDigitalOutputCommandAnswer::Load(IInputStream& input) {
    Record.Load(&input, BitField);
}

void NDrive::NNavTelecom::TDigitalOutputCommandAnswer::Save(IOutputStream& output) const {
    Record.Save(&output);
}

TString NDrive::NNavTelecom::TDigitalOutputCommandAnswer::DebugString() const {
    return ToString(GetMessageTypeAs<EMessageType>()) + " " + Record.DebugString();
}

size_t NDrive::NNavTelecom::TICCIDAnswer::GetSize() const {
    return ICCID.Size();
}

void NDrive::NNavTelecom::TICCIDAnswer::Load(IInputStream& input) {
    ICCID = input.ReadAll();
}

void NDrive::NNavTelecom::TICCIDAnswer::Save(IOutputStream& output) const {
    ::SavePodArray(&output, ICCID.data(), ICCID.size());
}

TString NDrive::NNavTelecom::TICCIDAnswer::DebugString() const {
    return ToString(GetMessageTypeAs<EMessageType>()) + " " + ICCID;
}

namespace {
    using TAddCallback = std::function<void(ui32, NDrive::TSensorValue)>;
    using TConvertCallback = std::function<void(ui32 id, const NDrive::NNavTelecom::TParameter::TValue& value, const TAddCallback& callback)>;

    template<typename TValueType>
    void DefaultConvert(ui32 id, const NDrive::NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            add(id, static_cast<TValueType>(std::get<ui64>(value)));
        } else if (std::holds_alternative<i64>(value)) {
            add(id, static_cast<TValueType>(double(std::get<i64>(value))));
        } else if (std::holds_alternative<double>(value)) {
            add(id, static_cast<TValueType>(std::get<double>(value)));
        }
    }
}

TVector<NDrive::TSensor> NDrive::NNavTelecom::ToSensors(const TRecord& record) {
    NDrive::TMultiSensor sensors;

    auto timestamp = Now();
    auto timestampValue = record.Get(NNavTelecom::EParameterId::PI_TIMESTAMP);
    if (std::holds_alternative<ui64>(timestampValue)) {
        timestamp = TInstant::Seconds(std::get<ui64>(timestampValue));
    }

    const auto coordinateCallback = [] (ui32 id, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<i64>(value)) {
            add(id, NDrive::NNavTelecom::CoordinateConvert(std::get<i64>(value)));
        }
    };
    const auto gpsStatusCallback = [] (ui32, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 gpsStatus = std::get<ui64>(value);
            gpsStatus = (gpsStatus >> 2) & 0x1F;
            add(VEGA_SAT_USED, gpsStatus);
            add(VEGA_GPS_IS_ACTIVE, ui64(gpsStatus & 0x01));
        }
    };
    const auto accelerometerCallback = [] (ui32, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<TBuffer>(value)) {
            const TBuffer& data = std::get<TBuffer>(value);
            if (data.Size() < 6) {
                return;
            }

            TBufferInput stream(data);
            i16 x = 0;
            i16 y = 0;
            i16 z = 0;

            ::Load(&stream, x);
            ::Load(&stream, y);
            ::Load(&stream, z);

            add(VEGA_GSENSOR_AXIS_X, ui64(x));
            add(VEGA_GSENSOR_AXIS_Y, ui64(y));
            add(VEGA_GSENSOR_AXIS_Z, ui64(z));
        }
    };
    const auto navigationInfoCallback = [] (ui32, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<TBuffer>(value)) {
            const auto& data = std::get<TBuffer>(value);
            TBufferInput stream(data);
            if (data.Size() == 8) {
                ui8 glonassInview = 0;
                ui8 gpsInview = 0;

                ::Load(&stream, glonassInview);
                ::Load(&stream, gpsInview);

                add(VEGA_GLONASS_INVIEW, ui64(glonassInview));
                add(VEGA_GPS_INVIEW, ui64(gpsInview));
            }
        }
    };
    const auto moduleStatuses2Callback = [] (ui32, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            auto status = std::get<ui64>(value);
            add(VEGA_GPS_JAMMED, ui64((status >> 2) & 0x01));
        }
    };
    const auto canFuelCallback = [] (ui32, const NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            auto fuel = std::get<ui64>(value);
            auto isPercent = bool((fuel >> 15) & 0x01);
            ui64 maximumFuel = 0x7FFF;
            fuel &= maximumFuel;

            if (isPercent) {
                if (fuel > 100) {
                    fuel = 100;
                }
                add(CAN_FUEL_LEVEL_P, fuel);
            } else if (fuel < maximumFuel) {
                add(CAN_FUEL_LEVEL_P, double(fuel) / 10.0);
            }
        }
    };
    const auto voltageConvertCallback = [](ui32 id, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            double realValue = static_cast<double>(std::get<ui64>(value)) / 1000.0;
            add(id, realValue);
        }
    };
    const auto customBreakCallback = [](ui32 id, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 realValue = SelectBits<3, 1>(std::get<ui64>(value));
            add(id, realValue);
        }
    };
    const auto doorStateCallback = [](ui32, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 realValue = std::get<ui64>(value);
            add(CAN_DRIVER_DOOR, SelectBits<0, 1>(realValue));
            add(CAN_PASS_DOOR, SelectBits<1, 1>(realValue));
            add(CAN_L_REAR_DOOR, SelectBits<2, 1>(realValue));
            add(CAN_R_REAR_DOOR, SelectBits<3, 1>(realValue));
            add(CAN_TRUNK, SelectBits<4, 1>(realValue));
        }
    };
    const auto windowsPositionStateCallback = [](ui32, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 realValue = std::get<ui64>(value);
            add(VEGA_WINDOWS_POSITION_SENSOR, SelectBits<0, 1>(realValue));
        }
    };
    const auto customFuelLevelCallback = [](ui32, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 realValue = std::get<ui64>(value);
            add(CAN_CUSTOM_FUEL_VOLUME, SelectBits<0, 7>(realValue));
        }
    };
    const auto ignitionStateCallback = [](ui32, const NNavTelecom::TParameter::TValue value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            ui64 realValue = std::get<ui64>(value);
            add(CAN_KEY_IN_IGN, SelectBits<0, 1>(realValue));
            add(CAN_IGNITION, SelectBits<1, 1>(realValue));
        }
    };

    const TMap<NNavTelecom::TParameterId, std::pair<ui32, TConvertCallback>> parameterMap = {
        { NNavTelecom::EParameterId::PI_LATITUDE, { VEGA_LAT, coordinateCallback } },
        { NNavTelecom::EParameterId::PI_LONGITUDE, { VEGA_LON, coordinateCallback } },
        { NNavTelecom::EParameterId::PI_HEIGHT, { VEGA_ALT, DefaultConvert<double> } },
        { NNavTelecom::EParameterId::PI_SPEED, { VEGA_SPEED, DefaultConvert<double> } },
        { NNavTelecom::EParameterId::PI_COURSE, { VEGA_DIR, DefaultConvert<double> } },
        { NNavTelecom::EParameterId::PI_GSM_LEVEL, { VEGA_GSM_SIGNAL_LEVEL, DefaultConvert<ui64> } },
        { NNavTelecom::EParameterId::PI_GPS_STATUS, { 0, gpsStatusCallback } },
        { { NNavTelecom::EParameterId::PI_LBS_INFO, 0 }, { VEGA_CELLID, DefaultConvert<ui64> }},
        { { NNavTelecom::EParameterId::PI_LBS_INFO, 1 }, { VEGA_LAC, DefaultConvert<ui64> }},
        { { NNavTelecom::EParameterId::PI_LBS_INFO, 2 }, { VEGA_MCC, DefaultConvert<ui64> }},
        { { NNavTelecom::EParameterId::PI_LBS_INFO, 3 }, { VEGA_MNC, DefaultConvert<ui64> }},
        { NNavTelecom::EParameterId::PI_AXIS_X_Y_Z, { 0, accelerometerCallback } },
        { NNavTelecom::EParameterId::PI_NAVIGATION_INFO, { 0, navigationInfoCallback } },
        { NNavTelecom::EParameterId::PI_MODULE_STATUS_2, { 0, moduleStatuses2Callback } },
        { NNavTelecom::EParameterId::PI_CURRENT_MILEAGE, { CAN_ODOMETER_KM, DefaultConvert<double> } },
        { NNavTelecom::EParameterId::PI_MAIN_VOLTAGE, { VEGA_POWER_VOLTAGE, voltageConvertCallback } },
        { NNavTelecom::EParameterId::PI_ADDITIONAL_VOLTAGE, { VEGA_ACC_VOLTAGE, voltageConvertCallback } },
        { NNavTelecom::EParameterId::PI_CAN_FUEL, { 0, canFuelCallback } },
        { NNavTelecom::EParameterId::PI_CAN_ENGINE_RPM, { CAN_ENGINE_RPM, DefaultConvert<ui64> } },
        { NNavTelecom::EParameterId::PI_COOLANT_TEMPERATURE, { CAN_ENGINE_TEMP, DefaultConvert<double> } },
        { NNavTelecom::EParameterId::PI_CAN_BREAK, { CAN_BRAKE, DefaultConvert<ui64> } },
        { NNavTelecom::EParameterId::PI_CAN_ACCELERATOR, { CAN_ACCELERATOR, DefaultConvert<ui64> } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_1, { CAN_ACCELERATOR, DefaultConvert<ui64> } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_2, { CAN_BRAKE, customBreakCallback } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_3, { 0, doorStateCallback } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_4, { 0, windowsPositionStateCallback } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_5, { 0, customFuelLevelCallback } },
        { NNavTelecom::EParameterId::PI_CUSTOMER_PARAMETER_1_6, { 0, ignitionStateCallback } },
    };

    auto additionalVoltageValue = record.Get(NNavTelecom::EParameterId::PI_ADDITIONAL_VOLTAGE);

    auto add = [&sensors, timestamp] (ui32 id, NDrive::TSensorValue value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };

    for (auto&& parameter : record.Parameters) {
        auto id = static_cast<NNavTelecom::EParameterId>(parameter.GetId());
        if (!parameterMap.contains(id)) {
            continue;
        }
        auto handler = parameterMap.Value(id, std::make_pair(0, TConvertCallback()));

        if (handler.second) {
            handler.second(handler.first, parameter.GetValue(), add);
        }
    }

    return sensors;
}

TVector<NDrive::TSensor> NDrive::NNavTelecom::ToSensors(const TAdditional& record) {
    NDrive::TMultiSensor sensors;

    auto timestamp = TInstant::Seconds(record.Timestamp);

    auto add = [&sensors, timestamp] (ui32 id, auto value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };

    ui64 satellites = (record.GpsStatus >> 2) & 0x1F;
    double latitude = NDrive::NNavTelecom::CoordinateConvert(record.Latitude);
    double longitude = NDrive::NNavTelecom::CoordinateConvert(record.Longitude);

    add(VEGA_LAT, latitude);
    add(VEGA_LON, longitude);
    add(VEGA_ALT, static_cast<double>(record.Height));
    add(VEGA_SPEED, static_cast<double>(record.Speed));
    add(VEGA_DIR, static_cast<double>(record.Course));
    add(VEGA_SAT_USED, satellites);

    return sensors;
}

NDrive::TMultiSensor NDrive::NNavTelecom::ToSensors(const TICCIDAnswer& data) {
    NDrive::TMultiSensor sensors;
    auto timestamp = Now();

    auto add = [&sensors, timestamp] (ui32 id, auto value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };
    add(VEGA_SIM_ICCID, data.ICCID);

    return sensors;
}

template<>
class NPrivate::TSizer<NDrive::NNavTelecom::EParameterType> {
public:
    using TItem = NDrive::NNavTelecom::EParameterType;

    static size_t SizeOf(const TItem& item) {
        switch (item) {
        case TItem::Char:
            return sizeof(char);
        case TItem::I8:
        case TItem::U8:
            return sizeof(ui8);
        case TItem::I16:
        case TItem::U16:
            return sizeof(ui16);
        case TItem::I32:
        case TItem::U32:
            return sizeof(ui32);
        case TItem::I64:
        case TItem::U64:
            return sizeof(ui64);
        case TItem::Float:
            return sizeof(float);
        default:
            return 0;
        }
    }
};

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::NNavTelecom::TParameter::TValue& parameter) {
    NJson::TJsonValue result;
    if (std::holds_alternative<i64>(parameter)) {
        return std::get<i64>(parameter);
    } else if (std::holds_alternative<ui64>(parameter)) {
        return std::get<ui64>(parameter);
    } else if (std::holds_alternative<double>(parameter)) {
        return std::get<double>(parameter);
    } else if (std::holds_alternative<TBuffer>(parameter)) {
        const auto& buffer = std::get<TBuffer>(parameter);
        return HexEncode(buffer.Data(), buffer.Size());
    }
    return result;
}
