#pragma once

#include "common.h"
#include "protocol.h"
#include "sensor.h"
#include "types.h"

#include <util/datetime/base.h>
#include <util/datetime/parser.h>
#include <util/generic/bitmap.h>
#include <util/generic/map.h>
#include <util/generic/maybe.h>

namespace NDrive {
    namespace NNavTelecom {
        enum EMessageType: ui8 {
            MT_INCORRECT = 0 /* "incorrect" */,
            MT_HANDSHAKE_REQUEST /* "handshake request" */,
            MT_HANDSHAKE_ANSWER /* "handshake answer" */,
            MT_PROTOCOL_SETTING_REQUEST /* "protocol setting request" */,
            MT_PROTOCOL_SETTING_ANSWER /* "protocol setting answer" */,
            MT_PING /* "ping" */,
            MT_BLACKBOX_REQUEST /* "black box request" */,
            MT_BLACKBOX_ANSWER /* "black box answer" */,
            MT_ADDITIONAL_BLACKBOX_REQUEST /* "additional black box request" */,
            MT_ADDITIONAL_BLACKBOX_ANSWER /* "additional black box answer" */,
            MT_SINGLE_BLACKBOX_REQUEST /* "single black box request" */,
            MT_SINGLE_BLACKBOX_ANSWER /* "single black box answer" */,
            MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST /* "single black box request" */,
            MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER /* "single black box answer" */,
            MT_ONLINE_BLACKBOX_REQUEST /* "online black box request" */,
            MT_ONLINE_BLACKBOX_ANSWER /* "online black box answer" */,
            MT_DIGITAL_OUTPUT_COMMAND /* "digital output command" */,
            MT_DIGITAL_OUTPUT_ANSWER /* "digital output answer" */,
            MT_ICCID_REQUEST /* "iccid request" */,
            MT_ICCID_ANSWER /* "iccid answer" */,
        };

        enum EPreambleType: i8 {
            PT_INCORRECT = 0 /* "incorrect" */,
            PT_NTCB = '@' /* "NTCB" */,
            PT_FLEX = '~' /* "FLEX" */,
            PT_PING = 0x7F /* "PING" */,
        };

        enum class EProtocolVersion: ui8 {
            None = 0x00 /* "none" */,
            First = 0x0A /* "1.0" */,
            Second = 0x14 /* "2.0" */,
            Third = 0x1E /* "3.0" */,
        };

        enum class EStructVersion: ui8 {
            None = 0x00 /* "none" */,
            First = 0x0A /* "1.0" */,
            Second = 0x14 /* "2.0" */,
            Third = 0x1E /* "3.0" */,
        };

        const TMap<EStructVersion, size_t> StructSizes = {
            {EStructVersion::None, 0},
            {EStructVersion::First, 69},
            {EStructVersion::Second, 122},
            {EStructVersion::Third, 255},
        };

        using TBitField = TBitMapOps<TDynamicBitMapTraits<ui8>>;

        inline constexpr size_t DefaultNTCBPreambleSize = 3;
        inline constexpr size_t DefaultFlexPreambleSize = 2;
        inline constexpr ui8 DefaultFlexProtocolId = 0xB0;

        enum class EParameterType: ui8 {
            None /* "none" */,
            Char /* "char" */,
            U8 /* "u8" */,
            I8 /* "i8" */,
            U16 /* "u16" */,
            I16 /* "i16" */,
            U32 /* "u32" */,
            I32 /* "i32" */,
            U64 /* "u64" */,
            I64 /* "i64" */,
            Float /* "float" */,
            Array /* "array" */,
        };

        enum EParameterId : ui32 {
            PI_ID = 0 /* "id" */,
            PI_EVENT_ID = 1 /* "event_id" */,
            PI_TIMESTAMP = 2 /* "timestamp" */,
            PI_DEVICE_STATUS = 3 /* "device_status" */,
            PI_MODULE_STATUS_1 = 4 /* "module_status_1" */,
            PI_MODULE_STATUS_2 = 5 /* "module_status_2" */,
            PI_GSM_LEVEL = 6 /* "gsm_level" */,
            PI_GPS_STATUS = 7 /* "gps_status" */,
            PI_GPS_VALID_TIMESTAMP = 8 /* "gps_valid_timestamp" */,
            PI_LATITUDE = 9 /* "latitude" */,
            PI_LONGITUDE = 10 /* "longitude" */,
            PI_HEIGHT = 11 /* "height" */,
            PI_SPEED = 12 /* "speed" */,
            PI_COURSE = 13 /* "course" */,
            PI_CURRENT_MILEAGE = 14 /* "current_mileage" */,
            PI_MILEAGE_BETWEEN_CURRENT_AND_LAST = 15 /* "mileage_between_current_and_last" */,
            PI_TIMESTAMP_BETWEEN_CURRENT_AND_LAST = 16 /* "timestamp_between_current_and_last" */,
            PI_DIFFERENCE_TIMESTAMP_MILEAGE = 17 /* "difference_timestamp_mielage" */,
            PI_MAIN_VOLTAGE = 18 /* "main_voltage" */,
            PI_ADDITIONAL_VOLTAGE = 19 /* "additional_voltage" */,
            PI_ANALOG_INPUT_1 = 20 /* "analog_input_1" */,
            PI_ANALOG_INPUT_2 = 21 /* "analog_input_2" */,
            PI_ANALOG_INPUT_3 = 22 /* "analog_input_3" */,
            PI_ANALOG_INPUT_4 = 23 /* "analog_input_4" */,
            PI_ANALOG_INPUT_5 = 24 /* "analog_input_5" */,
            PI_ANALOG_INPUT_6 = 25 /* "analog_input_6" */,
            PI_ANALOG_INPUT_7 = 26 /* "analog_input_7" */,
            PI_ANALOG_INPUT_8 = 27 /* "analog_input_8" */,
            PI_DIGITAL_INPUT_CHUNK_1 = 28 /* "digital_input_chunk_1" */,
            PI_DIGITAL_INPUT_CHUNK_2 = 29 /* "digital_input_chunk_2" */,
            PI_DIGITAL_OUTPUT_CHUNK_1 = 30 /* "digital_output_chunk_1" */,
            PI_DIGITAL_OUTPUT_CHUNK_2 = 31 /* "digital_output_chunk_2" */,
            PI_IMPULSE_COUNTER_1 = 32 /* "impulse_counter_1" */,
            PI_IMPULSE_COUNTER_2 = 33 /* "impulse_counter_2" */,
            PI_FREQUENCY_SENSOR_1 = 34 /* "frequency_sensor_1" */,
            PI_FREQUENCY_SENSOR_2 = 35 /* "frequency_sensor_2" */,
            PI_ENGINE_HOUR = 36 /* "engine_hour" */,
            PI_FUEL_LEVEL_1 = 37 /* "fuel_level_1" */,
            PI_FUEL_LEVEL_2 = 38 /* "fuel_level_2" */,
            PI_FUEL_LEVEL_3 = 39 /* "fuel_level_3" */,
            PI_FUEL_LEVEL_4 = 40 /* "fuel_level_4" */,
            PI_FUEL_LEVEL_5 = 41 /* "fuel_level_5" */,
            PI_FUEL_LEVEL_6 = 42 /* "fuel_level_6" */,
            PI_FUEL_LEVEL_RS232 = 43 /* "fuel_level_rs232" */,
            PI_TEMPERATURE_SENSOR_1 = 44 /* "temperature_sensor_1" */,
            PI_TEMPERATURE_SENSOR_2 = 45 /* "temperature_sensor_2" */,
            PI_TEMPERATURE_SENSOR_3 = 46 /* "temperature_sensor_3" */,
            PI_TEMPERATURE_SENSOR_4 = 47 /* "temperature_sensor_4" */,
            PI_TEMPERATURE_SENSOR_5 = 48 /* "temperature_sensor_5" */,
            PI_TEMPERATURE_SENSOR_6 = 49 /* "temperature_sensor_6" */,
            PI_TEMPERATURE_SENSOR_7 = 50 /* "temperature_sensor_7" */,
            PI_TEMPERATURE_SENSOR_8 = 51 /* "temperature_sensor_8" */,
            PI_CAN_FUEL = 52 /* "can_fuel" */,
            PI_CAN_ENGINE_RPM = 53 /* "can_engine_rpm" */,
            PI_CAN_FUEL_DISTANCE = 54 /* "can_fuel_distance" */,
            PI_COOLANT_TEMPERATURE = 55 /* "coolant_temperature" */,
            PI_CAN_TOTAL_DISTANCE = 56 /* "total_distance" */,
            PI_CAN_AXLE_1 = 57 /* "can_axle_1" */,
            PI_CAN_AXLE_2 = 58 /* "can_axle_2" */,
            PI_CAN_AXLE_3 = 59 /* "can_axle_3" */,
            PI_CAN_AXLE_4 = 60 /* "can_axle_4" */,
            PI_CAN_AXLE_5 = 61 /* "can_axle_5" */,
            PI_CAN_ACCELERATOR = 62 /* "can_accelerator" */,
            PI_CAN_BREAK = 63 /* "can_break" */,
            PI_CAN_ENGINE_LOAD = 64 /* "can_engine_load" */,
            PI_CAN_DIESEL_FILTER_LEVEL = 65 /* "can_diesel_filter_level" */,
            PI_CAN_ENGINE_UPTIME = 66 /* "can_engine_uptime" */,
            PI_CAN_DISTANCE_TO_MAINTENANCE = 67 /* "can_distance_to_maintenance" */,
            PI_CAN_SPEED = 68 /* "can_speed" */,
            PI_NAVIGATION_INFO = 69 /* "navigation_info" */,
            PI_HDOP_PDOP = 70 /* "hdop_pdop" */,
            PI_ADDITIONAL_GPS_STATUS = 71 /* "additional_gps_status" */,
            PI_ADDITIONAL_LATITUDE_LONGITUDE = 72 /* "additional_latitude_longitude" */,
            PI_ADDITIONAL_HEIGHT = 73 /* "additional_height" */,
            PI_ADDITIONAL_COURSE = 74 /* "additional_course" */,
            PI_ADDITIONAL_SPEED = 75 /* "additional_speed" */,
            PI_LBS_INFO = 76 /* "lbs_info" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_1 = 77 /* "temperature_from_fuel_level_sensor_1" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_2 = 78 /* "temperature_from_fuel_level_sensor_2" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_3 = 79 /* "temperature_from_fuel_level_sensor_3" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_4 = 80 /* "temperature_from_fuel_level_sensor_4" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_5 = 81 /* "temperature_from_fuel_level_sensor_5" */,
            PI_TEMPEARTURE_FROM_FUEL_LEVEL_SENSOR_6 = 82 /* "temperature_from_fuel_level_sensor_6" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_7 = 83 /* "fuel_level_and_temperature_7" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_8 = 84 /* "fuel_level_and_temperature_8" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_9 = 85 /* "fuel_level_and_temperature_9" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_10 = 86 /* "fuel_level_and_temperature_10" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_11 = 87 /* "fuel_level_and_temperature_11" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_12 = 88 /* "fuel_level_and_temperature_12" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_13 = 89 /* "fuel_level_and_temperature_13" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_14 = 90 /* "fuel_level_and_temperature_14" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_15 = 91 /* "fuel_level_and_temperature_15" */,
            PI_FUEL_LEVEL_AND_TEMPERATURE_16 = 92 /* "fuel_level_and_temperature_16" */,
            PI_PRESSURE_SENSOR_FROM_1_TO_2 = 93 /* "pressure_sensor_1_to_2" */,
            PI_PRESSURE_SENSOR_FROM_3_TO_6 = 94 /* "pressure_sensor_3_to_6" */,
            PI_PRESSURE_SENSOR_FROM_7_TO_14 = 95 /* "pressure_sensor_7_to_14" */,
            PI_PRESSURE_SENSOR_FROM_15_TO_30 = 96 /* "pressure_sensor_15_to_30" */,
            PI_TAHO_INFO = 97 /* "taho_info" */,
            PI_TAHO_STATUS = 98 /* "taho_status" */,
            PI_TAHO_FLAGS = 99 /* "taho_flags" */,
            PI_TAHO_SPEED = 100 /* "taho_speed" */,
            PI_TAHO_ODOMETER = 101 /* "taho_odometer" */,
            PI_TAHO_TIME = 102 /* "taho_time" */,
            PI_DRIVER_STATUS = 103 /* "driver_status" */,
            PI_DISPLAY_MESSAGE_ID = 104 /* "display_message_id" */,
            PI_DIFFERENCE_TIME = 105 /* "difference_time" */,
            PI_AXIS_X_Y_Z = 106 /* "axis_x_y_z" */,
            PI_VECTOR_MODULE_ACCELERETION = 107 /* "vector_module_acceleration" */,
            PI_MAXIMUM_MINIMUM_ACCELERATION = 108 /* "maximum_minimum_acceleration" */,
            PI_PASSENGER_COUNTER_1_2 = 109 /* "passenger_counter_1_2" */,
            PI_PASSENGER_COUNTER_3_4 = 110 /* "passenger_counter_3_4" */,
            PI_PASSENGER_COUNTER_5_6 = 111 /* "passenger_counter_5_6" */,
            PI_PASSENGER_COUNTER_7_8 = 112 /* "passenger_counter_7_8" */,
            PI_PASSENGER_COUNTER_9_10 = 113 /* "passenger_counter_9_10" */,
            PI_PASSENGER_COUNTER_11_12 = 114 /* "passenger_counter_11_12" */,
            PI_PASSENGER_COUNTER_13_14 = 115 /* "passenger_counter_13_14" */,
            PI_PASSENGER_COUNTER_15_16 = 116 /* "passenger_counter_15_16" */,
            PI_INFORMATOR_STATUS = 117 /* "informator_status" */,
            PI_GEOZONE_LAST_ID = 118 /* "geozone_last_id" */,
            PI_LAST_STOP_ID = 119 /* "last_stop_id" */,
            PI_ROUTE_ID = 120 /* "route_id" */,
            PI_CAMERA_STATUS = 121 /* "camera_status" */,
            PI_DEVICE_STATUS_2 = 122 /* "device_status_2" */,
            PI_FUNCTIONAL_MODULE_STATUS = 123 /* "functional_module_status" */,
            PI_CONNECTION_STATUS = 124 /* "connection_status" */,
            PI_DIGITAL_SENSORS_VALUE = 125 /* "digital_sensors_value" */,
            PI_IMPULSE_SENSOR_VALUE_3 = 126 /* "impulse_sensor_value_3" */,
            PI_IMPULSE_SENSOR_VALUE_4 = 127 /* "impulse_sensor_value_4" */,
            PI_IMPULSE_SENSOR_VALUE_5 = 128 /* "impulse_sensor_value_5" */,
            PI_IMPULSE_SENSOR_VALUE_6 = 129 /* "impulse_sensor_value_6" */,
            PI_IMPULSE_SENSOR_VALUE_7 = 130 /* "impulse_sensor_value_7" */,
            PI_IMPULSE_SENSOR_VALUE_8 = 131 /* "impulse_sensor_value_8" */,
            PI_FREQUENCY_SENSOR_3 = 132 /* "frequency_sensor_3" */,
            PI_FREQUENCY_SENSOR_4 = 133 /* "frequency_sensor_4" */,
            PI_FREQUENCY_SENSOR_5 = 134 /* "frequency_sensor_5" */,
            PI_FREQUENCY_SENSOR_6 = 135 /* "frequency_sensor_6" */,
            PI_FREQUENCY_SENSOR_7 = 136 /* "frequency_sensor_7" */,
            PI_FREQUENCY_SENSOR_8 = 137 /* "frequency_sensor_8" */,
            PI_VIRTUAL_ACCELEROMETER = 138 /* "virtual_accelerometer" */,
            PI_INTERNAL_TILT_LOCAL_VERTICAL = 139 /* "internal_tilt_local_vertical" */,
            PI_INTERNAL_TILT_SHEER_VERTICAL = 140 /* "internal_tilt_sheer_vertical" */,
            PI_EXTERNAL_TILT = 141 /* "external_tilt" */,
            PI_ECO_DRIVING_MAXIMUM_ACCELERATION = 142 /* "eco_driving_maximum_acceleration" */,
            PI_ECO_DRIVING_MAXIMUM_SPEED = 143 /* "eco_driving_maximum_speed" */,
            PI_ECO_DRIVING_STEP_SPEED = 144 /* "eco_driving_step_speed" */,
            PI_ECO_DRIVING_STEP_ACCELERATION = 145 /* "eco_driving_step_acceleration" */,
            PI_FUEL_FREQUENCY_OUTPUT_1 = 146 /* "fuel_frequency_output_1" */,
            PI_FUEL_FREQUENCY_OUTPUT_2 = 147 /* "fuel_frequency_output_2" */,
            PI_FUEL_FREQUENCY_OUTPUT_3 = 148 /* "fuel_frequency_output_3" */,
            PI_FUEL_FREQUENCY_OUTPUT_4 = 149 /* "fuel_frequency_output_4" */,
            PI_FUEL_FREQUENCY_OUTPUT_5 = 150 /* "fuel_frequency_output_5" */,
            PI_FUEL_FREQUENCY_OUTPUT_6 = 151 /* "fuel_frequency_output_6" */,
            PI_FUEL_FREQUENCY_OUTPUT_7 = 152 /* "fuel_frequency_output_7" */,
            PI_FUEL_FREQUENCY_OUTPUT_8 = 153 /* "fuel_frequency_output_8" */,
            PI_FUEL_FREQUENCY_OUTPUT_9 = 154 /* "fuel_frequency_output_9" */,
            PI_FUEL_FREQUENCY_OUTPUT_10 = 155 /* "fuel_frequency_output_10" */,
            PI_FUEL_FREQUENCY_OUTPUT_11 = 156 /* "fuel_frequency_output_11" */,
            PI_FUEL_FREQUENCY_OUTPUT_12 = 157 /* "fuel_frequency_output_12" */,
            PI_FUEL_FREQUENCY_OUTPUT_13 = 158 /* "fuel_frequency_output_13" */,
            PI_FUEL_FREQUENCY_OUTPUT_14 = 159 /* "fuel_frequency_output_14" */,
            PI_FUEL_FREQUENCY_OUTPUT_15 = 160 /* "fuel_frequency_output_15" */,
            PI_FUEL_FREQUENCY_OUTPUT_16 = 161 /* "fuel_frequency_output_16" */,
            PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_1 = 162 /* "high_resolution_temperature_sensor_1" */,
            PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_2 = 163 /* "high_resolution_temperature_sensor_2" */,
            PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_3 = 164 /* "high_resolution_temperature_sensor_3" */,
            PI_HIGH_RESOLUTION_TEMPERATURE_SENSOR_4 = 165 /* "high_resolution_temperature_sensor_4" */,
            PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_1 = 166 /* "high_resolution_humidity_sensor_1" */,
            PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_2 = 167 /* "high_resolution_humidity_sensor_2" */,
            PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_3 = 168 /* "high_resolution_humidity_sensor_3" */,
            PI_HIGH_RESOLUTION_HUMIDITY_SENSOR_4 = 169 /* "high_resolution_humidity_sensor_4" */,
            PI_FUEL_FLOW_SENSOR_STATUS = 170 /* "fuel_flow_sensor_status" */,
            PI_FUEL_FLOW_SENSOR_ERROR = 171 /* "fuel_flow_sensor_error" */,
            PI_FUEL_FLOW_SENSOR_TOTAL = 172 /* "fuel_flow_sensor_total" */,
            PI_FUEL_FLOW_SENSOR_BY_SESSION = 173 /* "fuel_flow_sensor_by_session" */,
            PI_FUEL_FLOW_SENSOR_CURRENT = 174 /* "fuel_flow_sensor_current" */,
            PI_FUEL_FLOW_SENSOR_TOTAL_VOLUME = 175 /* "fuel_flow_sensor_total_volume" */,
            PI_FUEL_FLOW_SENSOR_SPEED = 176 /* "fuel_flow_sensor_speed" */,
            PI_FUEL_FLOW_SENSOR_TEMPERATURE = 177 /* "fuel_flow_sensor_temperature" */,
            PI_FUEL_FLOW_SENSOR_TOTAL_VOLUME_REVERSE = 178 /* "fuel_flow_sensor_total_volume_reverse" */,
            PI_FUEL_FLOW_SENSOR_SPEED_REVERSE = 179 /* "fuel_flow_sensor_speed_reverse" */,
            PI_FUEL_FLOW_SENSOR_TEMPERATURE_REVERSE = 180 /* "fuel_flow_sensor_temperature_reverse" */,
            PI_REFRIGERATOR_STATUS = 181 /* "refrigerator_status" */,
            PI_REFRIGERATOR_TEMPERATURE_1 = 182 /* "refrigerator_temperature_1" */,
            PI_REFRIGERATOR_TEMPERATURE_2 = 183 /* "refrigerator_temperature_2" */,
            PI_REFRIGERATOR_TEMPERATURE_3 = 184 /* "refrigerator_temperature_3" */,
            PI_REFRIGERATOR_DIRECT_TEMPERATURE_1 = 185 /* "refrigerator_direct_temperature_1" */,
            PI_REFRIGERATOR_DIRECT_TEMPERATURE_2 = 186 /* "refrigerator_direct_temperature_2" */,
            PI_REFRIGERATOR_DIRECT_TEMPERATURE_3 = 187 /* "refrigerator_direct_temperature_3" */,
            PI_REFRIGERATOR_OUTSIDE_TEMPERATURE = 188 /* "refrigerator_outside_temperature" */,
            PI_REFRIGERATOR_TEMPERATURE = 189 /* "refrigerator_temperature" */,
            PI_REFRIGERATOR_VOLTAGE = 190 /* "refrigerator_voltage" */,
            PI_REFRIGERATOR_AMPERAGE = 191 /* "refrigerator_amperage" */,
            PI_REFRIGERATOR_ENGINE_HOUR = 192 /* "refrigerator_engine_hour" */,
            PI_REFRIGERATOR_HOUR = 193 /* "refrigerator_hour" */,
            PI_REFRIGERATOR_ERROR_COUNT = 194 /* "refrigerator_error_count" */,
            PI_REFRIGERATOR_ERROR_2 = 195 /* "refrigerator_error_2" */,
            PI_REFRIGERATOR_ERROR_4 = 196 /* "refrigerator_error_4" */,
            PI_REFRIGERATOR_ENGINE_STATUS = 197 /* "refrigerator_engine_status" */,
            PI_REFRIGERATOR_COMPRESSOR_SETTING = 198 /* "refrigerator_compressor_setting" */,
            PI_GEOZONE_STATUS = 199 /* "geozone_status" */,
            PI_CAN_FLAGS = 200 /* "can_flags" */,
            PI_CAN_EVENT = 201 /* "can_event" */,
            PI_CAN_EMERGENCY_INDICATION = 202 /* "can_emergency_indication" */,
            PI_ERROR = 203 /* "error" */,
            PI_CUSTOMER_ENGINE_HOUR = 204 /* "customer_engine_hour" */,
            PI_DIAGNOSTIC_CODE = 205 /* "diagnostic_code" */,
            PI_CUSTOMER_PARAMETER_1_1 = 206 /* "customer_parameter_1_1" */,
            PI_CUSTOMER_PARAMETER_1_2 = 207 /* "customer_parameter_1_2" */,
            PI_CUSTOMER_PARAMETER_1_3 = 208 /* "customer_parameter_1_3" */,
            PI_CUSTOMER_PARAMETER_1_4 = 209 /* "customer_parameter_1_4" */,
            PI_CUSTOMER_PARAMETER_1_5 = 210 /* "customer_parameter_1_5" */,
            PI_CUSTOMER_PARAMETER_1_6 = 211 /* "customer_parameter_1_6" */,
            PI_CUSTOMER_PARAMETER_1_7 = 212 /* "customer_parameter_1_7" */,
            PI_CUSTOMER_PARAMETER_1_8 = 213 /* "customer_parameter_1_8" */,
            PI_CUSTOMER_PARAMETER_1_9 = 214 /* "customer_parameter_1_9" */,
            PI_CUSTOMER_PARAMETER_1_10 = 215 /* "customer_parameter_1_10" */,
            PI_CUSTOMER_PARAMETER_1_11 = 216 /* "customer_parameter_1_11" */,
            PI_CUSTOMER_PARAMETER_1_12 = 217 /* "customer_parameter_1_12" */,
            PI_CUSTOMER_PARAMETER_1_13 = 218 /* "customer_parameter_1_13" */,
            PI_CUSTOMER_PARAMETER_1_14 = 219 /* "customer_parameter_1_14" */,
            PI_CUSTOMER_PARAMETER_1_15 = 220 /* "customer_parameter_1_15" */,
            PI_CUSTOMER_PARAMETER_1_16 = 221 /* "customer_parameter_1_16" */,
            PI_CUSTOMER_PARAMETER_2_1 = 222 /* "customer_parameter_2_1" */,
            PI_CUSTOMER_PARAMETER_2_2 = 223 /* "customer_parameter_2_2" */,
            PI_CUSTOMER_PARAMETER_2_3 = 224 /* "customer_parameter_2_3" */,
            PI_CUSTOMER_PARAMETER_2_4 = 225 /* "customer_parameter_2_4" */,
            PI_CUSTOMER_PARAMETER_2_5 = 226 /* "customer_parameter_2_5" */,
            PI_CUSTOMER_PARAMETER_2_6 = 227 /* "customer_parameter_2_6" */,
            PI_CUSTOMER_PARAMETER_2_7 = 228 /* "customer_parameter_2_7" */,
            PI_CUSTOMER_PARAMETER_2_8 = 229 /* "customer_parameter_2_8" */,
            PI_CUSTOMER_PARAMETER_2_9 = 230 /* "customer_parameter_2_9" */,
            PI_CUSTOMER_PARAMETER_2_10 = 231 /* "customer_parameter_2_10" */,
            PI_CUSTOMER_PARAMETER_2_11 = 232 /* "customer_parameter_2_11" */,
            PI_CUSTOMER_PARAMETER_2_12 = 233 /* "customer_parameter_2_12" */,
            PI_CUSTOMER_PARAMETER_2_13 = 234 /* "customer_parameter_2_13" */,
            PI_CUSTOMER_PARAMETER_2_14 = 235 /* "customer_parameter_2_14" */,
            PI_CUSTOMER_PARAMETER_2_15 = 236 /* "customer_parameter_2_15" */,
            PI_CUSTOMER_PARAMETER_4_1 = 237 /* "customer_parameter_4_1" */,
            PI_CUSTOMER_PARAMETER_4_2 = 238 /* "customer_parameter_4_2" */,
            PI_CUSTOMER_PARAMETER_4_3 = 239 /* "customer_parameter_4_3" */,
            PI_CUSTOMER_PARAMETER_4_4 = 240 /* "customer_parameter_4_4" */,
            PI_CUSTOMER_PARAMETER_4_5 = 241 /* "customer_parameter_4_5" */,
            PI_CUSTOMER_PARAMETER_4_6 = 242 /* "customer_parameter_4_6" */,
            PI_CUSTOMER_PARAMETER_4_7 = 243 /* "customer_parameter_4_7" */,
            PI_CUSTOMER_PARAMETER_4_8 = 244 /* "customer_parameter_4_8" */,
            PI_CUSTOMER_PARAMETER_4_9 = 245 /* "customer_parameter_4_9" */,
            PI_CUSTOMER_PARAMETER_4_10 = 246 /* "customer_parameter_4_10" */,
            PI_CUSTOMER_PARAMETER_4_11 = 247 /* "customer_parameter_4_11" */,
            PI_CUSTOMER_PARAMETER_4_12 = 248 /* "customer_parameter_4_12" */,
            PI_CUSTOMER_PARAMETER_4_13 = 249 /* "customer_parameter_4_13" */,
            PI_CUSTOMER_PARAMETER_4_14 = 250 /* "customer_parameter_4_14" */,
            PI_CUSTOMER_PARAMETER_4_15 = 251 /* "customer_parameter_4_15" */,
            PI_CUSTOMER_PARAMETER_8_1 = 252 /* "customer_parameter_8_1" */,
            PI_CUSTOMER_PARAMETER_8_2 = 253 /* "customer_parameter_8_2" */,
            PI_CUSTOMER_PARAMETER_8_3 = 254 /* "customer_parameter_8_3" */,
        };

        class TParameterId {
        public:
            TParameterId(EParameterId id, ui16 subId = 0)
                : Id(id)
                , SubId(subId)
            {
            }

            bool operator==(ui32 id) const {
                return Id == id;
            }

            bool operator==(const TParameterId& other) const {
                return Id == other.Id && SubId == other.SubId;
            }

            bool operator<(const TParameterId& other) const {
                return Id < other.Id;
            }

        private:
            EParameterId Id;
            ui16 SubId;
        };

        class TParameter {
        public:
            using TValue = std::variant<i64, ui64, double, TBuffer, TNull>;

            TParameter() = default;

            TParameter(ui64 id, ui64 subId, const TValue& value)
                : Id(id)
                , SubId(subId)
                , Value(value)
            {
            }

            TParameter(ui64 id, ui64 subId, TValue&& value)
                : Id(id)
                , SubId(subId)
                , Value(std::move(value))
            {
            }

            ui64 GetId() const {
                return Id;
            }
            ui64 GetSubId() const {
                return SubId;
            }
            const TValue& GetValue() const {
                return Value;
            }
            size_t GetSize() const;
            TString DebugString() const;

        private:
            ui64 Id = PI_ID;
            ui64 SubId = 0;
            TValue Value;
        };

        class THeader {
        public:
            ui32 ReceiverId = 0;
            ui32 TransmitterId = 0;
            ui16 DataSize = 0;
            ui8 DataChecksum = 0;

            DEFINE_FIELDS(
                ReceiverId,
                TransmitterId,
                DataSize,
                DataChecksum
            );

            TString DebugString() const;
        };

        class TMessage: public NProtocol::IMessage {
        public:
            using TBaseMessageType = NProtocol::TMessageType;
            using TMessageType = NNavTelecom::EMessageType;

        public:
            enum class ESource {
                Server,
                Client
            };

        public:
            TMessage(EMessageType type = MT_INCORRECT, ESource source = ESource::Server, TBitField bitField = TBitField());
            TMessage(THolder<NProtocol::IPayload>&& payload, TBitField&& bitField = TBitField());

            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;

            THeader& GetHeader() {
                return Header;
            }
            THeader GetHeader() const {
                return Header;
            }

        private:
            THeader Header;
            TBitField BitField;
            ESource Source = ESource::Server;

        private:
            void LoadNTCB(IInputStream& input);
            void LoadFlex(IInputStream& input);
            void SaveNTCB(IOutputStream& output) const;
            void SaveFlex(IOutputStream& output) const;
        };

        template<EMessageType Type>
        class TPayload: public NProtocol::IPayload {
        public:
            TPayload() {
                MessageType = Type;
                ProtocolType = NDrive::NProtocol::PT_NAVTELECOM;
            }

            NProtocol::TSequenceId GetSequenceId() const {
                return 0;
            }
        };

        class THandShakeRequest: public TPayload<MT_HANDSHAKE_REQUEST> {
        public:
            using TImei = NProtocol::TFixedString<15>;

            TImei IMEI;

            DEFINE_MESSAGE_FIELDS(
                IMEI
            );

            TString DebugString() const override {
                return TString("HandShake Request: ") + IMEI.Get();
            }
        };

        class THandShakeAnswer: public TPayload<MT_HANDSHAKE_ANSWER> {
        public:
            NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return "HandShake Answer";
            }
        };

        class TProtocolSettingRequest: public TPayload<MT_PROTOCOL_SETTING_REQUEST> {
        public:
            static constexpr size_t BitFieldMaximumSize = 128;

            ui8 Protocol = 0;
            EProtocolVersion ProtocolVersion = EProtocolVersion::None;
            EStructVersion StructVersion = EStructVersion::None;
            ui8 DataSize = 0;
            TBitField BitField;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;
            TString GetBitFieldDebugString() const;

        private:
            inline size_t GetBitFieldSize() const {
                auto value = StructSizes.Value(StructVersion, 0);
                return CalculateBitFieldSize(value);
            }

            inline static constexpr size_t CalculateBitFieldSize(size_t bitSize) {
                return (bitSize / 8) + 1;
            }
        };

        class TProtocolSettingAnswer: public TPayload<MT_PROTOCOL_SETTING_ANSWER> {
        public:
            ui8 Protocol = DefaultFlexProtocolId;
            EProtocolVersion ProtocolVersion;
            EStructVersion StructVersion;

            DEFINE_MESSAGE_FIELDS(
                Protocol,
                ProtocolVersion,
                StructVersion
            );

            TString DebugString() const override;
        };

        class TRecord {
        public:
            TVector<TParameter> Parameters;

            size_t GetSize() const;
            void Load(IInputStream* input, const TBitField& bitField);
            void Save(IOutputStream* output) const;
            TString DebugString() const;
            TParameter::TValue Get(EParameterId, ui64 subId = 0) const;
        };

        class TAdditional {
        public:
            ui16 TotalLength;
            ui8 StructVersion;
            ui8 DataLength;
            ui32 Number;
            ui16 EventCode;
            ui32 Timestamp;
            ui8 GpsStatus;
            ui32 GpsTimestamp;
            i32 Latitude;
            i32 Longitude;
            i32 Height;
            float Speed;
            ui16 Course;
            float Mileage;
            TVector<TParameter> Fields;

            size_t GetSize() const;
            size_t GetHeadSize() const;
            void Load(IInputStream* input);
            void Save(IOutputStream* output);
            TString DebugString() const;
            TParameter::TValue Get(ui64 id) const;
        };

        class TBlackBoxRequest: public TPayload<MT_BLACKBOX_REQUEST> {
        public:
            TBlackBoxRequest(const TBitField& bitField)
                : BitField(bitField)
            {
            }

            ui8 Count = 0;
            TVector<TRecord> Records;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;

            const TBitField& GetBitField() const {
                return BitField;
            }
            void SetBitField(const TBitField& bitField) {
                BitField = bitField;
            }

        private:
            TBitField BitField;
        };

        class TBlackBoxAnswer: public TPayload<MT_BLACKBOX_ANSWER> {
        public:
            ui8 Count = 0;

            DEFINE_MESSAGE_FIELDS(
                Count
            );

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>()) + " " + ToString(Count);
            }
        };

        class TAdditionalBlackBoxRequest: public TPayload<MT_ADDITIONAL_BLACKBOX_REQUEST> {
        public:
            ui8 Count = 0;
            TVector<TAdditional> Records;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;
        };

        class TAdditionalBlackBoxAnswer: public TPayload<MT_ADDITIONAL_BLACKBOX_ANSWER> {
        public:
            ui8 Count = 0;

            DEFINE_MESSAGE_FIELDS(
                Count
            );

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>()) + " " + ToString(Count);
            }
        };

        class TSingleBlackBoxRequest: public TPayload<MT_SINGLE_BLACKBOX_REQUEST> {
        public:
            TSingleBlackBoxRequest(const TBitField& bitField)
                : BitField(bitField)
            {
            }

            ui32 EventIndex = 0;
            TRecord Record;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;

            const TBitField& GetBitField() const {
                return BitField;
            }
            void SetBitField(const TBitField& bitField) {
                BitField = bitField;
            }

        private:
            TBitField BitField;
        };

        class TSingleBlackBoxAnswer: public TPayload<MT_SINGLE_BLACKBOX_ANSWER> {
        public:
            ui32 EventIndex = 0;

            DEFINE_MESSAGE_FIELDS(
                EventIndex
            );

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>()) + " " + ToString(EventIndex);
            }
        };

        class TAdditionalSingleBlackBoxRequest: public TPayload<MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST> {
        public:
            ui32 EventIndex = 0;
            TAdditional Record;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;
        };

        class TAdditionalSingleBlackBoxAnswer: public TPayload<MT_ADDITIONAL_SINGLE_BLACKBOX_ANSWER> {
        public:
            ui32 EventIndex = 0;

            DEFINE_MESSAGE_FIELDS(
                EventIndex
            );

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>()) + " " + ToString(EventIndex);
            }
        };

        class TOnlineBlackBoxRequest: public TPayload<MT_ONLINE_BLACKBOX_REQUEST> {
        public:
            TOnlineBlackBoxRequest(const TBitField& bitField)
                : BitField(bitField)
            {
            }

            TRecord Record;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;

            const TBitField& GetBitField() const {
                return BitField;
            }
            void SetBitField(const TBitField& bitField) {
                BitField = bitField;
            }

        private:
            TBitField BitField;
        };

        class TOnlineBlackBoxAnswer: public TPayload<MT_ONLINE_BLACKBOX_ANSWER> {
        public:
            NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>());
            }
        };

        class TPing: public TPayload<MT_PING> {
        public:
            NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return ToString(MT_PING);
            }
        };

        class TDigitalOutputCommandRequest: public TPayload<MT_DIGITAL_OUTPUT_COMMAND> {
        public:
            ui8 Number;
            bool State;

            size_t GetSize() const override {
                return sizeof(Number) + sizeof(State);
            }

            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>()) + " " + ToString(Number) + " " + ToString(State);
            }
        };

        class TDigitalOutputCommandAnswer: public TPayload<MT_DIGITAL_OUTPUT_ANSWER> {
        public:
            TDigitalOutputCommandAnswer(const TBitField& bitField)
                : BitField(bitField)
            {
            }

            TRecord Record;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;

        private:
            TBitField BitField;
        };

        class TICCIDRequest: public TPayload<MT_ICCID_REQUEST> {
        public:
            NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return ToString(GetMessageTypeAs<EMessageType>());
            }
        };

        class TICCIDAnswer: public TPayload<MT_ICCID_ANSWER> {
        public:
            TString ICCID;

            size_t GetSize() const override;
            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            TString DebugString() const override;
        };

        double CoordinateConvert(i64 coordinate);
        NDrive::TMultiSensor ToSensors(const TRecord& data);
        NDrive::TMultiSensor ToSensors(const TAdditional& data);
        NDrive::TMultiSensor ToSensors(const TICCIDAnswer& data);
    }
}
