#include "navtelecom.h"
#include "nonce.h"

#include <rtline/library/json/cast.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/str.h>
#include <util/string/hex.h>

namespace {
    void FillAnswer(NDrive::NNavTelecom::TMessage& message) {
        message.GetHeader().ReceiverId = 0;
        message.GetHeader().TransmitterId = 1;
    }

    NDrive::NNavTelecom::TMessage CreateAnswerMessage(NDrive::NNavTelecom::EMessageType type) {
        NDrive::NNavTelecom::TMessage result{type};
        FillAnswer(result);
        return std::move(result);
    }

    NDrive::NNavTelecom::TMessage CreateAnswerMessage(THolder<NDrive::NProtocol::IPayload>&& payload) {
        NDrive::NNavTelecom::TMessage result{std::move(payload)};
        FillAnswer(result);
        return result;
    }
}

Y_UNIT_TEST_SUITE(TelematicsNavtelecomProtocol) {
    Y_UNIT_TEST(Login) {
        {
            TString data = HexDecode("404E544301000000000000001300474C2A3E533A383632353331303433323034333231");
            TMemoryInput dataStream(data);

            NDrive::NNavTelecom::TMessage message;
            message.Load(dataStream);
            UNIT_ASSERT(message.IsValid());

            const auto& payload = message.As<NDrive::NNavTelecom::THandShakeRequest>();
            UNIT_ASSERT_EQUAL(payload.IMEI.Get(), "862531043204321");

            TString result;
            TStringOutput stream(result);
            message.Save(stream);

            UNIT_ASSERT_STRINGS_EQUAL(result, data);
        }

        {
            TString buffer;
            TStringOutput outputStream(buffer);
            auto message = CreateAnswerMessage(NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER);
            message.Save(outputStream);
            UNIT_ASSERT_EQUAL(HexEncode(buffer.Data(), buffer.Size()), "404E544300000000010000000300455E2A3C53");
        }
    }

    Y_UNIT_TEST(Setting) {
        {
            TString buffer = HexDecode("404E544301000000000000001A001B192A3E464C4558B014147AF2002000000000000000000000000000");
            TMemoryInput input(buffer);

            NDrive::NNavTelecom::TMessage message;
            message.Load(input);

            UNIT_ASSERT(message.IsValid());

            TString result;
            TStringOutput stream(result);
            message.Save(stream);

            UNIT_ASSERT_STRINGS_EQUAL(buffer, result);
        }

        {
            TString buffer;
            TStringOutput outputStream(buffer);

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TProtocolSettingAnswer>();
            UNIT_ASSERT(payload);

            payload->Protocol = NDrive::NNavTelecom::DefaultFlexProtocolId;
            payload->ProtocolVersion = NDrive::NNavTelecom::EProtocolVersion::Second;
            payload->StructVersion = NDrive::NNavTelecom::EStructVersion::Second;

            auto message = CreateAnswerMessage(std::move(payload));
            message.Save(outputStream);

            UNIT_ASSERT_EQUAL(HexEncode(buffer.Data(), buffer.Size()), "404E544300000000010000000900B1A02A3C464C4558B01414");
        }
    }

    Y_UNIT_TEST(BlackBox) {
        {
            NDrive::NNavTelecom::TBitField bitField;

            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_ID));
            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_EVENT_ID));
            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_TIMESTAMP));
            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_DEVICE_STATUS));
            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_GSM_LEVEL));
            bitField.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_MAIN_VOLTAGE));

            TString buffer = HexDecode("7E41011E000000B017E11D7456001F142F24");
            TMemoryInput input(buffer);

            NDrive::NNavTelecom::TMessage message(NDrive::NNavTelecom::MT_INCORRECT, NDrive::NNavTelecom::TMessage::ESource::Server, bitField);
            message.Load(input);
            UNIT_ASSERT(message.IsValid());

            auto& payload = message.As<NDrive::NNavTelecom::TBlackBoxRequest>();
            UNIT_ASSERT_EQUAL(payload.Records.size(), 1);

            const auto& record = payload.Records[0];
            const auto& parameters = record.Parameters;
            UNIT_ASSERT_EQUAL(parameters.size(), 6);

            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_ID), NDrive::NNavTelecom::TParameter::TValue(ui64(30)));
            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_EVENT_ID), NDrive::NNavTelecom::TParameter::TValue(ui64(6064)));
            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_TIMESTAMP), NDrive::NNavTelecom::TParameter::TValue(ui64(1450450401)));
            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_DEVICE_STATUS), NDrive::NNavTelecom::TParameter::TValue(ui64(0)));
            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_GSM_LEVEL), NDrive::NNavTelecom::TParameter::TValue(ui64(31)));
            UNIT_ASSERT_EQUAL(record.Get(NDrive::NNavTelecom::PI_MAIN_VOLTAGE), NDrive::NNavTelecom::TParameter::TValue(ui64(12052)));

            TString result;
            TStringOutput output(result);
            message.Save(output);

            UNIT_ASSERT_STRINGS_EQUAL(buffer, result);
        }

        {
            TString buffer;
            TStringOutput outputStream(buffer);

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TBlackBoxAnswer>();
            UNIT_ASSERT(payload);

            payload->Count = 1;
            NDrive::NNavTelecom::TMessage message(std::move(payload));
            message.Save(outputStream);

            UNIT_ASSERT_EQUAL(HexEncode(buffer.Data(), buffer.Size()), "7E4101DF");
        }

        {
            TString data = HexDecode("7E4101DF");
            TStringInput stream(data);
            NDrive::NNavTelecom::TMessage message(NDrive::NNavTelecom::EMessageType::MT_INCORRECT, NDrive::NNavTelecom::TMessage::ESource::Client);
            message.Load(stream);
            UNIT_ASSERT(message.IsValid());

            auto type = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();
            UNIT_ASSERT_EQUAL_C(type, NDrive::NNavTelecom::MT_BLACKBOX_ANSWER, ToString(type));
        }
    }

    Y_UNIT_TEST(AdditionalBlackbox) {
        {
            TString buffer = HexDecode("7E450139000A251000000030250537D75533EC36D7557E2DF9013BCC14010F070000000000000000000000000210120000000000000000000000000000349A");
            TMemoryInput stream(buffer);

            NDrive::NNavTelecom::TMessage message;
            message.Load(stream);

            UNIT_ASSERT(message.IsValid());

            auto& payload = message.As<NDrive::NNavTelecom::TAdditionalBlackBoxRequest>();
            UNIT_ASSERT_EQUAL(payload.Count,  1);
            UNIT_ASSERT_EQUAL(payload.Records.size(), 1);

            auto& record = payload.Records[0];
            UNIT_ASSERT_EQUAL(record.TotalLength, 57);
            UNIT_ASSERT_EQUAL(record.StructVersion, 0x0A);
            UNIT_ASSERT_EQUAL(record.Number, 16);
        }

        {
            TString buffer;
            TStringOutput stream(buffer);

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TAdditionalBlackBoxAnswer>();
            payload->Count = 1;

            NDrive::NNavTelecom::TMessage message(std::move(payload));
            message.Save(stream);

            UNIT_ASSERT_EQUAL(buffer, HexDecode("7E45015C"));
        }
    }

    Y_UNIT_TEST(SingleBlackBox) {
        {
            TString settingBuffer = HexDecode("404E544301000000000000002A0007352A3E464C4558B01E1EFFFBEEB00808000EFFBD2812000020000000380000000000000080000000000000");
            TStringInput settingInput(settingBuffer);

            NDrive::NNavTelecom::TMessage settingMessage;
            settingMessage.Load(settingInput);
            UNIT_ASSERT(settingMessage.IsValid());
            UNIT_ASSERT_EQUAL(settingMessage.GetMessageType(), NDrive::NNavTelecom::MT_PROTOCOL_SETTING_REQUEST);

            auto& settingPayload = settingMessage.As<NDrive::NNavTelecom::TProtocolSettingRequest>();

            TString requestBuffer = HexDecode(
                "7E5407000000070000000010D337B761" \
                "00006300D337B76130C45501F0A8FB01" \
                "00000000000000000000000000000000" \
                "00000000000080000080BFFFFF000080" \
                "BFFFFFFFFFFFFFFFFFFFFFFFFFFF0000" \
                "0000FFFFFF0000000000000000000000" \
                "00000000000000000000DD0000000000" \
                "0000000000DD00000000000000000000" \
                "DDD337B7610000000000000000000000" \
                "00000000000000F7"
            );
            TStringInput requestStream(requestBuffer);

            NDrive::NNavTelecom::TMessage requestMessage(NDrive::NNavTelecom::MT_INCORRECT, NDrive::NNavTelecom::TMessage::ESource::Server, settingPayload.BitField);
            requestMessage.Load(requestStream);

            UNIT_ASSERT(requestMessage.IsValid());
            UNIT_ASSERT_EQUAL(requestMessage.GetMessageType(), NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST);
        }
    }

    Y_UNIT_TEST(SingleAdditionalBlackbox) {
        {
            TString buffer = HexDecode("7E580700000027000A25070000000010DB37B76100DB37B76130C45501F0A8FB010000000000000000000000000000BD");
            TStringInput stream(buffer);

            NDrive::NNavTelecom::TMessage message;
            message.Load(stream);

            UNIT_ASSERT(message.IsValid());
            UNIT_ASSERT_EQUAL(message.GetMessageType(), NDrive::NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST);
        }
    }

    Y_UNIT_TEST(Ping) {
        TString buffer = HexDecode("7F");
        TMemoryInput input(buffer);

        NDrive::NNavTelecom::TMessage message;
        message.Load(input);

        UNIT_ASSERT(message.IsValid());
    }

    Y_UNIT_TEST(CoordinateConverter) {
        double longitudeExpected = 56.47250666667;
        double latitudeExpected = 85.00806666667;

        i32 longitudeBase = 33883504;
        i32 latitudeBase = 51004840;

        UNIT_ASSERT_DOUBLES_EQUAL(longitudeExpected, NDrive::NNavTelecom::CoordinateConvert(longitudeBase), 0.0000000001);
        UNIT_ASSERT_DOUBLES_EQUAL(latitudeExpected, NDrive::NNavTelecom::CoordinateConvert(latitudeBase), 0.0000000001);
    }

    Y_UNIT_TEST(DigitalOutput) {
        {
            TString data;
            TStringOutput stream(data);

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TDigitalOutputCommandRequest>();
            payload->Number = 2;
            payload->State = true;

            auto message = CreateAnswerMessage(std::move(payload));
            message.Save(stream);

            UNIT_ASSERT_EQUAL(HexEncode(data.Data(), data.Size()), "404E544300000000010000000400607C2A213259");
        }

        {
            TString data = HexDecode("404E544300000000000000000400637E2A21315920");
            TStringInput stream(data);

            NDrive::NNavTelecom::TMessage message;
            message.Load(stream);

            auto payload = message.As<NDrive::NNavTelecom::TDigitalOutputCommandRequest>();

            UNIT_ASSERT_EQUAL(payload.Number, 1);
            UNIT_ASSERT_EQUAL(payload.State, true);
        }
    }

    Y_UNIT_TEST(IccidCommand) {
        {
            TString data;
            TStringOutput stream(data);

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TICCIDRequest>();

            auto message = CreateAnswerMessage(std::move(payload));
            message.Save(stream);

            UNIT_ASSERT_EQUAL(HexEncode(data.Data(), data.Size()), "404E544300000000010000000700514E2A3F4943434944");
        }
        {
            TString data = HexDecode("404E544301000000000000001A00696B2A23494343494420383937303130323639343936303032353735");
            TStringInput stream(data);

            NDrive::NNavTelecom::TMessage message;
            message.Load(stream);

            UNIT_ASSERT(message.IsValid());
            auto& payload = message.As<NDrive::NNavTelecom::TICCIDAnswer>();
            UNIT_ASSERT_EQUAL(payload.ICCID, "897010269496002575");
        }
    }

    Y_UNIT_TEST(CollisionOfChecksum) {
        NDrive::NNavTelecom::TBitField bitField;
        {
            TString data = HexDecode("404E54432A010000000000001A00F2DA2A3E464C4558B014147AFFFE3000080000000200000000000000");
            TStringInput stream(data);

            NDrive::NNavTelecom::TMessage message;
            message.Load(stream);

            UNIT_ASSERT(message.IsValid());
            auto& payload = message.As<NDrive::NNavTelecom::TProtocolSettingRequest>();
            bitField = payload.BitField;
        }

        {
            TString data = HexDecode("7E4103BDA30D00091798F3DE6200A9001C3B96F3DE625FE8EC01ED061E03C5070000DD465B40AD0048BED14786320D107558FA00070ABEA30D000B17CAF3DE6200A9001C3BC9F3DE625BE8EC01F9061E03BB07000000000000B60049BED1478E320D10A858FA00070ABFA30D000B17FCF3DE6200A9001C37FBF3DE625BE8EC01F9061E03BB07000000000000B60049BED14795320D10DA58FA00090B18");
            TStringInput stream(data);

            NDrive::NNavTelecom::TMessage message(NDrive::NNavTelecom::MT_INCORRECT, NDrive::NNavTelecom::TMessage::ESource::Server, bitField);
            message.Load(stream);

            UNIT_ASSERT(message.IsValid());
            auto type = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();
            UNIT_ASSERT_EQUAL(type, NDrive::NNavTelecom::MT_BLACKBOX_REQUEST);
        }
    }
}
