#pragma once

#include "common.h"
#include "mio.h"
#include "types.h"

#include <util/generic/buffer.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

#include <utility>


namespace NDrive {
    struct TCommandOptions;
    class TCommonTask;
    namespace NVega {
        class TTaskPtr;
        using TCommandCode = ui8;
        enum ECommandCode : TCommandCode;
    }
    namespace NProtocol {
        enum EProtocolType: ui8 {
            PT_INVALID = 0 /* "invalid" */,
            PT_VEGA /* "vega" */,
            PT_AUTODETECT /* "autodetect" */,
            PT_WIALON_IPS /* "wialon_ips" */,
            PT_NAVTELECOM /* "navtelecom" */,
        };

        TString GetProtocolName(EProtocolType type);

        using TSequenceId = ui64;
        using TProtocolType = EProtocolType;
        using TMessageType = ui64;

        class TArgument;
        class IMessage; // we can't define IMessage in "mio.h" because "mio.h" included in "protocol.h"
        using TMessageInput = TMessageInput<IMessage>;
        using TMessageOutput = TMessageOutput<IMessage>;

        using TTaskPtr = TIntrusivePtr<NDrive::TCommonTask>;;
        using ECommandCode = NVega::ECommandCode;
        using TCommandOptions = NDrive::TCommandOptions;

        class IPayload {
        public:
            virtual ~IPayload() = default;

            virtual TString DebugString() const = 0;
            virtual TSequenceId GetSequenceId() const = 0;
            virtual size_t GetSize() const = 0;
            virtual void Load(IInputStream& input) = 0;
            virtual void Save(IOutputStream& output) const = 0;
            virtual void PostLoad(IInputStream& input) { Y_UNUSED(input); }
            virtual void PostSave(IOutputStream& output) const { Y_UNUSED(output); }

            template<typename TMessageTypeResult>
            TMessageTypeResult GetMessageTypeAs() const {
                return static_cast<TMessageTypeResult>(GetMessageType());
            }

            TMessageType GetMessageType() const {
                return MessageType;
            }
            TProtocolType GetProtocolType() const {
                return ProtocolType;
            }

            template<class T, typename... Args>
            static THolder<T> Create(Args&&... args) {
                static_assert(std::is_base_of<IPayload, T>::value, "not base of IPayload");

                auto result = MakeHolder<T>(std::forward<Args>(args)...);
                return std::move(result);
            }

        protected:
            TMessageType MessageType;
            TProtocolType ProtocolType;
        };

        class IMessage {
        public:
            IMessage() = default;
            IMessage(IMessage&& other) noexcept = default;

            IMessage& operator=(IMessage&& other) noexcept = default;

            virtual ~IMessage() = default;

            template<class T>
            const T& As() const {
                Y_ENSURE(Payload);
                return *VerifyDynamicCast<const T*>(Payload.Get());
            }

            template<class T>
            T& As() {
                Y_ENSURE(Payload);
                return *VerifyDynamicCast<T*>(Payload.Get());
            }

            template<class T>
            T GetMessageTypeAs() const {
                return Payload ? Payload->GetMessageTypeAs<T>() : static_cast<T>(0);
            }

            TString DebugString() const {
                return Payload ? Payload->DebugString() : "INCORRECT";
            }
            TSequenceId GetSequenceId() const {
                return Payload ? Payload->GetSequenceId() : 0;
            }
            TMessageType GetMessageType() const {
                return Payload ? Payload->GetMessageType() : static_cast<TMessageType>(0);
            }
            TProtocolType GetProtocolType() const {
                return Payload ? Payload->GetProtocolType() : static_cast<TProtocolType>(0);
            }
            bool IsValid() const {
                return !!Payload;
            }

            virtual void Load(IInputStream& input) = 0;
            virtual void Save(IOutputStream& ouput) const = 0;

        protected:
            THolder<IPayload> Payload;
        };

        class IProtocol {
        public:
            using TSendHandler = std::function<void(const IMessage&)>;

            virtual ~IProtocol() = default;

            virtual void Launch() noexcept = 0;
            virtual THolder<NProtocol::IMessage> Load(IInputStream& stream) = 0;
            virtual void Process(const IMessage& message) = 0;
            virtual void Drop() noexcept = 0;
            virtual TTaskPtr CreateCommand(const TString& id, ECommandCode command, TArgument argument, const TCommandOptions& options) = 0;

            virtual void Command(THolder<NProtocol::IMessage>&&) {
            }

            template<class T>
            const T& As() const {
                return *VerifyDynamicCast<const T*>(this);
            }

            template<class T, typename... Args>
            static THolder<IProtocol> Create(TSendHandler&& handler, Args&&... args) {
                static_assert(std::is_base_of<IProtocol, T>::value, "not base of IProtocol");

                auto result = MakeHolder<T>(std::forward<Args>(args)...);
                result->SendHandler = handler;
                return std::move(result);
            }

            template<class TMessage>
            void Send(THolder<IPayload>&& payload) {
                static_assert(std::is_base_of<IMessage, TMessage>::value, "Error verification message and protocol type");
                auto message = TMessage(std::move(payload));
                if (SendHandler) {
                    SendHandler(std::move(message));
                }
            }

            EProtocolType GetType() const;
            TString DebugString() const;

        protected:
            EProtocolType Type;
            TSendHandler SendHandler;
        };

        class TArgument {
        private:
            TArray<ui8, ui8> Value;

        public:
            DEFINE_FIELDS(Value);

        public:
            inline bool IsNull() const {
                return Value.empty();
            }

            template <class T>
            inline T Get() const {
                T result;
                TMemoryInput input(Value.data(), Value.size());
                result.Load(&input);
                Y_ENSURE(input.Exhausted());
                return result;
            }

            template <class T>
            inline bool TryGet(T& result, TString* error = nullptr) const try {
                TMemoryInput input(Value.data(), Value.size());
                result.Load(&input);
                Y_ENSURE(input.Exhausted());
                return true;
            } catch (const std::exception& e) {
                if (error) {
                    *error = FormatExc(e);
                }
                return false;
            }

            template <class T>
            TMaybe<T> TryGet() const {
                T result;
                if (!TryGet(result)) {
                    return {};
                }
                return result;
            }

            template <class T>
            inline void Set(const T& value) {
                Value.resize(value.CalcSize());
                TMemoryOutput output(Value.data(), Value.size());
                value.Save(&output);
            }

            template <class TContainer>
            inline void SetData(const TContainer& value) {
                Value = { value.begin(), value.end() };
            }

            TString DebugString() const;
        };
    }
}
