#include "sensor.h"
#include "settings.h"
#include "vega.h"

#include <drive/telematics/protocol/proto/sensor.pb.h>
#include <rtline/library/json/cast.h>

#include <util/stream/output.h>
#include <util/string/hex.h>
#include <util/string/type.h>

#include <cmath>

bool NDrive::TSensorId::TryFromJson(const NJson::TJsonValue& value) {
    if (value.IsString()) {
        return TryFromString(value.GetString(), *this);
    }
    if (value.IsUInteger()) {
        return NJson::TryFromJson(value, Id);
    }
    unsigned long long id;
    if (value["id"].GetUInteger(&id)) {
        if (id > Max<decltype(Id)>()) {
            return false;
        }
        Id = id;
    } else {
        return false;
    }
    if (value.Has("subid")) {
        unsigned long long subid;
        if (value["subid"].GetUInteger(&subid)) {
            if (subid > Max<decltype(SubId)>()) {
                return false;
            }
            SubId = subid;
        } else {
            return false;
        }
    }
    return true;
}

bool NDrive::TSensorId::FromProto(const NDrive::NProto::TSensor& proto) {
    Id = proto.GetId();
    if (proto.HasSubId()) {
        SubId = proto.GetSubId();
    }
    return true;
}

NDrive::NProto::TSensor NDrive::TSensorId::ToProto() const {
    NDrive::NProto::TSensor result;
    result.SetId(Id);
    if (SubId) {
        result.SetSubId(SubId);
    }
    return result;
}

NJson::TJsonValue NDrive::TSensorId::ToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    if (SubId) {
        result["subid"] = SubId;
    }
    return result;
}

NJson::TJsonValue NDrive::TSensorId::GetJsonReport() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    if (SubId) {
        result["subid"] = SubId;
    }
    result["name"] = GetName();
    return result;
}

TString NDrive::TSensorId::GetName() const {
    return ToString(*this);
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::TSensorId>(const NDrive::TSensorId& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::TSensorValue>(const NDrive::TSensorValue& object) {
    return NDrive::TSensorValueOperator<NDrive::TSensorValue>::ToJson(object);
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::TSensor>(const NDrive::TSensor& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<NDrive::TSensorId>(const NJson::TJsonValue& value, NDrive::TSensorId& result) {
    return result.TryFromJson(value);
}

template <>
bool NJson::TryFromJson<NDrive::TSensor>(const NJson::TJsonValue& value, NDrive::TSensor& result) {
    return result.TryFromJson(value);
}

template <>
void Out<NDrive::TSensorId>(IOutputStream& out, const NDrive::TSensorId& value) {
    TStringBuf native = NDrive::NVega::GetSensorName(value);
    if (native) {
        out << native;
        return;
    }
    native = NDrive::NVega::GetSensorName(NDrive::TSensorId{value.Id, 0});
    if (native) {
        out << native;
    } else {
        out << value.Id;
    }
    if (value.SubId) {
        out << '-' << value.SubId;
    }
}

template <>
void Out<NDrive::TSensor>(IOutputStream& out, const NDrive::TSensor& value) {
    out << value.ToJson().GetStringRobust();
}

template <>
NDrive::TSensorId FromStringImpl(const char* data, size_t len) {
    NDrive::TSensorId result;
    TStringBuf s(data, len);
    if (s.Contains('-')) {
        result.Id = FromString<NDrive::TSensorId>(s.Before('-')).Id;
        result.SubId = FromString<ui16>(s.After('-'));
    } else {
        result = NDrive::NVega::GetSensorId(s);
        if (!result.Id) {
            result.Id = FromString<ui16>(s);
        }
    }
    return result;
}

template <>
bool TryFromStringImpl<NDrive::TSensorId>(const char* data, size_t len, NDrive::TSensorId& result) {
    TStringBuf s(data, len);
    if (s.Contains('-')) {
        return TryFromString(s.Before('-'), result) && TryFromString(s.After('-'), result.SubId);
    } else {
        auto sensorId = NDrive::NVega::GetSensorId(s);
        if (sensorId) {
            result = sensorId;
            return true;
        }
        if (TryFromString<ui16>(s, result.Id)) {
            result.SubId = 0;
            return true;
        }
        return false;
    }
}

bool NDrive::TSensor::TryFromJson(const NJson::TJsonValue& value) {
    if (!TBase::TryFromJson(value)) {
        return false;
    }
    unsigned long long since;
    if (value["since"].GetUInteger(&since)) {
        Since = TInstant::Seconds(since);
    } else {
        Since = Timestamp;
    }
    if (Since > Timestamp) {
        return false;
    }
    return true;
}

bool NDrive::TSensor::FromProto(const NDrive::NProto::TSensor& proto) {
    if (!TBase::FromProto(proto)) {
        return false;
    }
    if (proto.HasSince()) {
        Since = TInstant::Seconds(proto.GetSince());
    } else {
        Since = Timestamp;
    }
    if (Since > Timestamp) {
        return false;
    }
    return true;
}

NDrive::NProto::TSensor NDrive::TSensor::ToProto() const {
    NDrive::NProto::TSensor result = TBase::ToProto();
    if (Since != Timestamp) {
        result.SetSince(Since.Seconds());
    }
    return result;
}

NDrive::TSensor::TSensor(const TSensorId id)
    : TBase(id)
{
}

NJson::TJsonValue NDrive::TSensor::FormatValue() const {
    if (std::holds_alternative<TBuffer>(Value)) {
        return NDrive::NVega::TSetting::FormatValue(std::get<TBuffer>(Value), Id);
    } else {
        return GetJsonValue();
    }
}

NJson::TJsonValue NDrive::TSensor::ToJson() const {
    NJson::TJsonValue result = TBase::ToJson();
    if (Since) {
        result["since"] = Since.Seconds();
    }
    return result;
}

NJson::TJsonValue NDrive::TSensor::GetJsonReport() const {
    NJson::TJsonValue result = TBase::GetJsonReport();
    if (Since) {
        result["since"] = Since.Seconds();
    }
    return result;
}

bool NDrive::TryFromString(const TString& value, NDrive::TSensorValue& result, NDrive::TSensorId sensorId) noexcept {
    double resultDouble;
    ui64 resultInteger;
    NDrive::NVega::EValueType type = NDrive::NVega::GetValueType(sensorId);
    if (type != NDrive::NVega::EValueType::ASCII && value == "null") {
        result = TNull();
        return true;
    }
    switch (type) {
        case NDrive::NVega::EValueType::ASCII:
            result = value;
            return true;
        case NDrive::NVega::EValueType::Binary: try {
            TString decoded = HexDecode(value);
            TBuffer buffer(decoded.data(), decoded.size());
            result = std::move(buffer);
            return true;
        } catch (const std::exception& e) {
            Cdbg << "cannot parse Binary SensorValue: " << value << ": " << FormatExc(e) << Endl;
            return false;
        }
        case NDrive::NVega::EValueType::Float32:
            if (TryFromString(value, resultDouble)) {
                result = resultDouble;
                return true;
            } else {
                return false;
            }
        case NDrive::NVega::EValueType::UI8:
        case NDrive::NVega::EValueType::UI16:
        case NDrive::NVega::EValueType::UI32:
            if (TryFromString(value, resultInteger)) {
                result = resultInteger;
                return true;
            } else {
                return true;
            }
        case NDrive::NVega::EValueType::Unknown:
            if (TryFromString(value, resultDouble)) {
                result = resultDouble;
                return true;
            }
            result = value;
            return true;
    }
}

NDrive::TSensorValue NDrive::SensorValueFromJson(const NJson::TJsonValue& value, NDrive::TSensorId sensorId) {
    NDrive::TSensorValue result = TNull();
    Y_ENSURE(NDrive::TSensorValueOperator<NDrive::TSensorValue>::TryFromJson(value, result, sensorId.Id, sensorId.SubId), "cannot parse SensorValue " << sensorId.ToJson().GetStringRobust() << " from json: " << value.GetStringRobust());
    return result;
}

NDrive::TSensorValue NDrive::SensorValueFromString(const TString& value, NDrive::TSensorId sensorId) {
    NDrive::TSensorValue result = TNull();
    Y_ENSURE(TryFromString(value, result, sensorId), "cannot parse SensorValue " << sensorId.ToJson().GetStringRobust() << " from string: " << value);
    return result;
}
