#pragma once

#include "sensor_value.h"

#include <rtline/util/types/timestamp.h>

#include <library/cpp/json/json_value.h>

#include <util/generic/maybe.h>

namespace NDrive::NProto {
    class TSensor;
}

namespace NDrive {
    struct TSensorId {
    public:
        ui16 Id = 0;
        ui16 SubId = 0;

    public:
        constexpr TSensorId() = default;
        constexpr TSensorId(ui16 id, ui16 subid = 0)
            : Id(id)
            , SubId(subid)
        {
        }

        explicit operator bool() const {
            return Id || SubId;
        }
        inline bool operator<(const TSensorId other) const {
            return std::make_pair(Id, SubId) < std::make_pair(other.Id, other.SubId);
        }
        inline bool operator==(const TSensorId other) const {
            return std::make_pair(Id, SubId) == std::make_pair(other.Id, other.SubId);
        }

        bool TryFromJson(const NJson::TJsonValue& value);
        bool FromProto(const NDrive::NProto::TSensor& proto);

        NDrive::NProto::TSensor ToProto() const;
        NJson::TJsonValue ToJson() const;
        NJson::TJsonValue GetJsonReport() const;

        TString GetName() const;
    };

    template <class TValue>
    struct TSensorImpl: public TSensorId {
    private:
        using TBase = TSensorId;

    public:
        TValue Value = TNull();
        TTimestamp Timestamp = TInstant::Zero();

    public:
        TSensorImpl() = default;
        explicit TSensorImpl(const TSensorId id)
            : TBase(id)
        {
        }

        NJson::TJsonValue GetJsonValue() const {
            return TSensorValueOperator<TValue>::ToJson(Value);
        }

        bool IsZero() const {
            return TSensorValueOperator<TValue>::IsZero(Value);
        }

        template <class T>
        inline T ConvertTo(TMaybe<T> fallback = {}) const {
            return TSensorValueOperator<TValue>::ConvertTo(Value, fallback);
        }

        template <class T>
        inline TMaybe<T> TryConvertTo() const {
            T result;
            if (TSensorValueOperator<TValue>::TryConvertTo(Value, result)) {
                return result;
            }
            return {};
        }

        template <class T>
        inline bool TryConvertTo(T& result) const {
            return TSensorValueOperator<TValue>::TryConvertTo(Value, result);
        }

        bool TryFromJson(const NJson::TJsonValue& value) {
            unsigned long long timestamp;
            if (!TBase::TryFromJson(value)) {
                return false;
            }
            if (value["timestamp"].GetUInteger(&timestamp) && TSensorValueOperator<TValue>::TryFromJson(value["value"], Value, Id)) {
                Timestamp = TInstant::Seconds(timestamp);
            } else {
                return false;
            }
            return true;
        }

        bool FromProto(const NDrive::NProto::TSensor& proto) {
            if (!TBase::FromProto(proto)) {
                return false;
            }
            Timestamp = TInstant::Seconds(proto.GetTimestamp());
            return TSensorValueOperator<TValue>::FromProto(proto, Value);
        }

        NDrive::NProto::TSensor ToProto() const {
            NDrive::NProto::TSensor result = TBase::ToProto();
            result.SetTimestamp(Timestamp.Seconds());
            TSensorValueOperator<TValue>::ToProto(Value, result);
            return result;
        }

        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue result = TBase::ToJson();
            result["value"] = TSensorValueOperator<TValue>::ToJson(Value);
            result["timestamp"] = Timestamp.Seconds();
            return result;
        }

        NJson::TJsonValue GetJsonReport() const {
            NJson::TJsonValue result = TBase::GetJsonReport();
            result["value"] = TSensorValueOperator<TValue>::ToJson(Value);
            result["timestamp"] = Timestamp.Seconds();
            return result;
        }
    };

    struct TSensor: public TSensorImpl<TSensorValue> {
    private:
        using TBase = TSensorImpl<TSensorValue>;

    public:
        TTimestamp Since = TInstant::Zero();

    public:
        template <class T>
        static TMaybe<T> TryConvertTo(const TSensor* sensor) {
            return sensor ? sensor->TryConvertTo<T>() : Nothing();
        }
        template <class T>
        static TMaybe<T> TryConvertTo(const TMaybe<TSensor>& sensor) {
            return TryConvertTo<T>(sensor.Get());
        }

    public:
        TSensor() = default;
        explicit TSensor(const TSensorId id);

        using TBase::TryConvertTo;

        NJson::TJsonValue FormatValue() const;

        NJson::TJsonValue ToJson() const;
        bool TryFromJson(const NJson::TJsonValue& value);
        NJson::TJsonValue GetJsonReport() const;

        bool FromProto(const NDrive::NProto::TSensor& proto);
        NDrive::NProto::TSensor ToProto() const;

        TDuration GetSinceDelta(TInstant timestamp) const {
            if (timestamp < Since.Get()) {
                return TDuration::Zero();
            }
            return timestamp - Since.Get();
        }
    };
    using TSensors = TVector<TSensor>;
    using TMultiSensor = TSensors;

    TSensorValue SensorValueFromJson(const NJson::TJsonValue& value, NDrive::TSensorId sensorId = {});
    TSensorValue SensorValueFromString(const TString& value, NDrive::TSensorId sensorId);
    bool TryFromString(const TString& value, NDrive::TSensorValue& result, NDrive::TSensorId sensorId) noexcept;
}
