#include "sensor_filter.h"

#include <rtline/library/json/field.h>

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::TSensorFilterCondition);
DECLARE_FIELDS_JSON_SERIALIZER(NDrive::TSensorFilter);

template <class T, class V>
bool MatchRange(const TRange<T>& range, const V& value) {
    if (range.From && value < range.From) {
        return false;
    }
    if (range.To && !(value < range.To)) {
        return false;
    }
    return true;
}

bool NDrive::TSensorFilterCondition::Match(const TSensor& sensor, TInstant now) const {
    if (sensor != SensorId) {
        return false;
    }
    if (SinceAge) {
        auto age = now - sensor.Since;
        if (!MatchRange(SinceAge, age)) {
            return false;
        }
    }
    if (TimestampAge) {
        auto age = now - sensor.Timestamp;
        if (!MatchRange(TimestampAge, age)) {
            return false;
        }
    }
    if (ValueRange) {
        auto value = sensor.TryConvertTo<double>();
        if (!value) {
            return false;
        }
        if (!MatchRange(ValueRange, value)) {
            return false;
        }
    }
    return true;
}

bool NDrive::TSensorFilterCondition::Match(TConstArrayRef<TSensor> sensors, TInstant now) const {
    for (auto&& sensor : sensors) {
        if (Match(sensor, now)) {
            return true;
        }
    }
    return false;
}

bool NDrive::TSensorFilter::Match(TConstArrayRef<TSensor> sensors, TInstant now) const {
    switch (Op) {
    case EOp::And:
        return MatchAnd(sensors, now);
    case EOp::Or:
        return MatchOr(sensors, now);
    }
}

bool NDrive::TSensorFilter::MatchAnd(TConstArrayRef<TSensor> sensors, TInstant now) const {
    for (auto&& condition : Conditions) {
        if (!condition.Match(sensors, now)) {
            return false;
        }
    }
    for (auto&& subfilter : Subfilters) {
        if (!subfilter.Match(sensors, now)) {
            return false;
        }
    }
    return true;
}

bool NDrive::TSensorFilter::MatchOr(TConstArrayRef<TSensor> sensors, TInstant now) const {
    for (auto&& condition : Conditions) {
        if (condition.Match(sensors, now)) {
            return true;
        }
    }
    for (auto&& subfilter : Subfilters) {
        if (subfilter.Match(sensors, now)) {
            return true;
        }
    }
    return false;
}
