#include "sensor_filter.h"
#include "vega.h"

#include <rtline/library/json/cast.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/simple.h>

auto serializedFilter = NJson::ReadJsonFastTree(R"(
{
    "op": "And",
    "conditions": [
        {
            "sensor": "fuel_level",
            "timestamp_age": [null, 30],
            "value_range": [4, 42]
        },
        {
            "sensor": "engine_on",
            "value_range": [1, null]
        }
    ]
}
)");

auto legacyFilterCondition = NJson::ReadJsonFastTree(R"(
{
    "sensor": {
        "id": 2107
    },
    "timestamp_age": [null, 30],
    "value_range": [4, 42]
}
)");

Y_UNIT_TEST_SUITE(SensorFilter) {
    Y_UNIT_TEST(Serialization) {
        auto filter = NJson::FromJson<NDrive::TSensorFilter>(serializedFilter);
        UNIT_ASSERT(filter);
        UNIT_ASSERT_VALUES_EQUAL(filter.Conditions.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(filter.Subfilters.size(), 0);

        auto serialized = NJson::ToJson(filter);
        Cerr << serialized << Endl;
        auto deserialized = NJson::FromJson<NDrive::TSensorFilter>(serialized);
        UNIT_ASSERT(deserialized);
        UNIT_ASSERT_VALUES_EQUAL(deserialized.Conditions.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(deserialized.Subfilters.size(), 0);
    }

    Y_UNIT_TEST(LegacySerialization) {
        auto filterCondition = NJson::FromJson<NDrive::TSensorFilterCondition>(legacyFilterCondition);
        UNIT_ASSERT_VALUES_EQUAL(filterCondition.SensorId, CAN_FUEL_LEVEL_P);
    }

    Y_UNIT_TEST(Match) {
        auto filter = NJson::FromJson<NDrive::TSensorFilter>(serializedFilter);
        NDrive::TSensors sensors;
        TInstant now = Now();
        {
            sensors.emplace_back(CAN_FUEL_LEVEL_P);
            sensors.back().Timestamp = now - TDuration::Seconds(5);
            sensors.back().Since = now - TDuration::Seconds(15);
            sensors.back().Value = ui64(20);
        }
        {
            sensors.emplace_back(CAN_ENGINE_IS_ON);
            sensors.back().Timestamp = now - TDuration::Seconds(42);
            sensors.back().Since = now - TDuration::Seconds(60);
            sensors.back().Value = ui64(1);
        }
        {
            sensors.emplace_back(VEGA_MCU_FIRMWARE_VERSION);
            sensors.back().Timestamp = now - TDuration::Seconds(100);
            sensors.back().Since = now - TDuration::Seconds(20000);
            sensors.back().Value = "FIRMWARE";
        }
        UNIT_ASSERT(filter.Match(sensors, now));
        {
            sensors[1].Value = ui64(0);
        }
        UNIT_ASSERT(!filter.Match(sensors, now));
        filter.Op = NDrive::TSensorFilter::EOp::Or;
        UNIT_ASSERT(filter.Match(sensors, now));
        {
            sensors[0].Timestamp = now - TDuration::Seconds(100);
            sensors[0].Since = sensors[0].Timestamp;
        }
        UNIT_ASSERT(!filter.Match(sensors, now));
    }
}
