#include "sensor.h"
#include "vega.h"

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TelematicsSensor) {
    Y_UNIT_TEST(JsonSensor) {
        NDrive::TSensor sensor;
        sensor.Id = CAN_ODOMETER_KM;
        sensor.Value = 42.0;
        UNIT_ASSERT(std::holds_alternative<double>(sensor.Value));

        auto serialized = sensor.ToJson();

        NDrive::TSensor deserialized;
        UNIT_ASSERT(deserialized.TryFromJson(serialized));
        UNIT_ASSERT_VALUES_EQUAL(sensor.Id, deserialized.Id);
        UNIT_ASSERT(std::holds_alternative<double>(deserialized.Value));
        UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(sensor.Value), std::get<double>(deserialized.Value), 0.001);
    }

    Y_UNIT_TEST(Names) {
        {
            NDrive::TSensorId id(VEGA_MCU_FIRMWARE_VERSION);
            UNIT_ASSERT_VALUES_EQUAL(id.GetName(), "mcu_firmware");
        }
        {
            NDrive::TSensorId id(VEGA_DEVICE_SERIAL);
            UNIT_ASSERT_VALUES_EQUAL(id.GetName(), "VEGA_DEVICE_SERIAL");
        }
        {
            NDrive::TSensorId id(VEGA_SETTING_APN, 3);
            UNIT_ASSERT_VALUES_EQUAL(id.GetName(), "VEGA_SETTING_APN-3");
        }
        {
            NDrive::TSensorId id(VEGA_SETTING_ODOMETER_BY_IGN);
            UNIT_ASSERT_VALUES_EQUAL(id.GetName(), "3107");
        }
    }

    Y_UNIT_TEST(FromToString) {
        {
            NDrive::TSensorId sensor(VEGA_MCU_FIRMWARE_VERSION, VEGA_SETTING_ODOMETER_BY_IGN);
            NDrive::TSensorId rere = FromString<NDrive::TSensorId>(ToString(sensor));
            auto str = ToString(sensor);

            UNIT_ASSERT_VALUES_EQUAL(str, "mcu_firmware-3107");
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, rere.Id);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, rere.SubId);
        }
        {
            NDrive::TSensorId sensor(VEGA_MCU_FIRMWARE_VERSION);
            NDrive::TSensorId rere = FromString<NDrive::TSensorId>(ToString(sensor));
            UNIT_ASSERT_VALUES_EQUAL(ToString(sensor), "mcu_firmware");
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, rere.Id);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, rere.SubId);
        }
        {
            NDrive::TSensorId sensor(VEGA_DEVICE_SERIAL);
            NDrive::TSensorId rere = FromString<NDrive::TSensorId>(ToString(sensor));
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, rere.Id);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, rere.SubId);
        }
        {
            NDrive::TSensorId sensor(VEGA_DEVICE_SERIAL, VEGA_SETTING_ODOMETER_BY_IGN);
            NDrive::TSensorId rere = FromString<NDrive::TSensorId>(ToString(sensor));
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, rere.Id);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, rere.SubId);
        }
        {
            NDrive::TSensorId sensor = FromString<NDrive::TSensorId>("3007-1");
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, VEGA_SETTING_SERVER_PIN);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, 1);
        }
        {
            NDrive::TSensorId sensor = FromString<NDrive::TSensorId>("3007");
            UNIT_ASSERT_VALUES_EQUAL(sensor.Id, VEGA_SETTING_SERVER_PIN);
            UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, 0);
        }
        {
            NDrive::TSensorId sensor = NDrive::NVega::VegaMcuFirmwareVersionRevision;
            UNIT_ASSERT_VALUES_EQUAL(ToString(sensor), "mcu_firmware_revision");
            NDrive::TSensorId rere = FromString<NDrive::TSensorId>(ToString(sensor));
            UNIT_ASSERT_VALUES_EQUAL(rere.Id, sensor.Id);
            UNIT_ASSERT_VALUES_EQUAL(rere.SubId, sensor.SubId);
        }
        {
            NDrive::TSensorId sensor{1, 2};
            UNIT_ASSERT(TryFromString("mcu_firmware_revision", sensor));
            UNIT_ASSERT_VALUES_EQUAL(sensor, NDrive::NVega::VegaMcuFirmwareVersionRevision);
        }
        {
            NDrive::TSensorId sensor{1, 2};
            UNIT_ASSERT(TryFromString("mcu_firmware-1", sensor));
            UNIT_ASSERT_VALUES_EQUAL(sensor, NDrive::NVega::VegaMcuFirmwareVersionRevision);
        }
        {
            NDrive::TSensorId sensor{1, 2};
            UNIT_ASSERT(TryFromString("1232", sensor));
            NDrive::TSensorId expected{1232, 0};
            UNIT_ASSERT_VALUES_EQUAL(sensor, expected);
        }
        {
            NDrive::TSensorId sensor{1, 2};
            UNIT_ASSERT(TryFromString("1232-1", sensor));
            NDrive::TSensorId expected{1232, 1};
            UNIT_ASSERT_VALUES_EQUAL(sensor, expected);
        }
    }

    Y_UNIT_TEST(GetValueType) {
        {
            NDrive::TSensorId sensorId(VEGA_MCU_FIRMWARE_VERSION, VEGA_SETTING_ODOMETER_BY_IGN);
            UNIT_ASSERT_VALUES_EQUAL(NDrive::NVega::GetValueType(sensorId), NDrive::NVega::EValueType::ASCII);
        }
        {
            NDrive::TSensorId sensorId = NDrive::NVega::VegaMcuFirmwareVersionRevision;
            UNIT_ASSERT_VALUES_EQUAL(NDrive::NVega::GetValueType(sensorId), NDrive::NVega::EValueType::UI32);
        }
        {
            NDrive::TSensorId sensorId(VEGA_MCU_FIRMWARE_VERSION);
            UNIT_ASSERT_VALUES_EQUAL(NDrive::NVega::GetValueType(sensorId), NDrive::NVega::EValueType::ASCII);
        }
        {
            NDrive::TSensorId sensorId(VEGA_SETTING_SERVER_PIN);
            UNIT_ASSERT_VALUES_EQUAL(NDrive::NVega::GetValueType(sensorId), NDrive::NVega::EValueType::Binary);
        }
        {
            NDrive::TSensorId sensorId(VEGA_SETTING_SERVER_PIN, 1);
            UNIT_ASSERT_VALUES_EQUAL(NDrive::NVega::GetValueType(sensorId), NDrive::NVega::EValueType::Binary);
        }
    }
}
