#include "sensor_value.h"
#include "vega.h"

#include <rtline/library/json/cast.h>

#include <util/string/cast.h>
#include <util/string/type.h>
#include <util/string/hex.h>

NDrive::TSensorValue NDrive::SensorValueFromRef(const NDrive::TSensorRef& value) {
    if (std::holds_alternative<ui64>(value)) {
        return std::get<ui64>(value);
    } else if (std::holds_alternative<double>(value)) {
        return std::get<double>(value);
    } else if (std::holds_alternative<TStringBuf>(value)) {
        return TString(std::get<TStringBuf>(value));
    } else if (std::holds_alternative<TConstArrayRef<char>>(value)) {
        auto v = std::get<TConstArrayRef<char>>(value);
        return TBuffer(v.data(), v.size());
    } else {
        return TNull();
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSensorRef& value) {
    if (std::holds_alternative<ui64>(value)) {
        return std::get<ui64>(value);
    } else if (std::holds_alternative<double>(value)) {
        return std::get<double>(value);
    } else if (std::holds_alternative<TStringBuf>(value)) {
        return std::get<TStringBuf>(value);
    } else if (std::holds_alternative<TConstArrayRef<char>>(value)) {
        auto v = std::get<TConstArrayRef<char>>(value);
        return HexEncode(v.data(), v.size());
    } else {
        return NJson::JSON_NULL;
    }
}

namespace NDrive {
    bool TSensorValueOperator<TSensorValue>::TryFromJson(const NJson::TJsonValue& value, TSensorValue& result, const ui16 id, const ui16 subId) {
        if (!value.IsDefined()) {
            result = TNull();
            return true;
        }

        auto type = NDrive::NVega::GetValueType(TSensorId{id, subId});
        switch (type) {
            case NDrive::NVega::EValueType::UI8:
            case NDrive::NVega::EValueType::UI16:
            case NDrive::NVega::EValueType::UI32:
                if (value.IsUInteger()) {
                    result = static_cast<ui64>(value.GetUInteger());
                    return true;
                } else {
                    return false;
                }
            case NDrive::NVega::EValueType::Float32:
                if (value.IsDouble()) {
                    result = value.GetDouble();
                    return true;
                } else {
                    return false;
                }
            case NDrive::NVega::EValueType::ASCII:
                if (value.IsString()) {
                    result = value.GetString();
                    return true;
                } else {
                    return false;
                }
            case NDrive::NVega::EValueType::Binary:
                if (value.IsString()) {
                    auto data = HexDecode(value.GetString());
                    result = TBuffer(data.data(), data.size());
                    return true;
                } else {
                    return false;
                }
            default:
                break;
        }
        if (value.IsDouble()) {
            result = value.GetDoubleRobust();
        } else if (value.IsUInteger()) {
            result = static_cast<ui64>(value.GetUIntegerRobust());
        } else if (value.IsString()) {
            result = value.GetStringRobust();
        } else {
            result = TNull();
        }
        return true;
    }
}
