#pragma once

#include <drive/telematics/protocol/proto/sensor.pb.h>

#include <library/cpp/json/json_value.h>

#include <util/generic/buffer.h>
#include <util/generic/string.h>
#include <util/generic/utility.h>
#include <util/generic/fwd.h>
#include <util/generic/array_ref.h>
#include <util/generic/maybe.h>
#include <util/string/cast.h>
#include <util/string/type.h>
#include <util/string/hex.h>

#include <variant>

namespace NDrive {
    using TSensorValue = std::variant<ui64, double, TString, TBuffer, TNull>;
    using TSensorRef = std::variant<ui64, double, TStringBuf, TConstArrayRef<char>, TNull>;

    template <class TValue>
    class TSensorValueOperator;

    template <class TValue>
    class TBaseValueOperator {
    private:
        template <class To, class From>
        static bool ConvertImpl(const From& from, To& to) {
            to = from;
            return true;
        }

        template <class To>
        static bool ConvertImpl(const TString& from, To& to) {
            To result;
            if (!TryFromString(from, result)) {
                return false;
            }
            to = result;
            return true;
        }

        static bool ConvertImpl(const TString& from, bool& to) {
            to = IsTrue(from);
            return true;
        }

        static bool ConvertImpl(const TNull /*from*/, bool& to) {
            to = false;
            return true;
        }

        static bool ConvertImpl(const ui64 from, bool& to) {
            to = from;
            return true;
        }

        static bool ConvertImpl(const double from, ui64& to) {
            auto result = static_cast<ui64>(from);
            if (from == result) {
                to = result;
                return true;
            }
            return false;
        }
    public:
        template <class From, class To>
        static inline bool TryStrictConvertTo(const TValue& value, To& result) {
            if (!std::holds_alternative<From>(value)) {
                return false;
            }
            return ConvertImpl(std::get<From>(value), result);
        }
    };

    template <>
    class TSensorValueOperator<TSensorValue>: public TBaseValueOperator<TSensorValue> {
    public:
        static bool IsZero(const TSensorValue& value) {
            if (std::holds_alternative<ui64>(value)) {
                return std::get<ui64>(value) == 0;
            } else if (std::holds_alternative<double>(value)) {
                return std::abs(std::get<double>(value)) < 0.001;
            } else if (std::holds_alternative<TString>(value)) {
                return std::get<TString>(value).empty();
            } else if (std::holds_alternative<TBuffer>(value)) {
                return std::get<TBuffer>(value).Empty();
            } else {
                return true;
            }
        }

        static bool TryConvertTo(const TSensorValue& value, double& result, const TMaybe<double> fallback = {}) {
            if (TryStrictConvertTo<double>(value, result) || TryStrictConvertTo<ui64>(value, result) || TryStrictConvertTo<TString>(value, result)) {
                return true;
            }
            if (!!fallback) {
                result = *fallback;
                return true;
            }
            return false;
        }

        static double ConvertTo(const TSensorValue& value, const TMaybe<double> fallback) {
            double result;
            if (TryConvertTo(value, result, fallback)) {
                return result;
            }
            ythrow yexception() << "cannot cast Sensor " << ToJson(value).GetStringRobust() << " to double";
        }

        static bool TryConvertTo(const TSensorValue& value, bool& result, const TMaybe<bool> fallback = {}) {
            if (TryStrictConvertTo<ui64>(value, result) || TryStrictConvertTo<TString>(value, result) || TryStrictConvertTo<TNull>(value, result)) {
                return true;
            }
            if (!!fallback) {
                result = *fallback;
                return true;
            }
            return false;
        }

        static bool ConvertTo(const TSensorValue& value, const TMaybe<bool> fallback) {
            bool result;
            if (TryConvertTo(value, result, fallback)) {
                return result;
            }
            ythrow yexception() << "cannot cast Sensor " << ToJson(value).GetStringRobust() << " to boolean";
        }

        static bool TryConvertTo(const TSensorValue& value, ui64& result, const TMaybe<ui64> fallback = {}) {
            if (TryStrictConvertTo<ui64>(value, result) || TryStrictConvertTo<double>(value, result) || TryStrictConvertTo<TString>(value, result)) {
                return true;
            }
            if (!!fallback) {
                result = *fallback;
                return true;
            }
            return false;
        };

        static ui64 ConvertTo(const TSensorValue& value, const TMaybe<ui64> fallback) {
            ui64 result;
            if (TryConvertTo(value, result, fallback)) {
                return result;
            }
            ythrow yexception() << "cannot cast Sensor " << ToJson(value).GetStringRobust() << " to ui64";
        }

        static bool FromProto(const NDrive::NProto::TSensor& proto, TSensorValue& result) {
            if (proto.HasValueUI64()) {
                result = proto.GetValueUI64();
            } else if (proto.HasValueDouble()) {
                result = proto.GetValueDouble();
            } else if (proto.HasValueFloat()) {
                result = static_cast<double>(proto.GetValueFloat());
            } else if (proto.HasValueString()) {
                result = proto.GetValueString();
            } else if (proto.HasValueBinary()) {
                auto data = HexDecode(proto.GetValueBinary());
                result = TBuffer(data.data(), data.size());
            } else {
                result = TNull();
            }
            return true;
        }

        static void ToProto(const TSensorValue& value, NDrive::NProto::TSensor& result) {
            if (std::holds_alternative<ui64>(value)) {
                result.SetValueUI64(std::get<ui64>(value));
            } else if (std::holds_alternative<double>(value)) {
                result.SetValueFloat(std::get<double>(value));
            } else if (std::holds_alternative<TString>(value)) {
                result.SetValueString(std::get<TString>(value));
            } else if (std::holds_alternative<TBuffer>(value)) {
                const auto& v = std::get<TBuffer>(value);
                result.SetValueBinary(HexEncode(v.Data(), v.Size()));
            }
        }

        static TBuffer ConvertTo(const TSensorValue& valueExt, const TMaybe<TBuffer> fallback) {
            if (std::holds_alternative<TBuffer>(valueExt)) {
                return std::get<TBuffer>(valueExt);
            }
            if (std::holds_alternative<TString>(valueExt)) {
                const auto& value = std::get<TString>(valueExt);
                return {value.data(), value.size()};
            }
            if (std::holds_alternative<double>(valueExt)) {
                auto value = std::get<double>(valueExt);
                return {reinterpret_cast<char*>(&value), sizeof(value)};
            }
            if (std::holds_alternative<ui64>(valueExt)) {
                auto value = std::get<ui64>(valueExt);
                return {reinterpret_cast<char*>(&value), sizeof(value)};
            }
            if (fallback) {
                return *fallback;
            }
            ythrow yexception() << "cannot cast Sensor " << ToJson(valueExt).GetStringRobust() << " to TBuffer";
        }

        static TString ConvertTo(const TSensorValue& valueExt, const TMaybe<TString> fallback) {
            if (std::holds_alternative<TString>(valueExt)) {
                return std::get<TString>(valueExt);
            }
            if (std::holds_alternative<TBuffer>(valueExt)) {
                const auto& buffer = std::get<TBuffer>(valueExt);
                return {buffer.Data(), buffer.Size()};
            }
            if (std::holds_alternative<double>(valueExt)) {
                auto value = std::get<double>(valueExt);
                return ToString(value);
            }
            if (std::holds_alternative<ui64>(valueExt)) {
                auto value = std::get<ui64>(valueExt);
                return ToString(value);
            }
            if (fallback) {
                return *fallback;
            }
            ythrow yexception() << "cannot cast Sensor " << ToJson(valueExt).GetStringRobust() << " to String";
        }

        static NJson::TJsonValue ToJson(const TSensorValue& value) {
            if (std::holds_alternative<ui64>(value)) {
                return std::get<ui64>(value);
            } else if (std::holds_alternative<double>(value)) {
                return std::get<double>(value);
            } else if (std::holds_alternative<TString>(value)) {
                return std::get<TString>(value);
            } else if (std::holds_alternative<TBuffer>(value)) {
                const auto& v = std::get<TBuffer>(value);
                return HexEncode(v.Data(), v.Size());
            } else {
                return NJson::JSON_NULL;
            }
        }

        static bool TryFromJson(const NJson::TJsonValue& value, TSensorValue& result, const ui16 id, const ui16 subId = 0);
    };

    TSensorValue SensorValueFromRef(const NDrive::TSensorRef& value);
}

inline bool operator==(const TNull&, const TNull&) {
    return true;
}
