#include "crc.h"
#include "settings.h"
#include "vega.h"

#include <drive/vega/lib/trunk/vega_libs/tea_crypt/tea.h>

#include <library/cpp/charset/recyr.hh>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/digest/fnv.h>
#include <util/generic/bitops.h>

namespace {
    template <typename T>
    TMaybe<T> TryGetValueFromSensor(const NDrive::TSensor& sensor) {
        auto value = sensor.ConvertTo<TBuffer>(TBuffer());
        if (value.Size() < sizeof(T)) {
            return {};
        }

        T result;
        TBufferInput stream(value);
        ::Load(&stream, result);

        return result;
    }

    template <typename From, typename To>
    void ConvertValue(NDrive::TSensor& sensor) {
        if (auto value = TryGetValueFromSensor<From>(sensor)) {
            sensor.Value = static_cast<To>(*value);
        }
    }
}

ui16 NDrive::NVega::TBleSessionKeyParameter::GetId() {
    return BleSessionKey.Id;
}

TConstArrayRef<char> NDrive::NVega::TBleSessionKeyParameter::Raw() const {
    return { reinterpret_cast<const char*>(Value.data()), Value.size() };
}

ui16 NDrive::NVega::TServerSettingsParameter::GetId() {
    return VEGA_SETTING_SERVER_ADDR;
}

ui16 NDrive::NVega::TServerSettingsParameter::GetCount() {
    return NVega::ServerSettingsCount;
}

NJson::TJsonValue NDrive::NVega::TServerSettingsParameter::ToJson() const {
    NJson::TJsonValue result;
    result["address"] = HostPort.Get();
    result["period"] = InteractionPeriod;
    result["protocol"] = ToString(static_cast<TServerSettingsParameter::EProtocol>(Protocol));
    return result;
}

ui64 NDrive::NVega::TServerSettingsParameter::CalcHash() const {
    return
        FnvHash<ui64>(HostPort.View()) ^
        FnvHash<ui64>(&InteractionPeriod, sizeof(InteractionPeriod)) ^
        FnvHash<ui64>(&Protocol, sizeof(Protocol));
}

ui16 NDrive::NVega::TAPNParameter::GetId() {
    return VEGA_SETTING_APN;
}

ui16 NDrive::NVega::TAPNParameter::GetCount() {
    return APNSettingsCount;
}

NJson::TJsonValue NDrive::NVega::TAPNParameter::ToJson() const {
    NJson::TJsonValue result;
    result["APN"] = APN.Get();
    result["user"] = User.Get();
    result["password"] = Password.Get();
    return result;
}

ui64 NDrive::NVega::TAPNParameter::CalcHash() const {
    return
        FnvHash<ui64>(APN.View()) ^
        FnvHash<ui64>(User.View()) ^
        FnvHash<ui64>(Password.View());
}

ui16 NDrive::NVega::THardPasswordParameter::GetId() {
    return VEGA_SETTING_SERVER_PIN;
}

ui64 NDrive::NVega::THardPasswordParameter::CalcHash() const {
    return FnvHash<ui64>(Value.View());
}

ui16 NDrive::NVega::TUsePasswordParameter::GetId() {
    return VEGA_SETTING_USE_SERVER_PIN;
}

ui64 NDrive::NVega::TUsePasswordParameter::CalcHash() const {
    return FnvHash<ui64>(&Mode, sizeof(Mode));
}

ui16 NDrive::NVega::TWiredPasswordParameter::GetId() {
    return VEGA_SETTING_PIN_CODE;
}

ui16 NDrive::NVega::TUseWiredPasswordParameter::GetId() {
    return VEGA_SETTING_USE_PIN_CODE;
}

ui16 NDrive::NVega::TSensorTranslation::GetId() {
    return VEGA_SETTING_TRANSLATE_SENSORS;
}

ui16 NDrive::NVega::TCanParameter::GetId() {
    return VEGA_SETTING_CAN_CUSTOM_SENSORS;
}

ui16 NDrive::NVega::TCanParameter::GetCount() {
    return CanParametersCount;
}

NDrive::TSensor NDrive::NVega::TCanParameter::Parse(TSensor&& sensor, const TVector<NVega::TCanParameter>& parameters) {
    auto comparator = [id = sensor.Id] (const TCanParameter& parameter) {
        return parameter.Id == id;
    };

   const auto* parameter = std::find_if(parameters.begin(), parameters.end(), comparator);

    if (parameter == parameters.end()) {
        return std::move(sensor);
    }

    if (parameter->Id == sensor.Id) {
        switch (parameter->Type) {
        case CVT_BOOL:
            ConvertValue<bool, ui64>(sensor);
            break;
        case CVT_UINT8:
            ConvertValue<ui8, ui64>(sensor);
            break;
        case CVT_UINT16:
            ConvertValue<ui16, ui64>(sensor);
            break;
        case CVT_UINT32:
            ConvertValue<ui32, ui64>(sensor);
            break;
        case CVT_UINT64:
            ConvertValue<ui64, ui64>(sensor);
            break;
        case CVT_INT8:
            ConvertValue<i8, double>(sensor);
            break;
        case CVT_INT16:
            ConvertValue<i16, double>(sensor);
            break;
        case CVT_INT32:
            ConvertValue<i32, double>(sensor);
            break;
        case CVT_INT64:
            ConvertValue<i64, double>(sensor);
            break;
        case CVT_FLOAT:
            ConvertValue<float, double>(sensor);
            break;
        case CVT_DOUBLE:
            ConvertValue<double, double>(sensor);
            break;
        default:
            break;
        };
    }

    return std::move(sensor);
}

NJson::TJsonValue NDrive::NVega::TSensorTranslation::ToJson() const {
    NJson::TJsonValue result;
    result["on_track"] = OnTrack;
    result["on_period"] = OnPeriod;
    result["on_change"] = OnChange;
    return result;
}

ui64 NDrive::NVega::TSensorTranslation::CalcHash() const {
    return
        FnvHash<ui64>(&OnChange, sizeof(OnChange)) ^
        FnvHash<ui64>(&OnPeriod, sizeof(OnPeriod)) ^
        FnvHash<ui64>(&OnTrack, sizeof(OnTrack));
}

NJson::TJsonValue NDrive::NVega::TSetting::ToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["subid"] = SubId;
    result["value"] = FormatValue(Payload, Id);
    return result;
}

NJson::TJsonValue NDrive::NVega::TSetting::FormatValue(TConstArrayRef<char> data, ui16 id) {
    switch (id) {
    case VEGA_SETTING_SERVER_ADDR: {
        TServerSettingsParameter parameter;
        parameter.Parse(data);
        return parameter.ToJson();
    }
    case VEGA_SETTING_APN: {
        TAPNParameter parameter;
        parameter.Parse(data);
        return parameter.ToJson();
    }
    case VEGA_SETTING_TRANSLATE_SENSORS: {
        TSensorTranslation parameter;
        parameter.Parse(data);
        return parameter.ToJson();
    }
    case VEGA_SETTING_SERVER_PIN: {
        THardPasswordParameter parameter;
        parameter.Parse(data);
        return parameter.Value.Get();
    }
    case TSvrRawState::GetId(): {
        TSvrRawState parameter;
        parameter.Parse(data);
        return NJson::ToJson(parameter.Tuple());
    }
    case TSvrDtcState::GetId(): {
        TSvrDtcState parameter;
        parameter.Parse(data);
        return parameter.StringValue.Get();
    }
    default:
        return NJson::ToJson(TValue::Get(data, id));
    }
}

NJson::TJsonValue NDrive::NVega::TSettings::ToJson() const {
    NJson::TJsonValue result(NJson::JSON_ARRAY);
    for (auto&& setting : *this) {
        result.AppendValue(setting.ToJson());
    }
    return result;
}

void NDrive::NVega::TServerSettingsParameter::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

NDrive::NVega::TServerSettingsParameter::TType NDrive::NVega::TServerSettingsParameter::Dump() const {
    TBufferOutput output;
    Save(&output);
    return output.Buffer();
}

void NDrive::NVega::TServerSettingsParameter::Parse(const TType& data) {
    Parse(TConstArrayRef<char>{ data.Data(), data.Size() });
}

void NDrive::NVega::TAPNParameter::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

NDrive::NVega::TAPNParameter::TType NDrive::NVega::TAPNParameter::Dump() const {
    TBufferOutput output;
    Save(&output);
    return output.Buffer();
}

void NDrive::NVega::TAPNParameter::Parse(const TType& data) {
    Parse(TConstArrayRef<char>{ data.Data(), data.Size() });
}

void NDrive::NVega::THardPasswordParameter::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

NDrive::NVega::THardPasswordParameter::TType NDrive::NVega::THardPasswordParameter::Dump() const {
    TBufferOutput output;
    Save(&output);
    return output.Buffer();
}

void NDrive::NVega::THardPasswordParameter::Parse(const TType& data) {
    Parse(TConstArrayRef<char>{ data.Data(), data.Size() });
}

NDrive::NVega::TUsePasswordParameter::TType NDrive::NVega::TUsePasswordParameter::Dump() const {
    return Mode;
}

void NDrive::NVega::TUsePasswordParameter::Parse(TType data) {
    Mode = data;
}

void NDrive::NVega::TSensorTranslation::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

NDrive::NVega::TSensorTranslation::TType NDrive::NVega::TSensorTranslation::Dump() const {
    TBufferOutput output;
    Save(&output);
    return output.Buffer();
}

void NDrive::NVega::TSensorTranslation::Parse(const TType& data) {
    Parse(TConstArrayRef<char>{ data.Data(), data.Size() });
}

void NDrive::NVega::TSettings::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    while (!input.Exhausted()) {
        TSetting setting;
        setting.Load(&input);
        push_back(std::move(setting));
    }
}

TBuffer NDrive::NVega::TSettings::Dump() const {
    std::array<ui32, 4> key = {
        0X6021528A,
        0X47F50670,
        0X8D99AAFB,
        0XC6AD9EFA
    };

    return ToSigned(key);
}

TBuffer NDrive::NVega::TSettings::ToSigned(std::array<ui32, 4> key) const {
    TBuffer data;
    {
        TBufferOutput so(data);
        NProtocol::TFixedSizeZeroTerminatedString<16> filename;
        filename.Set("SETTINGS");
        filename.Save(&so);
        for (auto&& setting : *this) {
            setting.Save(&so);
        }
    }

    tea_context context;
    {
        tea_certify_init(&context, key.data());
        auto step = Max<uint16_t>();
        for (size_t i = 0; i < data.size(); i += step) {
            auto buf = reinterpret_cast<uint8_t*>(data.Begin()) + i;
            auto len = data.size() < i + step ? uint16_t(data.size() - i) : step;
            tea_certify_update(&context, buf, len);
        }
        tea_certify_finish(&context);
    }

    TBuffer result;
    TBufferOutput so(result);
    Save(&so, context.cert[0]);
    Save(&so, context.cert[1]);
    so.Write(data.data(), data.size());
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TUnitedValue>(const NDrive::NVega::TUnitedValue& object) {
    return std::visit([](auto&& arg){
        return NJson::TJsonValue(arg);
    }, object.Value);
}

void NDrive::NVega::TSvrRawState::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

void NDrive::NVega::TSvrDtcState::Parse(TConstArrayRef<char> data) {
    TMemoryInput input(data.data(), data.size());
    Load(&input);
}

void NDrive::NVega::TCanParameter::TCanBusSetting::Load(IInputStream* stream) {
    ui8 data;
    ::Load(stream, data);
    CanChannel = static_cast<ECanChannel>(SelectBits<0, 3>(data));
    Version = SelectBits<3, 5>(data);
}

void NDrive::NVega::TCanParameter::TCanBusSetting::Save(IOutputStream* stream) const {
    ui8 result = 0;
    SetBits<0, 3, ui8>(result, CanChannel);
    SetBits<3, 5, ui8>(result, Version);
    ::Save(stream, result);
}

void NDrive::NVega::TCanParameter::THeaderSetting::Load(IInputStream* stream) {
    ui8 data;
    ::Load(stream, data);
    IdentifierType = static_cast<EIdentifierType>(SelectBits<0, 3>(data));
    StructureType = SelectBits<3, 1>(data);
    IsCrypted = SelectBits<7, 1>(data);
}

void NDrive::NVega::TCanParameter::THeaderSetting::Save(IOutputStream* stream) const {
    ui8 result = 0;
    SetBits<0, 3, ui8>(result, IdentifierType);
    SetBits<3, 1>(result, StructureType);
    SetBits<7, 1, ui8>(result, IsCrypted);
    ::Save(stream, result);
}

void NDrive::NVega::TCanParameter::TSettings::TLength::Load(IInputStream* stream) {
    ui8 data = 0;
    ::Load(stream, data);
    Length = SelectBits<0, 7>(data);
    IsSigned = SelectBits<7, 1>(data);
}

void NDrive::NVega::TCanParameter::TSettings::TLength::Save(IOutputStream* stream) const {
    ui8 result = 0;
    SetBits<0, 7, ui8>(result, Length);
    SetBits<7, 1, ui8>(result, IsSigned);
    ::Save(stream, result);
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TLength::ToJson() const {
    NJson::TJsonValue result;

    result["length"] = Length;
    result["is_signed"] = IsSigned;

    return result;
}

void NDrive::NVega::TCanParameter::TSettings::TPositions::Load(IInputStream* stream) {
    ui8 result = 0;
    ::Load(stream, result);
    Order = static_cast<EOrder>(SelectBits<0, 1>(result));
    Invert = SelectBits<1, 1>(result);
    ByteIndex = SelectBits<2, 3>(result);
    StartBit = SelectBits<5, 3>(result);
}

void NDrive::NVega::TCanParameter::TSettings::TPositions::Save(IOutputStream* stream) const {
    ui8 result = 0;
    SetBits<0, 1, ui8>(result, Order);
    SetBits<1, 1, ui8>(result, Invert);
    SetBits<2, 3, ui8>(result, ByteIndex);
    SetBits<5, 3, ui8>(result, StartBit);
    ::Save(stream, result);
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TPositions::ToJson() const {
    NJson::TJsonValue result;

    result["bit_order"] = ToString(Order);
    result["invert"] = Invert;
    result["byte_index"] = ByteIndex;
    result["start_bit"] = StartBit;

    return result;
}

void NDrive::NVega::TCanParameter::TSettings::TResetSettings::Load(IInputStream* stream) {
    ui8 result = 0;
    ::Load(stream, result);
    ResetSeconds = SelectBits<0, 4>(result);
    ResetByIgnitionOff = SelectBits<4, 1>(result);
    FastBool = SelectBits<5, 1>(result);
    IsMaskFilter = SelectBits<6, 1>(result);
    IsDestination = SelectBits<7, 1>(result);
}

void NDrive::NVega::TCanParameter::TSettings::TResetSettings::Save(IOutputStream* stream) const {
    ui8 result = 0;
    SetBits<0, 4, ui8>(result, ResetSeconds);
    SetBits<4, 1, ui8>(result, ResetByIgnitionOff);
    SetBits<5, 1, ui8>(result, FastBool);
    SetBits<6, 1, ui8>(result, IsMaskFilter);
    SetBits<7, 1, ui8>(result, IsDestination);
    ::Save(stream, result);
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TResetSettings::ToJson() const {
    NJson::TJsonValue result;

    result["reset_seconds"] = ResetSeconds;
    result["reset_by_ignition_off"] = ResetByIgnitionOff;
    result["fast_bool"] = FastBool;

    return result;
}

size_t NDrive::NVega::TCanParameter::TSettings::CalcSize() const {
    return 42;
}

ui16 NDrive::NVega::TCanParameter::TSettings::CalcCrc() const {
    TBuffer data;
    TBufferOutput stream(data);
    Save(&stream);
    return NDrive::VegaCrc16(data.Data(), data.End());
}

void NDrive::NVega::TCanParameter::TSettings::Load(IInputStream* stream) {
    ::Load(stream, CanId);
    ::Load(stream, ResetValue);
    ::Load(stream, ResetSettings);

    if (ResetSettings.IsDestination) {
        TDestination destination;
        ::Load(stream, destination);
        Bits = destination;
    } else {
        TLength length;
        ::Load(stream, length);
        Bits = length;
    }

    ::Load(stream, Positions);

    if (ResetSettings.IsMaskFilter) {
        TMask mask;
        ::Load(stream, mask);
        Filter = mask;
    } else {
        TBorders borders;
        ::Load(stream, borders);
        Filter = borders;
    }

    ::Load(stream, Scale);
    ::Load(stream, Offset);
    ::Load(stream, Reserve);
}

void NDrive::NVega::TCanParameter::TSettings::Save(IOutputStream* stream) const {
    ::Save(stream, CanId);
    ::Save(stream, ResetValue);

    auto resetSettings = ResetSettings;
    resetSettings.IsDestination = std::holds_alternative<TDestination>(Bits);
    resetSettings.IsMaskFilter = std::holds_alternative<TMask>(Filter);

    ::Save(stream, resetSettings);

    if (std::holds_alternative<TDestination>(Bits)) {
        ::Save(stream, std::get<TDestination>(Bits));
    } else {
        ::Save(stream, std::get<TLength>(Bits));
    }

    ::Save(stream, Positions);

    if (std::holds_alternative<TMask>(Filter)) {
        ::Save(stream, std::get<TMask>(Filter));
    } else {
        ::Save(stream, std::get<TBorders>(Filter));
    }

    ::Save(stream, Scale);
    ::Save(stream, Offset);
    ::Save(stream, Reserve);
}

auto NDrive::NVega::TUnitedValue::GetTypeIndex() -> EType {
    return TypeMap.Value(Type, T_UINT64);
}

void NDrive::NVega::TUnitedValue::Load(IInputStream* stream) {
    ui8 index = GetTypeIndex();

    size_t dataSize = CalcSize() + sizeof(index);
    TBuffer data(dataSize);
    TBufferInput input(data);

    data.Append(static_cast<char>(index));
    data.Advance(stream->Read(data.End(), CalcSize()));

    ::Load(&input, Value);
}

void NDrive::NVega::TUnitedValue::Save(IOutputStream* stream) const {
    TBuffer data(CalcSize());
    TBufferOutput dataStream(data);

    std::visit([&dataStream](const auto& data) {
        ::Save(&dataStream, data);
    }, Value);

    while (data.Size() < CalcSize()) {
        data.Append(0x00);
    }

    ::SavePodArray(stream, data.Data(), data.Size());
}

size_t NDrive::NVega::TCanParameter::CalcSize() const {
    return 84;
}

void NDrive::NVega::TCanParameter::Load(IInputStream* stream) {
    TBuffer data(CalcSize());
    data.Advance(stream->Read(data.End(), CalcSize()));
    Parse(data);
}

auto NDrive::NVega::TCanParameter::Dump() const -> TType {
    TType result(CalcSize());
    TBufferOutput stream(result);

    ::Save(&stream, CanBusSetting);
    ::Save(&stream, HeaderSetting);
    ::Save(&stream, Nonce);

    TBuffer settingsData(Settings.CalcSize());
    TBufferOutput settingsStream(settingsData);
    ::Save(&settingsStream, Settings);

    ui16 crc = NDrive::VegaCrc16(settingsData.Data(), settingsData.End());
    ::Save(&stream, crc);
    ::Save(&stream, static_cast<ui8>(Type));
    ::Save(&stream, Id);
    ::Save(&stream, Name);
    ::Save(&stream, Reserve);
    ::SavePodArray(&stream, settingsData.Data(), settingsData.Size());

    return result;
}

void NDrive::NVega::TCanParameter::Parse(const TType& data) {
    TBufferInput stream(data);
    ::Load(&stream, CanBusSetting);
    ::Load(&stream, HeaderSetting);
    ::Load(&stream, Nonce);

    ui16 crc = 0;
    ::Load(&stream, crc);
    ::Load(&stream, Type);
    ::Load(&stream, Id);
    ::Load(&stream, Name);
    ::Load(&stream, Reserve);

    TBuffer settingsData(Settings.CalcSize());
    TBufferInput settingsStream(settingsData);
    settingsData.Advance(stream.Load(settingsData.End(), Settings.CalcSize()));

    ui16 calculatedCrc = NDrive::VegaCrc16(settingsData.Begin(), settingsData.End());

    if (!HeaderSetting.IsCrypted && crc == calculatedCrc) {
        Settings.ResetValue.Type = Type;
        ::Load(&settingsStream, Settings);
    }
}

ui64 NDrive::NVega::TCanParameter::CalcHash() const {
    auto data = Dump();
    return FnvHash<ui64>(data);
}

ui64 NDrive::NVega::TCanParameterName::CalcHash() const {
    return FnvHash<ui64>(Name.View());
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TBorders::ToJson() const {
    NJson::TJsonValue result;
    result["minimum"] = Minimum;
    result["maximum"] = Maximum;
    return result;
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TMask::ToJson() const {
    NJson::TJsonValue result;
    result["mask"] = Mask;
    result["value"] = Value;
    return result;
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::TDestination::ToJson() const {
    NJson::TJsonValue result;
    result["position"] = Position;
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TCanParameter::TSettings::TFilter>(const NDrive::NVega::TCanParameter::TSettings::TFilter& object) {
    NJson::TJsonValue result;
    if (std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TMask>(object)) {
        result["mask"] = std::get<NDrive::NVega::TCanParameter::TSettings::TMask>(object).ToJson();
    } else {
        result["borders"] = std::get<NDrive::NVega::TCanParameter::TSettings::TBorders>(object).ToJson();
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TCanParameter::TSettings::TBits>(const NDrive::NVega::TCanParameter::TSettings::TBits& object) {
    NJson::TJsonValue result;
    if (std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TDestination>(object)) {
        result["destination"] = std::get<NDrive::NVega::TCanParameter::TSettings::TDestination>(object).ToJson();
    } else {
        result["length"] = std::get<NDrive::NVega::TCanParameter::TSettings::TLength>(object).ToJson();
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicCommand::TArgument>(const NDrive::NVega::TLogicCommand::TArgument& object) {
    NJson::TJsonValue result;
    if (std::holds_alternative<NDrive::NVega::TLogicCommandSetValue>(object)) {
        result["set_value"] = std::get<NDrive::NVega::TLogicCommandSetValue>(object).ToJson();
    } else {
        result["variable"] = std::get<ui64>(object);
    }
    return result;
}

NJson::TJsonValue NDrive::NVega::TCanParameter::TSettings::ToJson() const {
    NJson::TJsonValue result;

    result["can_id"] = CanId;
    result["reset_value"] = NJson::ToJson(ResetValue);
    result["reset_settings"] = ResetSettings.ToJson();
    result["filter"] = NJson::ToJson(Filter);
    result["bits"] = NJson::ToJson(Bits);
    result["positions"] = Positions.ToJson();
    result["scale"] = NJson::ToJson(Scale);
    result["offset"] = NJson::ToJson(Offset);

    return result;
}

NJson::TJsonValue NDrive::NVega::TCanParameter::ToJson() const {
    NJson::TJsonValue result;

    result["can_channel"] = ToString(CanBusSetting.CanChannel);
    result["can_id_type"] = ToString(HeaderSetting.IdentifierType);
    result["is_crypted"] = ToString(HeaderSetting.IsCrypted);
    result["sensor_type"] = ToString(Type);
    result["sensor_id"] = Id;
    result["sensor_name"] = Name.Get();
    result["settings"] = Settings.ToJson();

    return result;
}

bool NDrive::NVega::TCanParameter::IsEmpty() const {
    return HeaderSetting.IsCrypted || Name.Get().empty();
}

TBuffer NDrive::NVega::TCanParameterName::Dump() {
    TBuffer buffer(Name.CalcSize());
    TBufferOutput stream(buffer);
    Name.Save(&stream);
    return buffer;
}

TString NDrive::NVega::TCanParameter::TName::Get() const {
    TString value = Value.Get();

    try {
        value = Recode(CODES_WIN, CODES_UTF8, value);
    } catch (...) {
        value = Value.Get();
    }

    return std::move(value);
}

void NDrive::NVega::TCanParameter::TName::Set(TStringBuf value) {
    TString result(value);

    try {
        result = Recode(CODES_UTF8, CODES_WIN, result);
    } catch (...) {
        result = value;
    }

    Value.Set(result);
}

bool NDrive::NVega::TCanParameter::TName::IsUTF8(TStringBuf value) const {
    try {
        UTF8ToWide(value);
        return true;
    } catch (...) {
        return false;
    }
}

void NDrive::NVega::TLogicSteps::Load(IInputStream* stream) {
    for (size_t i = 0; i < Count; ++i) {
        TLogicStep step;
        ::Load(stream, step);

        if (!step.IsEmpty()) {
            Steps[i] = step;
        }
    }
}

void NDrive::NVega::TLogicSteps::Save(IOutputStream* stream) const {
    for (size_t i = 0; i < Count; ++i) {
        ::Save(stream, Steps.Value(i, TLogicStep()));
    }
}

size_t NDrive::NVega::TLogicCheckValue::CalcSize() const {
    return Value.CalcSize() + sizeof(Type);
}

void NDrive::NVega::TLogicCheckValue::Load(IInputStream *stream) {
    TBuffer value(Value.CalcSize());

    value.Advance(stream->Read(value.End(), Value.CalcSize()));
    ::Load(stream, Type);

    TBufferInput valueStream(value);
    Value.Type = Type;
    ::Load(&valueStream, Value);
}

void NDrive::NVega::TLogicCheckValue::Save(IOutputStream *stream) const {
    ::Save(stream, Value);
    ::Save(stream, Type);
}

void NDrive::NVega::TLogicTriggerChecks::Load(IInputStream* stream) {
    for (size_t i = 0; i < Count; ++i) {
        TLogicActiveId checkId;
        ::Load(stream, checkId);
        if (!checkId.IsEmpty()) {
            Checks[i] = checkId;
        }
    }
}

void NDrive::NVega::TLogicTriggerChecks::Save(IOutputStream* stream) const {
    for (size_t i = 0; i < Count; ++i) {
        ::Save(stream, Checks.Value(i, TLogicActiveId()));
    }
}

void NDrive::NVega::TLogicCommand::Load(IInputStream* stream) {
    ::Load(stream, Type);

    if (Type == T_SET_SENSOR) {
        LoadArgument<TLogicCommandSetValue>(stream) ;
    } else {
        LoadArgument<ui64>(stream);
    }
}

void NDrive::NVega::TLogicCommand::Save(IOutputStream* stream) const {
    ::Save(stream, Type);
    std::visit([stream](auto&& arg){
        ::Save(stream, arg);
    }, Argument);
}

NJson::TJsonValue NDrive::NVega::TLogicStep::ToJson() const {
    NJson::TJsonValue result;

    result["type"] = ToString(Type);
    result["action_id"] = ActionId;

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicSteps::ToJson() const {
    return NJson::ToJson(NJson::KeyValue(Steps));
}

NJson::TJsonValue NDrive::NVega::TLogicScript::ToJson() const {
    NJson::TJsonValue result;

    result["command_id"] = CommandId;
    result["name"] = Name.Get();
    result["steps"] = Steps.ToJson();

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicActiveId::ToJson() const {
    NJson::TJsonValue result;

    result["active"] = Active;
    result["id"] = Id;

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicTriggerChecks::ToJson() const {
    return NJson::ToJson(NJson::KeyValue(Checks));
}

NJson::TJsonValue NDrive::NVega::TLogicTrigger::ToJson() const {
    NJson::TJsonValue result;

    result["type"] = ToString(Type);
    result["checks"] = Checks.ToJson();
    result["triggered_script_id"] = TriggeredScriptId.ToJson();
    result["untriggered_script_id"] = UntriggeredScriptId.ToJson();
    result["timeout"] = Timeout;

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicNotify::ToJson() const {
    NJson::TJsonValue result;
    result["message"] = Message.Get();
    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicCheckValue::ToJson() const {
    NJson::TJsonValue result;

    result["value"] = NJson::ToJson(Value);
    result["type"] = ToString(Type);

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicCheck::ToJson() const {
    NJson::TJsonValue result;

    result["type"] = ToString(Type);
    result["operation"] = ToString(Operation);
    result["sensor_id"] = ToString(SensorId);
    result["sensor_sub_id"] = ToString(SensorSubId);
    result["value"] = Value.ToJson();
    result["notify"] = NotifyId.ToJson();
    result["argument"] = Argument;

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicCommandSetValue::ToJson() const {
    NJson::TJsonValue result;

    result["sensor_id"] = Id;
    result["sensor_sub_id"] = SubId;
    result["value"] = Value;

    return result;
}

NJson::TJsonValue NDrive::NVega::TLogicCommand::ToJson() const {
    NJson::TJsonValue result;

    result["type"] = ToString(Type);
    result["argument"] = NJson::ToJson(Argument);

    return result;
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TCanParameter>(const NDrive::NVega::TCanParameter& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TAPNParameter>(const NDrive::NVega::TAPNParameter& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TSensorTranslation>(const NDrive::NVega::TSensorTranslation& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TServerSettingsParameter>(const NDrive::NVega::TServerSettingsParameter& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::THardPasswordParameter>(const NDrive::NVega::THardPasswordParameter& object) {
    return object.Value.View();
}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TWiredPasswordParameter>(const NDrive::NVega::TWiredPasswordParameter& object) {
    return object.Value.View();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicSteps>(const NDrive::NVega::TLogicSteps& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicStep>(const NDrive::NVega::TLogicStep& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicActiveId>(const NDrive::NVega::TLogicActiveId& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicTriggerChecks>(const NDrive::NVega::TLogicTriggerChecks& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicTrigger>(const NDrive::NVega::TLogicTrigger& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicNotify>(const NDrive::NVega::TLogicNotify& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicCheckValue>(const NDrive::NVega::TLogicCheckValue& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicCheck>(const NDrive::NVega::TLogicCheck& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicCommandSetValue>(const NDrive::NVega::TLogicCommandSetValue& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicCommand>(const NDrive::NVega::TLogicCommand& object) {
    return object.ToJson();
}

template<>
NJson::TJsonValue NJson::ToJson<NDrive::NVega::TLogicScript>(const NDrive::NVega::TLogicScript& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TAPNParameter>(const NJson::TJsonValue& value, NDrive::NVega::TAPNParameter& result) {
    TString name;
    TString user;
    TString password;
    bool parsed =
        NJson::ParseField(value["APN"], name) &&
        NJson::ParseField(value["user"], user) &&
        NJson::ParseField(value["password"], password);
    if (parsed) {
        result.APN.Set(name);
        result.User.Set(user);
        result.Password.Set(password);
        return true;
    } else {
        return false;
    }
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TSensorTranslation>(const NJson::TJsonValue& value, NDrive::NVega::TSensorTranslation& result) {
    return
        NJson::ParseField(value["on_track"], result.OnTrack) &&
        NJson::ParseField(value["on_period"], result.OnPeriod) &&
        NJson::ParseField(value["on_change"], result.OnChange);
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TServerSettingsParameter>(const NJson::TJsonValue& value, NDrive::NVega::TServerSettingsParameter& result) {
    TString address;
    NDrive::NVega::TServerSettingsParameter::EProtocol protocol;
    if (value["address"].GetString(&address) && TryFromString(value["protocol"].GetStringRobust(), protocol) && TryFromString(value["period"].GetStringRobust(), result.InteractionPeriod)) {
        result.HostPort.Set(address);
        result.Protocol = protocol;
        return true;
    } else {
        return false;
    }
}

template <>
bool NJson::TryFromJson<NDrive::NVega::THardPasswordParameter>(const NJson::TJsonValue& value, NDrive::NVega::THardPasswordParameter& result) {
    TString v;
    if (NJson::TryFromJson(value, v)) {
        result.Value.Set(v);
        return true;
    } else {
        return false;
    }
}

template <>
bool NJson::TryFromJson<NDrive::NVega::TWiredPasswordParameter>(const NJson::TJsonValue& value, NDrive::NVega::TWiredPasswordParameter& result) {
    TString v;
    if (NJson::TryFromJson(value, v)) {
        result.Value.Set(v);
        return true;
    } else {
        return false;
    }
}

namespace {
    template<typename TValue>
    bool TryUnitedValueParse(const NJson::TJsonValue& value, NDrive::NVega::TUnitedValue& object) {
        if constexpr (
            std::is_integral<TValue>::value && std::is_unsigned<TValue>::value) {
            if (value.IsBoolean()) {
                object.Value = static_cast<TValue>(value.GetBoolean());
                return true;
            }
            if (value.IsUInteger()) {
                object.Value = static_cast<TValue>(value.GetUInteger());
                return true;
            }
        } else if constexpr (std::is_integral<TValue>::value) {
            if (value.IsInteger()) {
                object.Value = static_cast<TValue>(value.GetInteger());
                return true;
            }
        } else if constexpr (std::is_floating_point<TValue>::value) {
            if (value.IsDouble()) {
                object.Value = static_cast<TValue>(value.GetDouble());
                return true;
            }
        }

        return false;
    }
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TUnitedValue>(const NJson::TJsonValue& value, NDrive::NVega::TUnitedValue& object) {
    if (!value.IsDefined()) {
        return false;
    }

    auto index = object.GetTypeIndex();
    switch(index) {
    case NDrive::NVega::TUnitedValue::T_UINT8:
        return TryUnitedValueParse<ui8>(value, object);
    case NDrive::NVega::TUnitedValue::T_UINT16:
        return TryUnitedValueParse<ui16>(value, object);
    case NDrive::NVega::TUnitedValue::T_UINT32:
        return TryUnitedValueParse<ui32>(value, object);
    case NDrive::NVega::TUnitedValue::T_UINT64:
        return TryUnitedValueParse<ui64>(value, object);
    case NDrive::NVega::TUnitedValue::T_INT8:
        return TryUnitedValueParse<i8>(value, object);
    case NDrive::NVega::TUnitedValue::T_INT16:
        return TryUnitedValueParse<i16>(value, object);
    case NDrive::NVega::TUnitedValue::T_INT32:
        return TryUnitedValueParse<i32>(value, object);
    case NDrive::NVega::TUnitedValue::T_INT64:
        return TryUnitedValueParse<i64>(value, object);
    case NDrive::NVega::TUnitedValue::T_FLOAT:
        return TryUnitedValueParse<float>(value, object);
    case NDrive::NVega::TUnitedValue::T_DOUBLE:
        return TryUnitedValueParse<double>(value, object);
    case NDrive::NVega::TUnitedValue::T_BOOL:
        return TryUnitedValueParse<bool>(value, object);
    default:
        return false;
    }

    return false;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TBorders>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TBorders& object) {
    return
        NJson::ParseField(value["minimum"], object.Minimum) &&
        NJson::ParseField(value["maximum"], object.Maximum);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TMask>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TMask& object) {
    return
        NJson::ParseField(value["value"], object.Value) &&
        NJson::ParseField(value["mask"], object.Mask);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TLength>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TLength& object) {
    return
        NJson::ParseField(value["length"], object.Length) &&
        NJson::ParseField(value["is_signed"], object.IsSigned);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TDestination>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TDestination& object) {
    return NJson::ParseField(value["position"], object.Position);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TPositions::EOrder>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TPositions::EOrder& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TResetSettings>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TResetSettings& resetSettings) {
    return
        NJson::ParseField(value["reset_seconds"], resetSettings.ResetSeconds) &&
        NJson::ParseField(value["reset_by_ignition_off"], resetSettings.ResetByIgnitionOff) &&
        NJson::ParseField(value["fast_bool"], resetSettings.FastBool);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TPositions>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TPositions& positions) {
    return
        NJson::ParseField(value["bit_order"], positions.Order) &&
        NJson::ParseField(value["invert"], positions.Invert) &&
        NJson::ParseField(value["byte_index"], positions.ByteIndex) &&
        NJson::ParseField(value["start_bit"], positions.StartBit);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TBits>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TBits& bits) {
    NDrive::NVega::TCanParameter::TSettings::TLength length;
    NDrive::NVega::TCanParameter::TSettings::TDestination destination;

    if (ParseField(value["destination"], destination, true)) {
        bits = destination;
        return true;
    }
    if (ParseField(value["length"], length, true)) {
        bits = length;
        return true;
    }

    return false;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings::TFilter>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings::TFilter& filter) {
    NDrive::NVega::TCanParameter::TSettings::TMask mask;
    NDrive::NVega::TCanParameter::TSettings::TBorders borders;

    if (NJson::ParseField(value["mask"], mask, true)) {
        filter = mask;
        return true;
    }
    if (NJson::ParseField(value["borders"], borders, true)) {
        filter = borders;
        return true;
    }

    return false;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::TSettings>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::TSettings& object) {
    return
        NJson::ParseField(value["can_id"], object.CanId) &&
        NJson::ParseField(value["reset_value"], object.ResetValue) &&
        NJson::ParseField(value["reset_settings"], object.ResetSettings) &&
        NJson::ParseField(value["positions"], object.Positions) &&
        NJson::ParseField(value["filter"], object.Filter) &&
        NJson::ParseField(value["bits"], object.Bits) &&
        NJson::ParseField(value["scale"], object.Scale) &&
        NJson::ParseField(value["offset"], object.Offset);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::ECanChannel>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::ECanChannel& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter::EIdentifierType>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter::EIdentifierType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::ECommonValueType>(const NJson::TJsonValue& value, NDrive::NVega::ECommonValueType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TCanParameter>(const NJson::TJsonValue& value, NDrive::NVega::TCanParameter& object) {
    TString name;
    bool result =
        NJson::ParseField(value["can_channel"], object.CanBusSetting.CanChannel) &&
        NJson::ParseField(value["can_id_type"], object.HeaderSetting.IdentifierType) &&
        NJson::ParseField(value["sensor_type"], object.Type) &&
        NJson::ParseField(value["sensor_id"], object.Id) &&
        NJson::ParseField(value["sensor_name"], name);

    if (result) {
        ui64 maxSize = object.Name.CalcSize() - 1;
        if (name.Size() > maxSize) {
            name.resize(maxSize);
        }

        object.Settings.ResetValue.Type = object.Type;
        object.Name.Set(name);

        result = result && NJson::TryFromJson(value["settings"], object.Settings);
    }

    return result;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicStep::EType>(const NJson::TJsonValue& value, NDrive::NVega::TLogicStep::EType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicStep>(const NJson::TJsonValue& value, NDrive::NVega::TLogicStep& object) {
    return
        ParseField(value["type"], object.Type) &&
        ParseField(value["action_id"], object.ActionId);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicSteps>(const NJson::TJsonValue& value, NDrive::NVega::TLogicSteps& object) {
    return ParseField(value, NJson::KeyValue(object.Steps));
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicScript>(const NJson::TJsonValue& value, NDrive::NVega::TLogicScript& object) {
    TString name;

    bool result =
        NJson::ParseField(value["command_id"], object.CommandId) &&
        NJson::ParseField(value["name"], name) &&
        NJson::ParseField(value["steps"], object.Steps);

    if (result) {
        size_t maxSize = object.Name.CalcSize() - 1;
        if (name.Size() > maxSize) {
            name.resize(maxSize);
        }
        object.Name.Set(name);
        return true;
    }

    return false;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicTrigger::EType>(const NJson::TJsonValue& value, NDrive::NVega::TLogicTrigger::EType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicActiveId>(const NJson::TJsonValue& value, NDrive::NVega::TLogicActiveId& object) {
    return
        NJson::ParseField(value["active"], object.Active) &&
        NJson::ParseField(value["id"], object.Id);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicTriggerChecks>(const NJson::TJsonValue& value, NDrive::NVega::TLogicTriggerChecks& object) {
    return ParseField(value, NJson::KeyValue(object.Checks));
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicTrigger>(const NJson::TJsonValue& value, NDrive::NVega::TLogicTrigger& object) {
    return
        NJson::ParseField(value["type"], object.Type) &&
        NJson::ParseField(value["checks"], object.Checks) &&
        NJson::ParseField(value["triggered_script_id"], object.TriggeredScriptId) &&
        NJson::ParseField(value["untriggered_script_id"], object.UntriggeredScriptId) &&
        NJson::ParseField(value["timeout"], object.Timeout);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCheck::EType>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCheck::EType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCheck::EOperation>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCheck::EOperation& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCheckValue>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCheckValue& object) {
    bool result = NJson::ParseField(value["type"], object.Type);

    if (!result) {
        return false;
    }

    object.Value.Type = object.Type;
    return result && NJson::ParseField(value["value"], object.Value);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCheck>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCheck& object) {
    return
        NJson::ParseField(value["type"], object.Type, true) &&
        NJson::ParseField(value["operation"], object.Operation, true) &&
        NJson::ParseField(value["sensor_id"], object.SensorId, true) &&
        NJson::ParseField(value["sensor_sub_id"], object.SensorSubId) &&
        NJson::ParseField(value["value"], object.Value, true) &&
        NJson::ParseField(value["notify"], object.NotifyId) &&
        NJson::ParseField(value["argument"], object.Argument);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCommandSetValue>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCommandSetValue& object) {
    return
        NJson::ParseField(value["sensor_id"], object.Id, true) &&
        NJson::ParseField(value["sensor_sub_id"], object.SubId) &&
        NJson::ParseField(value["value"], object.Value, true);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCommand::EType>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCommand::EType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCommand::TArgument>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCommand::TArgument& object) {
    NDrive::NVega::TLogicCommandSetValue setValue;
    ui64 variable = 0;

    if (NJson::ParseField(value["set_value"], setValue, true)) {
        object = setValue;
        return true;
    }

    if (ParseField(value["variable"], variable, true)) {
        object = variable;
        return true;
    }

    return false;
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicCommand>(const NJson::TJsonValue& value, NDrive::NVega::TLogicCommand& object) {
    return
        NJson::ParseField(value["type"], object.Type, true) &&
        NJson::ParseField(value["argument"], object.Argument, true);
}

template<>
bool NJson::TryFromJson<NDrive::NVega::TLogicNotify>(const NJson::TJsonValue& value, NDrive::NVega::TLogicNotify& object) {
    TString message;
    if (!NJson::ParseField(value["message"], message, true)) {
        return false;
    }

    size_t maxSize = object.Message.CalcSize() - 1;
    if (message.Size() > maxSize) {
        message.resize(maxSize);
    }
    object.Message.Set(message);
    return true;
}
