#pragma once

#include "sensor.h"
#include "types.h"
#include "vega.h"

#include <library/cpp/json/json_value.h>

#include <util/generic/map.h>
#include <util/stream/buffer.h>

namespace NDrive {
    namespace NVega {
        using TBlePasskeyParameter = NProtocol::TFixedSizeZeroTerminatedString<7>;

        struct TBleSessionKeyParameter {
        public:
            std::array<ui8, 32> Value;

            DEFINE_FIELDS(
                Value
            );

        public:
            static ui16 GetId();

        public:
            TBleSessionKeyParameter() {
                Value.fill(0);
            }

            TConstArrayRef<char> Raw() const;
        };

        struct TServerSettingsParameter {
        public:
            enum EProtocol : ui8 {
                DISABLED = 0,
                EGTS     = 1,
                WIPS     = 2,
                VEGA     = 3,
                NDTP     = 4,
                WCOMBINE = 5,
            };

            using TType = TBuffer;

        public:
            NProtocol::TFixedSizeZeroTerminatedString<32> HostPort;
            ui16 InteractionPeriod = 0;
            ui8 Protocol = DISABLED;

            DEFINE_FIELDS(
                HostPort,
                InteractionPeriod,
                Protocol
            );

        public:
            static ui16 GetId();
            static ui16 GetCount();

        public:
            NJson::TJsonValue ToJson() const;
            bool IsEmpty() const {
                return false;
            }

            ui64 CalcHash() const;
            void Parse(TConstArrayRef<char> data);

            TType Dump() const;
            void Parse(const TType& data);
        };

        struct TAPNParameter {
        public:
            using TType = TBuffer;

        public:
            NProtocol::TFixedSizeZeroTerminatedString<30> APN;
            NProtocol::TFixedSizeZeroTerminatedString<16> User;
            NProtocol::TFixedSizeZeroTerminatedString<16> Password;

            DEFINE_FIELDS(
                APN,
                User,
                Password
            );

        public:
            static ui16 GetId();
            static ui16 GetCount();

        public:
            NJson::TJsonValue ToJson() const;
            bool IsEmpty() const {
                return false;
            }

            ui64 CalcHash() const;
            void Parse(TConstArrayRef<char> data);

            TType Dump() const;
            void Parse(const TType& data);
        };

        struct TNetworkScanModeParameter {
            enum {
                AUTO              = 0,
                GSM_ONLY          = 1,
                WCDMA_ONLY        = 2,
                LTE_ONLY          = 3,
                TD_SCDMA_ONLY     = 4,
                UMTS_ONLY         = 5,
                CDMA_ONLY         = 6,
                HDR_ONLY          = 7,
                CDMA_AND_HDR_ONLY = 8,
            };
        };

        struct THardPasswordParameter {
        public:
            using TType = TBuffer;

        public:
            NProtocol::TFixedSizeZeroTerminatedString<17> Value;

            DEFINE_FIELDS(Value);

        public:
            static ui16 GetId();

        public:
            ui64 CalcHash() const;
            void Parse(TConstArrayRef<char> data);

            TType Dump() const;
            void Parse(const TType& data);
        };

        struct TUsePasswordParameter {
        public:
            enum EMode : ui8 {
                DISABLE = 1,
                ENABLE = 2,
            };

            using TType = ui64;

        public:
            ui8 Mode = DISABLE;

            DEFINE_FIELDS(Mode);

        public:
            static ui16 GetId();

        public:
            ui64 CalcHash() const;

            TType Dump() const;
            void Parse(TType data);
        };

        struct TWiredPasswordParameter : public THardPasswordParameter {
        public:
            static ui16 GetId();
        };

        struct TUseWiredPasswordParameter : public TUsePasswordParameter {
        public:
            static ui16 GetId();
        };

        struct TSensorTranslation {
        public:
            using TType = TBuffer;

        public:
            ui8 OnTrack = 0;
            ui16 OnPeriod = 0;
            ui8 OnChange = 0;

            DEFINE_FIELDS(
                OnTrack,
                OnPeriod,
                OnChange
            );

            bool IsEmpty() const {
                return OnTrack == 0 && OnPeriod == 0 && OnChange == 0;
            }

        public:
            static ui16 GetId();

        public:
            NJson::TJsonValue ToJson() const;

            ui64 CalcHash() const;
            void Parse(TConstArrayRef<char> data);

            TType Dump() const;
            void Parse(const TType& data);
        };

        struct TSetting {
        public:
            ui16 Id = 0;
            ui16 SubId = 0;
            NProtocol::TArray<char, ui16> Payload;

            DEFINE_FIELDS(
                Id,
                SubId,
                Payload
            );

        public:
            NJson::TJsonValue ToJson() const;

        public:
            static NJson::TJsonValue FormatValue(TConstArrayRef<char> data, ui16 id);
        };

        class TSettings: public TVector<TSetting> {
        private:
            using TBase = TVector<TSetting>;

        public:
            using TBase::TBase;

            NJson::TJsonValue ToJson() const;
            TBuffer ToSigned(std::array<ui32, 4> key) const;

            TBuffer Dump() const;
            void Parse(TConstArrayRef<char> data);
        };

		enum ECommonValueType: ui8 {
            CVT_UNKNOWN = 0 /* "unknown" */,
			CVT_UINT8 = 1 /* "uint8_t" */,
			CVT_UINT16 = 2 /* "uint16_t" */,
			CVT_UINT32 = 3 /* "uint32_t" */,
			CVT_FLOAT = 4 /* "float" */,
			CVT_STRING = 5 /* "string" */,
			CVT_ARRAY = 6 /* "array" */,
			CVT_BOOL = 7 /* "bool" */,
			CVT_UINT64 = 8 /* "ui64" */,
			CVT_INT8 = 9 /* "i8" */,
			CVT_INT16 = 10 /* "i16" */,
			CVT_INT32 = 11 /* "i32" */,
			CVT_INT64 = 12 /* "i64" */,
			CVT_DOUBLE = 13 /* "double" */,
			CVT_UNITED = 14 /* "united" */,
		};

		class TUnitedValue {
		public:
			enum EType {
				T_UINT64 = 0,
				T_INT64 = 1,
				T_UINT32 = 2,
				T_INT32 = 3,
				T_UINT16 = 4,
				T_INT16 = 5,
				T_UINT8 = 6,
				T_INT8 = 7,
				T_FLOAT = 8,
				T_DOUBLE = 9,
				T_BOOL = 10
			};

		public:
			using TValue = std::variant<ui64, i64, ui32, i32, ui16, i16, ui8, i8, float, double, bool>;

		public:
			TValue Value;
			ECommonValueType Type = CVT_UINT64;

			size_t CalcSize() const {
				return sizeof(ui64);
			}

			TUnitedValue::EType GetTypeIndex();
			void Load(IInputStream* stream);
			void Save(IOutputStream* stream) const;

			template<class To>
			bool TryConvertTo(To& data) {
				if (std::holds_alternative<To>(Value)) {
					data = std::get<To>(Value);
					return true;
				}
				return false;
			}

		private:
			inline static const TMap<ECommonValueType, EType> TypeMap = {
				{ CVT_UINT64, T_UINT64 },
				{ CVT_INT64, T_INT64 },
				{ CVT_UINT32, T_UINT32 },
				{ CVT_INT32, T_INT32 },
				{ CVT_UINT16, T_UINT16 },
				{ CVT_INT16, T_INT16 },
				{ CVT_UINT8, T_UINT8 },
				{ CVT_INT8, T_INT8 },
				{ CVT_FLOAT, T_FLOAT },
				{ CVT_DOUBLE, T_DOUBLE },
				{ CVT_BOOL, T_BOOL }
			};
		};

        class TCanParameter {
        public:
            enum EIdentifierType: ui8 {
                IT_EMPTY = 0 /* "none" */,
                IT_STANDARD = 1 /* "standard" */,
                IT_EXTENDED = 2 /* "extended" */,
                IT_EXTENDED_PGN = 3 /* "extended_pgn" */,
            };

            enum ECanChannel: ui8 {
                CC_FIRST = 0 /* "first" */,
                CC_SECOND = 1 /* "second" */,
                CC_THIRD = 2 /* "third" */,
            };

            class TCanBusSetting {
            public:
                ECanChannel CanChannel = CC_FIRST;
                ui8 Version = 1;

                size_t CalcSize() const {
                    return sizeof(ui8);
                }

                void Load(IInputStream* stream);
                void Save(IOutputStream* stream) const;
            };

            class THeaderSetting {
            public:
                EIdentifierType IdentifierType = IT_STANDARD;
                ui8 StructureType = 0; //always 0
                ui8 SettingType = 0;
                bool IsCrypted = false;

                size_t CalcSize() const {
                    return sizeof(ui8);
                }

                void Load(IInputStream* stream);
                void Save(IOutputStream* stream) const;
            };

            class TSettings {
            public:
                class TLength {
                public:
                    ui8 Length = 0;
                    bool IsSigned = false;

                    size_t CalcSize() const {
                        return sizeof(ui8);
                    }

                    void Load(IInputStream* stream);
                    void Save(IOutputStream* stream) const;

                    NJson::TJsonValue ToJson() const;
                };

                class TDestination {
                public:
                    ui8 Position = 0;

                    DEFINE_FIELDS(
                        Position
                    )

                    NJson::TJsonValue ToJson() const;
                };

                class TBorders {
                public:
                    i64 Minimum = 0;
                    i64 Maximum = 0;

                    DEFINE_FIELDS(
                        Minimum,
                        Maximum
                    )

                    NJson::TJsonValue ToJson() const;
                };

                class TMask {
                public:
                    ui64 Mask = 0;
                    ui64 Value = 0;

                    DEFINE_FIELDS(
                        Mask,
                        Value
                    )

                    NJson::TJsonValue ToJson() const;
                };

                class TResetSettings {
                public:
                    TResetSettings() = default;

                    TResetSettings(const TResetSettings&) = default;
                    TResetSettings& operator=(const TResetSettings&) = default;

                    ui8 ResetSeconds = 0;
                    bool ResetByIgnitionOff = false;
                    bool FastBool = false;
                    bool IsMaskFilter = false;
                    bool IsDestination = false;

                    size_t CalcSize() const {
                        return sizeof(ui8);
                    }

                    void Load(IInputStream* stream);
                    void Save(IOutputStream* stream) const;

                    NJson::TJsonValue ToJson() const;
                };

                class TPositions {
                public:
                    enum EOrder: bool {
                        O_LITTLE_ENDIAN = false /* "little_endian" */,
                        O_BIG_ENDIAN = true /* "big_endian" */,
                    };

                    EOrder Order = O_LITTLE_ENDIAN;
                    bool Invert = false;
                    ui8 ByteIndex = 0;
                    ui8 StartBit = 0;

                    size_t CalcSize() const {
                        return sizeof(ui8);
                    }

                    void Load(IInputStream* stream);
                    void Save(IOutputStream* stream) const;

                    NJson::TJsonValue ToJson() const;
                };

            public:
                using TBits = std::variant<TLength, TDestination>;
                using TFilter = std::variant<TBorders, TMask>;

            public:
                ui32 CanId = 0;
                TUnitedValue ResetValue;
                TResetSettings ResetSettings;

                TPositions Positions;
                std::variant<TLength, TDestination> Bits;
                std::variant<TBorders, TMask> Filter;

                float Scale = 1.;
                float Offset = 0.;

                std::array<ui8, 3> Reserve = MakeFilledArray(0);

                size_t CalcSize() const;
                ui16 CalcCrc() const;
                void Load(IInputStream* stream);
                void Save(IOutputStream* stream) const;

                NJson::TJsonValue ToJson() const;
            };

            class TName {
            public:
                using TValue = NProtocol::TFixedSizeZeroTerminatedString<30>;

            public:
                DEFINE_FIELDS(
                    Value
                )

                TString Get() const;
                void Set(TStringBuf value);

            private:
                TValue Value;

            private:
                bool IsUTF8(TStringBuf value) const;
            };

            using TType = TBuffer;

        public:
            TCanBusSetting CanBusSetting;
            THeaderSetting HeaderSetting;
            std::array<ui8, 4> Nonce = MakeFilledArray(0);

            ECommonValueType Type = CVT_UINT8;
            ui16 Id;
            TName Name;
            ui8 Reserve;

            TSettings Settings;

        public:
            size_t CalcSize() const;
            ui64 CalcHash() const;
            TType Dump() const;
            NJson::TJsonValue ToJson() const;
            bool IsEmpty() const;

            void Load(IInputStream* stream);
            void Parse(const TType& data);

            static ui16 GetId();
            static ui16 GetCount();
            static TSensor Parse(TSensor&& sensor, const TVector<TCanParameter>& paramteres);
        };

        class TCanParameterName {
        public:
            using TName = NProtocol::TFixedSizeZeroTerminatedString<48>;

            TName Name;

            static ui16 GetId();
            ui64 CalcHash() const;
            TBuffer Dump();
        };

        template<ui16 SettingId, ui16 Count>
        class ILogicSetting {
        public:
            static ui16 GetId() {
                return SettingId;
            }
            static ui16 GetCount() {
                return Count;
            }

            virtual size_t GetSize() const = 0;
            virtual void Load(IInputStream& stream) = 0;
            virtual void Save(IOutputStream& stream) const = 0;
            virtual NJson::TJsonValue ToJson() const = 0;

            TBuffer Dump() const {
                TBuffer result(GetSize());
                TBufferOutput stream(result);

                Save(stream);
                return result;
            }

            void Parse(const TBuffer& data) {
                TBufferInput stream(data);
                Load(stream);
            }

            ui64 CalcHash() const {
                auto data = Dump();
                return FnvHash<ui64>(data);
            }

            bool IsEmpty() const {
                TBuffer emptyValue;
                emptyValue.Fill(0, GetSize());
                return Dump() == emptyValue;
            }

        };

        class TLogicStep {
        public:
            enum EType: ui8 {
				T_UNKNOWN = 0 /* "unknown" */,
				T_COMMAND = 1 /* "command" */,
				T_CHECK = 2 /* "check" */,
				T_CANCEL = 3 /* "cancel" */,
				T_SUBSCRIPT = 4 /* "subscript" */,
            };

		public:
			EType Type = T_UNKNOWN;
			ui16 ActionId = 0;

			DEFINE_FIELDS(
				Type,
				ActionId
			)

            static constexpr size_t GetSize() {
                return sizeof(Type) + sizeof(ActionId);
            }

            bool IsEmpty() const {
                return Type == T_UNKNOWN && ActionId == 0;
            }

            NJson::TJsonValue ToJson() const;
        };

		class TLogicActiveId {
		public:
			bool Active = false;
			ui16 Id = 0;

			DEFINE_FIELDS(
				Active,
				Id
			)

            TLogicActiveId() = default;
            TLogicActiveId(bool active, ui16 id)
                : Active(active)
                , Id(id)
            {
            }

            bool IsEmpty() const {
                return Active == false && Id == 0;
            }

            inline bool operator==(const TLogicActiveId& other) {
                return std::tuple(Active, Id) == std::tuple(other.Active, other.Id);
            }

            static constexpr size_t GetSize() {
                return sizeof(Active) + sizeof(Id);
            }

            NJson::TJsonValue ToJson() const;
		};

        class TLogicSteps {
        public:
            static constexpr size_t Count = 40;

        public:
            using TSteps = TMap<ui32, TLogicStep>;

        public:
            TSteps Steps;

        public:
            inline size_t CalcSize() const {
                return TLogicStep::GetSize() * Count;
            }

            bool IsEmpty() const {
                return Steps.empty();
            }

            void Load(IInputStream* stream);
            void Save(IOutputStream* stream) const;
            NJson::TJsonValue ToJson() const;
        };

		class TLogicScript: public ILogicSetting<VEGA_SETTINGS_SCRIPT_SCRIPTS, LogicScriptScriptsCount> {
		public:
			static constexpr size_t NameSize = 64;
			static constexpr size_t StepSize = 40;

		public:
			using TName = NProtocol::TFixedSizeZeroTerminatedString<NameSize>;

		public:
			ui16 CommandId = 0;
			TName Name;
			TLogicSteps Steps;

			DEFINE_FIELDS(
				CommandId,
				Name,
				Steps
			)

            size_t GetSize() const override {
                return CalcSize();
            }
            void Load(IInputStream& stream) override {
                Load(&stream);
            }
            void Save(IOutputStream& stream) const override {
                Save(&stream);
            }
            NJson::TJsonValue ToJson() const override;
		};

        class TLogicTriggerChecks {
        public:
            static constexpr size_t Count = 6;

        public:
            using TChecks = TMap<ui32, TLogicActiveId>;

        public:
            TChecks Checks;

            inline size_t CalcSize() const {
                return TLogicActiveId::GetSize() * Count;
            }

            void Load(IInputStream* stream);
            void Save(IOutputStream* stream) const;
            NJson::TJsonValue ToJson() const;
        };

		class TLogicTrigger: public ILogicSetting<VEGA_SETTINGS_SCRIPT_TRIGGERS, LogicScriptTriggersCount> {
		public:
			enum EType: ui8 {
				T_UNKNOWN = 0 /* "unknown" */,
				T_AND = 1 /* "and" */,
				T_OR = 2 /* "or" */,
			};

		public:
			EType Type = T_UNKNOWN;
			TLogicTriggerChecks Checks;
			TLogicActiveId TriggeredScriptId;
			TLogicActiveId UntriggeredScriptId;
			ui32 Timeout = 0;

			DEFINE_FIELDS(
				Type,
				Checks,
				TriggeredScriptId,
				UntriggeredScriptId,
                Timeout
			)

            size_t GetSize() const override {
                return CalcSize();
            }
            void Load(IInputStream& stream) override {
                Load(&stream);
            }
            void Save(IOutputStream& stream) const override {
                Save(&stream);
            }
            NJson::TJsonValue ToJson() const override;
		};

		class TLogicNotify: public ILogicSetting<VEGA_SETTINGS_SCRIPT_NOTIFIES, LogicScriptNotifiesCount> {
		public:
			static constexpr size_t MessageSize = 64;

		public:
			using TMessage = NProtocol::TFixedSizeZeroTerminatedString<MessageSize>;

		public:
			TMessage Message;

			DEFINE_FIELDS(
				Message
			)

            size_t GetSize() const override {
                return CalcSize();
            }
            void Load(IInputStream& stream) override {
                Load(&stream);
            }
            void Save(IOutputStream& stream) const override {
                Save(&stream);
            }
            NJson::TJsonValue ToJson() const override;
		};

        class TLogicCheckValue {
        public:
            TUnitedValue Value;
			ECommonValueType Type = CVT_UINT64;

            template<class To>
			bool TryConvertTo(To& data) {
                return Value.TryConvertTo(data);
            }

            size_t CalcSize() const;
            void Load(IInputStream* stream);
            void Save(IOutputStream* stream) const;
            NJson::TJsonValue ToJson() const;
        };

		class TLogicCheck: public ILogicSetting<VEGA_SETTINGS_SCRIPT_CHECKS, LogicScriptChecksCount> {
		public:
			enum EType: ui8 {
				T_UNKNOWN = 0 /* "unknown" */,
				T_BREAK = 1 /* "break" */,
				T_CONTINUE = 2 /* "continue" */,
				T_SKIP = 3 /* "skip" */,
			};

			enum EOperation: ui8 {
				O_UNKNOWN = 0 /* "unknown" */,
				O_EQUAL = 1 /* "equal" */,
				O_MORE = 2 /* "more" */,
				O_MORE_OR_EQUAL = 3 /* "more_or_equal" */,
				O_LESS = 4 /* "less" */,
				O_LESS_OR_EQUAL = 5 /* "less_or_equal" */,
			};

		public:
			EType Type = T_UNKNOWN;
			EOperation Operation = O_UNKNOWN;
            ui16 SensorId = 0;
            ui16 SensorSubId = 0;
			TLogicCheckValue Value;
			TLogicActiveId NotifyId;
			ui32 Argument = 0;

            DEFINE_FIELDS(
                Type,
                Operation,
                SensorId,
                SensorSubId,
                Value,
                NotifyId,
                Argument
            )

            size_t GetSize() const override {
                return CalcSize();
            }
            void Load(IInputStream& stream) override {
                Load(&stream);
            }
            void Save(IOutputStream& stream) const override {
                Save(&stream);
            }
            NJson::TJsonValue ToJson() const override;
		};

        class TLogicCommandSetValue {
        public:
            ui16 Id = 0;
            ui16 SubId = 0;
            ui32 Value = 0;

            DEFINE_FIELDS(
                Id,
                SubId,
                Value
            )

            NJson::TJsonValue ToJson() const;
        };

        class TLogicCommand: public ILogicSetting<VEGA_SETTINGS_SCRIPT_COMMANDS, LogicScriptCommandsCount> {
        public:
            enum EType: ui16 {
                T_UNKNOWN = 0,
                T_RUN_CAN_SCRIPT,
                T_ON_DIGITAL_OUTPUT,
                T_OFF_DIGITAL_OUTPUT,
                T_DELAY,
                T_NOTIFY,
                T_SET_SENSOR,
            };

        public:
            using TArgument = std::variant<ui64, TLogicCommandSetValue>;

        public:
            EType Type = T_UNKNOWN;
            TArgument Argument;

            inline size_t CalcSize() const {
                return sizeof(Type) + sizeof(ui64);
            }

            void Load(IInputStream* stream);
            void Save(IOutputStream* stream) const;

            template<typename T>
            void LoadArgument(IInputStream* stream) {
                T argument;
                ::Load(stream, argument);
                Argument = argument;
            }

            size_t GetSize() const override {
                return CalcSize();
            }
            void Load(IInputStream& stream) override {
                Load(&stream);
            }
            void Save(IOutputStream& stream) const override {
                Save(&stream);
            }
            NJson::TJsonValue ToJson() const override;
        };

        struct TSvrRawState {
        public:
            ui8 PollState               = 0;
            ui64 SerialNumber           = 0;
            ui16 EngineRpm              = 0;
            ui16 Voltage                = 0;
            ui16 MAP                    = 0;
            ui16 SYP                    = 0;
            i16 ReductorTemperature     = 0;
            i16 GasTemperature          = 0;
            i16 PcbTemperature          = 0;
            ui8 PetrolLoad              = 0;
            ui8 GasLoad                 = 0;
            ui8 EngineLoad              = 0;
            ui8 EcuState                = 0;
            ui8 TankLevelAverage        = 0;
            ui8 DtcCount                = 0;
            ui8 EcuResetCounter         = 0;
            ui8 EmergencyStartCounter   = 0;
            i16 MaxPcbTemperature       = 0;
            ui32 WorkingOnPetrol        = 0;
            ui32 WorkingOnGas           = 0;
            ui16 DistanceLeft           = 0;

            DEFINE_FIELDS(
                PollState,
                SerialNumber,
                EngineRpm,
                Voltage,
                MAP,
                SYP,
                ReductorTemperature,
                GasTemperature,
                PcbTemperature,
                PetrolLoad,
                GasLoad,
                EngineLoad,
                EcuState,
                TankLevelAverage,
                DtcCount,
                EcuResetCounter,
                EmergencyStartCounter,
                MaxPcbTemperature,
                WorkingOnPetrol,
                WorkingOnGas,
                DistanceLeft
            );

        public:
            template <ui16 N>
            static constexpr TSensorId Field { 8350, N };

            static constexpr auto PollStateId = Field<1>;
            static constexpr auto SerialNumberId = Field<2>;
            static constexpr auto EngineRpmId = Field<3>;
            static constexpr auto VoltageId = Field<4>;
            static constexpr auto MapId = Field<5>;
            static constexpr auto SypId = Field<6>;
            static constexpr auto ReductorTemperatureId = Field<7>;
            static constexpr auto GasTemperatureId = Field<8>;
            static constexpr auto PcbTemperatureId = Field<9>;
            static constexpr auto PetrolLoadId = Field<10>;
            static constexpr auto GasLoadId = Field<11>;
            static constexpr auto EngineLoadId = Field<12>;
            static constexpr auto EcuStateId = Field<13>;
            static constexpr auto TankLevelAverageId = Field<14>;
            static constexpr auto DtcCountId = Field<15>;
            static constexpr auto EcuResetCounterId = Field<16>;
            static constexpr auto EmergencyStartCounterId = Field<17>;
            static constexpr auto MaxPcbTemperatureId = Field<18>;
            static constexpr auto WorkingOnPetrolId = Field<19>;
            static constexpr auto WorkingOnGasId = Field<20>;
            static constexpr auto DistanceLeftId = Field<21>;

            static constexpr ui16 GetId() {
                return Field<0>.Id;
            }

        public:
            float GetVoltage() const {
                return 0.1f * Voltage;
            }
            float GetMap() const {
                return 0.01f * MAP;
            }
            float GetSyp() const {
                return 0.01f * SYP;
            }
            float GetReductorTemperature() const {
                return 0.1f * ReductorTemperature;
            }
            float GetGasTemperature() const {
                return 0.1f * GasTemperature;
            }
            float GetPbcTemperature() const {
                return 0.1f * PcbTemperature;
            }
            float GetTankLevelAverage() const {
                return 0.1f * TankLevelAverage;
            }
            float GetMaxPcbTemperature() const {
                return 0.1f * MaxPcbTemperature;
            }

            void Parse(TConstArrayRef<char> data);
        };

        struct TSvrDtcState {
        public:
            NProtocol::TFixedSizeZeroTerminatedString<63> StringValue;

            DEFINE_FIELDS(
                StringValue
            );

        public:
            static constexpr ui16 GetId() {
                return 8351;
            }

        public:
            void Parse(TConstArrayRef<char> data);
        };
    }
}
