
#include "crc.h"
#include "settings.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/json/cast.h>

#include <util/stream/str.h>
#include <util/string/hex.h>

Y_UNIT_TEST_SUITE(TelematicsSettingsSuite) {
    Y_UNIT_TEST(SerializationAndDeserialization) {
        TString data = HexDecode(
            "080100000000D25F02F20ACEE1EEF0EE" \
            "F2FB0000000000000000000000000000" \
            "00000000000000000000160300000000" \
            "00000000000015101190010000000000" \
            "00FFFF0000000000000000803E000000" \
            "00000000"
        );
        TBuffer buffer(data.Data(), data.Size());

        NDrive::NVega::TCanParameter parameter;
        parameter.Parse(buffer);

        auto& canBusSetting = parameter.CanBusSetting;
        auto& headerSetting = parameter.HeaderSetting;
        auto& settings = parameter.Settings;

        auto& value = settings.ResetValue.Value;
        auto& filter = settings.Filter;
        auto& positions = settings.Positions;
        auto& bits = settings.Bits;

        UNIT_ASSERT_EQUAL(canBusSetting.CanChannel, 0);
        UNIT_ASSERT_EQUAL(canBusSetting.Version, 1);
        UNIT_ASSERT_EQUAL(headerSetting.IdentifierType, NDrive::NVega::TCanParameter::IT_STANDARD);
        UNIT_ASSERT_EQUAL(headerSetting.SettingType, 0);
        UNIT_ASSERT_EQUAL(headerSetting.IsCrypted, false);
        UNIT_ASSERT_EQUAL(parameter.Type, NDrive::NVega::CVT_UINT16);
        UNIT_ASSERT_EQUAL(parameter.Id, 2802);
        UNIT_ASSERT(!parameter.Name.Get().Empty());
        UNIT_ASSERT_EQUAL(settings.CanId, 0x316);
        UNIT_ASSERT_EQUAL(settings.ResetValue.Type, NDrive::NVega::CVT_UINT16);

        UNIT_ASSERT(std::holds_alternative<ui16>(value) && std::get<ui16>(value) == 0);

        UNIT_ASSERT(std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TBorders>(filter));
        auto& borders = std::get<NDrive::NVega::TCanParameter::TSettings::TBorders>(filter);
        UNIT_ASSERT_EQUAL(borders.Minimum, 0x190);
        UNIT_ASSERT_EQUAL(borders.Maximum, 0xFFFF);

        UNIT_ASSERT(!positions.Invert);
        UNIT_ASSERT_EQUAL(positions.ByteIndex, 4);
        UNIT_ASSERT_EQUAL(positions.Order, NDrive::NVega::TCanParameter::TSettings::TPositions::O_BIG_ENDIAN);
        UNIT_ASSERT_EQUAL(positions.StartBit, 0);

        UNIT_ASSERT(std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TLength>(bits));
        auto& length = std::get<NDrive::NVega::TCanParameter::TSettings::TLength>(bits);
        UNIT_ASSERT_EQUAL(length.Length, 16);

        UNIT_ASSERT_EQUAL(settings.Scale, 0.25);
        UNIT_ASSERT_EQUAL(settings.Offset, 0);

        auto saveBuffer = parameter.Dump();
        UNIT_ASSERT_EQUAL(saveBuffer, buffer);

        auto json = parameter.ToJson();
        UNIT_ASSERT(NJson::TryFromJson(json, parameter));
    }

    Y_UNIT_TEST(JsonParse) {
        using TResetValue = NDrive::NVega::TUnitedValue;
        NJson::TJsonValue value;
        TResetValue unsignedValue;
        TResetValue signedValue;
        TResetValue floatingValue;

        unsignedValue.Type = NDrive::NVega::CVT_UINT32;
        signedValue.Type = NDrive::NVega::CVT_INT8;
        floatingValue.Type = NDrive::NVega::CVT_DOUBLE;

        value = ui32(15);
        UNIT_ASSERT(NJson::TryFromJson(value, unsignedValue));

        value = i8(-12);
        UNIT_ASSERT(NJson::TryFromJson(value, signedValue));

        value = double(1.5);
        UNIT_ASSERT(NJson::TryFromJson(value, floatingValue));

        value = "Text";
        UNIT_ASSERT(!NJson::TryFromJson(value, floatingValue));
    }

    Y_UNIT_TEST(JsonDeserialize) {
        const TString jsonCanParameter = R"({
            "sensor_id": 2800,
            "is_crypted": "0",
            "can_id_type": "standard",
            "can_channel": "first",
            "settings": {
                "offset": 0,
                "bits": {
                    "length": {
                        "is_signed": false,
                        "length": 28
                    }
                },
                "reset_settings": {
                    "fast_bool": false,
                    "reset_by_ignition_off": false,
                    "reset_seconds": 0
                },
                "positions": {
                    "start_bit": 4,
                    "byte_index": 5,
                    "invert": false,
                    "bit_order": "big_endian"
                },
                "reset_value": 0,
                "can_id": 1495,
                "scale": 0.01,
                "filter": {
                    "borders": {
                        "minimum": 0,
                        "maximum": 268435455
                    }
                }
            },
            "sensor_type": "uint32_t"
        })";

        NJson::TJsonValue json;
        UNIT_ASSERT(NJson::ReadJsonFastTree(jsonCanParameter, &json));

        NDrive::NVega::TCanParameter parameter;
        UNIT_ASSERT(NJson::TryFromJson(json, parameter));

        UNIT_ASSERT(std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TLength>(parameter.Settings.Bits));

        auto& bits = std::get<NDrive::NVega::TCanParameter::TSettings::TLength>(parameter.Settings.Bits);
        UNIT_ASSERT_EQUAL(bits.IsSigned, false);
        UNIT_ASSERT_EQUAL(bits.Length, 28);

        UNIT_ASSERT(std::holds_alternative<NDrive::NVega::TCanParameter::TSettings::TBorders>(parameter.Settings.Filter));

        auto& filter = std::get<NDrive::NVega::TCanParameter::TSettings::TBorders>(parameter.Settings.Filter);
        UNIT_ASSERT_EQUAL(filter.Minimum, 0);
        UNIT_ASSERT_EQUAL(filter.Maximum, 0xFFFFFFF);
    }

    Y_UNIT_TEST(LogicalScript) {
        TString data = HexDecode(
            "2C00537461727420656E67696E652063"\
            "6F6D6D616E6400000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000100000102000200000101000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000"
        );

        TBuffer buffer(data.Data(), data.Size());

        NDrive::NVega::TLogicScript script;
        script.Parse(buffer);

        UNIT_ASSERT_EQUAL(script.CommandId, 44);
        UNIT_ASSERT_STRINGS_EQUAL(script.Name.Get(), "Start engine command");

        auto saveBuffer = script.Dump();
        UNIT_ASSERT_EQUAL(saveBuffer, buffer);
    }

    Y_UNIT_TEST(LogicalCheck) {
        TString data = HexDecode("0102F00A000090010000000000000201010000000000");
        TBuffer buffer(data.Data(), data.Size());

        NDrive::NVega::TLogicCheck check;
        check.Parse(buffer);

        UNIT_ASSERT_EQUAL(check.Type, NDrive::NVega::TLogicCheck::T_BREAK);
        UNIT_ASSERT_EQUAL(check.Operation, NDrive::NVega::TLogicCheck::O_MORE);
        UNIT_ASSERT_EQUAL(check.SensorId, NDrive::TSensorId(2800));
        UNIT_ASSERT_EQUAL(check.NotifyId, NDrive::NVega::TLogicActiveId(true, 1));
        UNIT_ASSERT_EQUAL(check.Argument, 0);

        ui16 value = 0;
        UNIT_ASSERT(check.Value.TryConvertTo(value));
        UNIT_ASSERT_EQUAL(value, 400);

        auto saveBuffer = check.Dump();
        UNIT_ASSERT_EQUAL(saveBuffer, buffer);
    }

    Y_UNIT_TEST(LogicalTrigger) {
        TString data = HexDecode("0101010000000000000000000000000000000001030000000030750000");
        TBuffer buffer(data.Data(), data.Size());

        NDrive::NVega::TLogicTrigger trigger;
        trigger.Parse(buffer);

        UNIT_ASSERT_EQUAL(trigger.Type, NDrive::NVega::TLogicTrigger::T_AND);
        UNIT_ASSERT_EQUAL(trigger.Checks.Checks.size(), 1);
        UNIT_ASSERT_EQUAL(trigger.TriggeredScriptId, NDrive::NVega::TLogicActiveId(true, 3));
        UNIT_ASSERT_EQUAL(trigger.UntriggeredScriptId, NDrive::NVega::TLogicActiveId(false, 0));
        UNIT_ASSERT_EQUAL(trigger.Timeout, 30000);

        auto saveBuffer = trigger.Dump();
        UNIT_ASSERT_EQUAL(saveBuffer, buffer);
    }

    Y_UNIT_TEST(LogicalCommand) {
        {
            TString data = HexDecode("06003C08000001000000");
            TBuffer buffer(data.Data(), data.Size());

            NDrive::NVega::TLogicCommand command;
            command.Parse(buffer);

            UNIT_ASSERT_EQUAL(command.Type, NDrive::NVega::TLogicCommand::T_SET_SENSOR);
            UNIT_ASSERT(std::holds_alternative<NDrive::NVega::TLogicCommandSetValue>(command.Argument));

            auto& argument = std::get<NDrive::NVega::TLogicCommandSetValue>(command.Argument);
            UNIT_ASSERT_EQUAL(argument.Id, 2108);
            UNIT_ASSERT_EQUAL(argument.SubId, 0);
            UNIT_ASSERT_EQUAL(argument.Value, 1);

            auto saveBuffer = command.Dump();
            UNIT_ASSERT_EQUAL(saveBuffer, buffer);
        }

        {
            TString data = HexDecode("04008813000000000000");
            TBuffer buffer(data.Data(), data.Size());

            NDrive::NVega::TLogicCommand command;
            command.Parse(buffer);

            UNIT_ASSERT_EQUAL(command.Type, NDrive::NVega::TLogicCommand::T_DELAY);
            UNIT_ASSERT(std::holds_alternative<ui64>(command.Argument));

            auto& argument = std::get<ui64>(command.Argument);
            UNIT_ASSERT_EQUAL(argument, 5000);
        }
    }

    Y_UNIT_TEST(LogicNotify) {
        TString data = HexDecode(
            "656E67696E6520737461727465640000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"\
            "00000000000000000000000000000000"
        );
        TBuffer buffer(data.Data(), data.Size());

        NDrive::NVega::TLogicNotify notify;
        notify.Parse(buffer);

        UNIT_ASSERT_STRINGS_EQUAL(notify.Message.Get(), "engine started");
    }
}
