#pragma once

#include "common_private.h"

#include <rtline/util/types/array.h>

#include <util/generic/array_ref.h>

namespace NDrive {
    namespace NProtocol {
        template <class T, class H>
        using TArray = ::NPrivate::TVectorWithTypedSize<T, H>;

        template <class T, class L>
        using TLArray = ::NPrivate::TVectorWithTypedContentLength<T, L>;

        template <class T>
        struct TStructure {
            union {
                T Data;
                std::array<ui8, sizeof(T)> Serializer;
            };

            TStructure() {
                Serializer.fill(0);
            }

            DEFINE_FIELDS(
                Serializer
            );
        };

        template<char Separator>
        class TSeparatedString {
        public:
            inline TSeparatedString() = default;
            inline TSeparatedString(TString&& value)
                : Value(std::move(value))
            {
            }

            template<char OtherSeparator>
            TSeparatedString& operator=(const TSeparatedString<OtherSeparator>& other) {
                Value = other.Value;
                return *this;
            }

            TSeparatedString& operator=(TStringBuf other) {
                Value = TString(other);
                return *this;
            }

            inline static char GetSeparator() {
                return Separator;
            }
            size_t CalcSize() const {
                return Value.Size();
            }
            void Load(IInputStream* input) {
                if (input) {
                    Value = input->ReadTo(GetSeparator());
                }
            }
            void Save(IOutputStream* output) const {
                if (output) {
                    output->Write(Value);
                    output->Write(GetSeparator());
                }
            }
            const TString& Get() const {
                return Value;
            }
            void Set(TStringBuf value) {
                Value = value;
            }

        private:
            TString Value;
        };

        template<size_t N>
        class TFixedString {
        public:
            public:
                inline TFixedString() = default;
                inline TFixedString(TString&& value)
                    : Value(std::move(value))
                {
                }
                inline TFixedString(const TString& value)
                    : Value(value)
                {
                }

                size_t CalcSize() const {
                    return N;
                }
                void Load(IInputStream* stream) {
                    size_t count = 0;
                    char c;
                    while (stream && count < N) {
                        if (!stream->ReadChar(c)) {
                            break;
                        }

                        Value.push_back(c);
                        count++;
                    }
                }
                void Save(IOutputStream* stream) const {
                    if (stream) {
                        size_t writeSize = std::min(Value.Size(), CalcSize());
                        stream->Write(TStringBuf(Value.begin(), Value.begin() + writeSize));
                    }
                }

                TStringBuf Get() const {
                    return Value;
                }

                void Set(TStringBuf value) {
                    Value = value;
                }

        private:
            TString Value;
        };

        class TZeroTerminatedString {
        private:
            TString Value;

        public:
            inline TZeroTerminatedString() = default;
            template <class T>
            inline TZeroTerminatedString(T&& value)
                : Value(std::forward<T>(value))
            {
            }

            size_t CalcSize() const;
            void Load(IInputStream* s);
            void Save(IOutputStream* s) const;

            TStringBuf Get() const;
            void Set(TStringBuf value);
        };

        template <size_t N>
        class TFixedSizeZeroTerminatedString {
        public:
            DEFINE_FIELDS(Data);

            TStringBuf View() const;
            TString Get() const;
            TConstArrayRef<char> Raw() const;
            void Set(TStringBuf value);

        private:
            std::array<char, N> Data = MakeFilledArray(0);
        };

        template <size_t N>
        TStringBuf NDrive::NProtocol::TFixedSizeZeroTerminatedString<N>::View() const {
            const auto data = Data.data();
            const auto size = std::min(std::strlen(data), Data.size());
            return { data, size };
        }

        template <size_t N>
        TString NDrive::NProtocol::TFixedSizeZeroTerminatedString<N>::Get() const {
            return ToString(View());
        }

        template <size_t N>
        TConstArrayRef<char> NDrive::NProtocol::TFixedSizeZeroTerminatedString<N>::Raw() const {
            return MakeArrayRef(Data);
        }

        template <size_t N>
        void NDrive::NProtocol::TFixedSizeZeroTerminatedString<N>::Set(TStringBuf value) {
            Y_ENSURE(value.size() < Data.size());
            Data.fill(0);
            MemCopy(Data.data(), value.data(), value.size());
        }
    }
}
