#include "vega.h"
#include "crc.h"
#include "mio.h"
#include "settings.h"

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/buffer.h>
#include <util/generic/map.h>
#include <util/generic/serialized_enum.h>
#include <util/stream/buffer.h>
#include <util/string/builder.h>
#include <util/string/hex.h>
#include <util/string/join.h>

#include <cmath>

namespace {
    size_t dstrlen(const char* begin, const char* end) {
        if (!begin) {
            return 0;
        }
        Y_ASSERT(begin <= end);
        return strnlen(begin, end > begin ? end - begin : 0);
    }

    size_t rstrlen(const char* begin, const char* end) {
        if (!begin) {
            return 0;
        }
        Y_ASSERT(begin <= end);
        for (auto i = end; i >= begin; --i) {
            auto p = i - 1;
            if (*p != '\0') {
                return i - begin;
            }
        }
        return 0;
    }

    template <class T>
    TString GetDebugString(const T& object) {
        TBuffer buffer;
        TBufferOutput bo(buffer);
        Save(&bo, object);
        return HexEncode(buffer.Data(), buffer.Size());
    }

    template <class T, class D>
    T GetIntegerFromByteArray(TConstArrayRef<D> data) {
        static_assert(1 == sizeof(D), "not a byte type");
        Y_ENSURE(data.size() == sizeof(T), "size mismatch: " << data.size() << " " << sizeof(T));
        return *reinterpret_cast<const T*>(&data[0]);
    }

    template <class V, class T>
    V GetByteArrayFromType(T data) {
        using B = typename V::value_type;
        static_assert(1 == sizeof(B), "not a byte type");

        V result;
        const auto p = reinterpret_cast<const B*>(&data);
        for (size_t i = 0; i < sizeof(data); ++i) {
            result.push_back(*(p + i));
        }
        return result;
    }

    template <class D>
    TStringBuf GetStringFromByteArray(TConstArrayRef<D> data) {
        return TStringBuf(
            data.data(),
            rstrlen(data.begin(), data.end())
        );
    }

    template <class D>
    TStringBuf GetDStringFromByteArray(TConstArrayRef<D> data) {
        return TStringBuf(
            data.data(),
            dstrlen(data.begin(), data.end())
        );
    }
}

namespace NDrive {
    namespace NVega {
        class TSensorTraits {
        public:
            struct TSensor {
            public:
                TSensorId Id;
                EValueType Type;
                TValueTraits Traits;
                TString Name;

            public:
                TSensor(TSensorId id, TString name, EValueType type = EValueType::Unknown, TValueTraits traits = EValueTrait::vtNone)
                    : Id(id)
                    , Type(type)
                    , Traits(traits)
                    , Name(name)
                {
                }

                bool operator< (const TSensor& other) const {
                    return Id < other.Id;
                }
            };

        public:
            TSensorTraits() {
                std::stable_sort(Sensors.begin(), Sensors.end());
                for (auto&& sensor : Sensors) {
                    NameView[sensor.Name] = &sensor;
                }
            }

        public:
            static TSensorId GetId(TStringBuf name) {
                auto sensor = Singleton<TSensorTraits>()->GetSensor(name);
                if (sensor) {
                    return sensor->Id;
                } else {
                    return 0;
                }
            }
            static TStringBuf GetName(TSensorId id) {
                auto sensor = Singleton<TSensorTraits>()->GetSensorImpl(id);
                if (sensor) {
                    return sensor->Name;
                } else {
                    return {};
                }
            }
            static TVector<TStringBuf> GetNames() {
                return Singleton<TSensorTraits>()->GetSensorNames();
            }
            static TVector<TSensorId> GetIds() {
                return Singleton<TSensorTraits>()->GetSensorIds();
            }
            static std::tuple<EValueType, TValueTraits> GetValueInfo(TSensorId id) {
                auto sensor = Singleton<TSensorTraits>()->GetSensor(id);
                if (sensor) {
                    return { sensor->Type, sensor->Traits };
                } else {
                    return { EValueType::Unknown, EValueTrait::vtNone };
                }
            }
            static void CheckValueType(TSensorId sensorId, EValueType type) {
                auto t = std::get<NDrive::NVega::EValueType>(GetValueInfo(sensorId));
                if (t != type && t != EValueType::Unknown) {
                    ythrow yexception() << "type mismatch for sensor "
                                        << sensorId.ToJson().GetStringRobust() << ": " << t << " " << type;
                }
            }

        private:
            const TSensor* GetSensor(TSensorId id) const {
                auto p = GetSensorImpl(id);
                if (p) {
                    return p;
                }
                if (id.SubId) {
                    return GetSensorImpl(id.Id);
                }
                return nullptr;
            }
            const TSensor* GetSensorImpl(TSensorId id) const {
                auto p = std::lower_bound(Sensors.begin(), Sensors.end(), id, [] (const TSensor& sensor, TSensorId i) {
                    return sensor.Id < i;
                });
                if (p != Sensors.end() && p->Id == id) {
                    return &*p;
                } else {
                    return nullptr;
                }
            }
            const TSensor* GetSensor(TStringBuf name) const {
                auto p = NameView.find(name);
                if (p != NameView.end()) {
                    return p->second;
                } else {
                    return nullptr;
                }
            }
            TVector<TStringBuf> GetSensorNames() const {
                return MakeVector<TStringBuf>(NContainer::Keys(NameView));
            }
            TVector<TSensorId> GetSensorIds() const {
                TVector<TSensorId> sensorIds(Sensors.size());
                for (size_t i = 0; i < Sensors.size(); ++i) {
                    sensorIds[i] = Sensors[i].Id;
                }
                return sensorIds;
            }

        private:
            TVector<TSensor> Sensors = {  // no `-` sign in sensor name
#define VEGA_SENSOR_A(X)    X, #X
#define VEGA_SENSOR_N(X, Y) X, Y
                // xxx
                { VEGA_SENSOR_N(VEGA_MCU_FIRMWARE_VERSION, "mcu_firmware"),               EValueType::ASCII, EValueTrait::vtLeftStrlen },
                { VEGA_SENSOR_N(VegaMcuFirmwareVersionRevision, "mcu_firmware_revision"), EValueType::UI32 },
                { VEGA_SENSOR_N(VEGA_GSM_FIRMWARE_VERSION, "gsm_firmware"),               EValueType::ASCII, EValueTrait::vtLeftStrlen },
                { VEGA_SENSOR_N(VEGA_GPS_FIRMWARE_VERSION, "gps_firmware"),               EValueType::ASCII, EValueTrait::vtLeftStrlen },
                { VEGA_SENSOR_A(VEGA_DEVICE_SERIAL),     EValueType::UI32 },
                { VEGA_SENSOR_A(VEGA_SIM_ICCID),         EValueType::ASCII },
                { VEGA_SENSOR_A(VEGA_SIM2_ICCID),        EValueType::ASCII },
                { VEGA_SENSOR_N(VEGA_OVERALL_WEAR_CONDITION, "overall_wear_condition"),   EValueType::Float32 },
                { VEGA_SENSOR_N(VEGA_FLASH_WEAR_CONDITION,   "flash_erase_time_ms"),      EValueType::UI32 },
                { VEGA_SENSOR_N(VEGA_CAN_1_LAST_ACTIVITY,    "can1_activity_ts"),         EValueType::UI32 },
                { VEGA_SENSOR_N(VEGA_CAN_2_LAST_ACTIVITY,    "can2_activity_ts"),         EValueType::UI32 },
                { VEGA_SENSOR_N(VEGA_CAN_3_LAST_ACTIVITY,    "can3_activity_ts"),         EValueType::UI32 },

                { VEGA_SENSOR_A(VEGA_INPUT_BUFFER_SIZE), EValueType::UI16 },
                { VEGA_SENSOR_A(VEGA_UPTIME),            EValueType::UI32 },
                { VEGA_SENSOR_A(VEGA_UTC),               EValueType::UI32 },
                { VEGA_SENSOR_A(VEGA_OPERATION_MODE) },
                { VEGA_SENSOR_A(VEGA_BB_MESSAGE_COUNT_1) },
                { VEGA_SENSOR_A(VEGA_BB_MESSAGE_COUNT_2) },
                { VEGA_SENSOR_A(VEGA_BB_MESSAGE_COUNT_3) },
                { VEGA_SENSOR_A(VEGA_BB_MESSAGE_COUNT_4) },
                { VEGA_SENSOR_A(VEGA_TCP_STATE1) },
                { VEGA_SENSOR_A(VEGA_TCP_STATE2) },
                { VEGA_SENSOR_A(VEGA_TCP_STATE3) },
                { VEGA_SENSOR_A(VEGA_TCP_STATE4) },
                { VEGA_SENSOR_A(VEGA_LAT),               EValueType::Float32, EValueTrait::vtSkipZero },
                { VEGA_SENSOR_A(VEGA_LON),               EValueType::Float32, EValueTrait::vtSkipZero },
                { VEGA_SENSOR_N(VEGA_SPEED, "speed"),    EValueType::Float32 },
                { VEGA_SENSOR_A(VEGA_DIR),               EValueType::Float32, EValueTrait::vtSkipZero },
                { VEGA_SENSOR_A(VEGA_ALT),               EValueType::Float32 },
                { VEGA_SENSOR_A(VEGA_HDOP),              EValueType::Float32 },
                { VEGA_SENSOR_A(VEGA_PDOP),              EValueType::Float32 },
                { VEGA_SENSOR_A(VEGA_VDOP),              EValueType::Float32 },
                //{ VEGA_SENSOR_A() },
                { VEGA_SENSOR_N(VEGA_SAT_USED,           "gps_used") },
                { VEGA_SENSOR_A(VEGA_GPS_INVIEW) },
                { VEGA_SENSOR_A(VEGA_GLONASS_INVIEW) },
                { VEGA_SENSOR_A(VEGA_GPS_ODOMETER) },
                { VEGA_SENSOR_A(VEGA_GPS_MOVE_SENSOR),   EValueType::UI8 },
                { VEGA_SENSOR_A(VEGA_TRIP_COUNTER),      EValueType::UI32 },
                { VEGA_SENSOR_N(VEGA_GPS_JAMMED,         "gps_jammed") },
                { VEGA_SENSOR_N(VEGA_GPS_SPOOF_SENSOR,   "gps_spoofed") },
                { VEGA_SENSOR_N(VEGA_GPS_IS_ACTIVE,      "gps_active") },

                { VEGA_SENSOR_A(VEGA_NRF_DESIRED_RELAY_STATE), EValueType::UI8 },
                { VEGA_SENSOR_A(VEGA_NRF_ACTUAL_RELAY_STATE), EValueType::UI8 },
                { VEGA_SENSOR_N(VEGA_NRF_VISIBLE_MARKS_BF, "visible_marks") },
                { VEGA_SENSOR_N(VEGA_NRF_BATTLOW_MARKS_BF, "low_battery_marks") },
                { VEGA_SENSOR_N(VEGA_NRF_MARK_ID_1,        "mark_1"),                     EValueType::Unknown, EValueTrait::vtSkipZero },

                // 1xxx
                { VEGA_SENSOR_N(DigitalInput<1>(),          "hood_locked_din1"),             EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalInput<2>(),          "low_wiper_fluid_din2"),         EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalInput<3>(),          "din3"),                         EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalInput<4>(),          "din4"),                         EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalInput<5>(),          "din5"),                         EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<3>(),         "horn_dout3"),                   EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<4>(),         "moveblock_dout4"),              EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<5>(),         "dout5"),                        EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<6>(),         "dout6"),                        EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<7>(),         "dout7"),                        EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<8>(),         "dout8"),                        EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<9>(),         "dout9"),                        EValueType::UI8 },
                { VEGA_SENSOR_N(DigitalOutput<10>(),        "radioblock_dout10"),            EValueType::UI8 },
                { VEGA_SENSOR_N(AuxFuelLevel<1>(),          "aux_fuel_level_1"),             EValueType::Float32 },
                { VEGA_SENSOR_N(AuxFuelLevel<2>(),          "aux_fuel_level_2"),             EValueType::Float32 },
                { VEGA_SENSOR_N(AuxFuelLevel<2>(),          "aux_fuel_level_2"),             EValueType::Float32 },
                { VEGA_SENSOR_N(AuxTemperature<1>(),        "aux_temperature_1"),            EValueType::Float32 },
                { VEGA_SENSOR_A(VEGA_TCM_DTC_LIST_SENSOR) },
                { VEGA_SENSOR_N(VEGA_CLS_STATE_SENSOR,      "cls_state"),                    EValueType::UI8 },
                { VEGA_SENSOR_N(VEGA_WINDOWS_POSITION_SENSOR, "windows_state") },
                { VEGA_SENSOR_N(VEGA_VIN_SENSOR,            "vin") },
                { VEGA_SENSOR_N(VEGA_ECM_DTC_LIST_SENSOR,   "dtc_list"),                     EValueType::ASCII },
                { VEGA_SENSOR_N(VEGA_CAN_EMERG_LIGHT_MODE,  "em_light"),                     EValueType::UI8 },
                { VEGA_SENSOR_N(VEGA_CAN_WASHER_LIQUID,     "washer_liquid") },
                { VEGA_SENSOR_N(VEGA_LAUNCH_SENSOR,         "launch") },
                { VEGA_SENSOR_N(VEGA_DRIFT_SENSOR,          "drift") },
                { VEGA_SENSOR_N(VEGA_TAMPER_1,              "tamper_1") },
                { VEGA_SENSOR_N(VEGA_TAMPER_1 + 1,          "tamper_2") },
                { VEGA_SENSOR_N(VEGA_IGNITION,              "ignition") },
                { VEGA_SENSOR_N(VEGA_GSENSOR_AXIS_X,        "accelerometer_X") },
                { VEGA_SENSOR_N(VEGA_GSENSOR_AXIS_Y,        "accelerometer_Y") },
                { VEGA_SENSOR_N(VEGA_GSENSOR_AXIS_Z,        "accelerometer_Z") },
                { VEGA_SENSOR_N(VEGA_EVACUATION,            "evacuation") },
                { VEGA_SENSOR_N(VEGA_GSENSOR_MOVE_SENSOR,   "accelerometer_move") },
                { VEGA_SENSOR_N(VEGA_INT_TEMP,              "internal_temperature") },
                { VEGA_SENSOR_N(VEGA_POWER_VOLTAGE,         "ext_voltage") },
                { VEGA_SENSOR_N(VEGA_ACC_VOLTAGE,           "acc_voltage") },
                { VEGA_SENSOR_N(VEGA_ACC_CHARGE,            "acc_charge") },

                // 2xxx
                { VEGA_SENSOR_N(VEGA_MCC,                "mcc") },
                { VEGA_SENSOR_N(VEGA_MNC,                "mnc") },
                { VEGA_SENSOR_N(VEGA_LAC,                "lac") },
                { VEGA_SENSOR_N(VEGA_CELLID,             "cellid") },
                { VEGA_SENSOR_N(VEGA_GSM_SIGNAL_LEVEL,   "gsm_signal_level") },
                { VEGA_SENSOR_N(VEGA_GSM_JAMMED,         "gsm_jammed") },
                { VEGA_SENSOR_N(VEGA_TX_DATA_SERV1,      "tx_server_1") },
                { VEGA_SENSOR_N(VEGA_TX_DATA_SERV2,      "tx_server_2") },
                { VEGA_SENSOR_N(VEGA_TX_DATA_SERV3,      "tx_server_3") },
                { VEGA_SENSOR_N(VEGA_TX_DATA_SERV4,      "tx_server_4") },
                { VEGA_SENSOR_N(VEGA_RX_DATA_SERV1,      "rx_server_1") },
                { VEGA_SENSOR_N(VEGA_RX_DATA_SERV2,      "rx_server_2") },
                { VEGA_SENSOR_N(VEGA_RX_DATA_SERV3,      "rx_server_3") },
                { VEGA_SENSOR_N(VEGA_RX_DATA_SERV4,      "rx_server_4") },
                { VEGA_SENSOR_N(VEGA_GSM_ACT,            "gsm_technology") },
                { VEGA_SENSOR_N(VEGA_EXT_SERVING_CELL_INF, "gsm_base_stations"),          EValueType::ASCII },
                { VEGA_SENSOR_N(VEGA_GSM_SCAN_MODE,      "gsm_scan_mode"),                EValueType::UI8 },
                { VEGA_SENSOR_N(VEGA_GSM_USED_SIM,       "gsm_used_sim") },

                { VEGA_SENSOR_N(CAN_ODOMETER_KM,         "mileage"),                      EValueType::Float32, EValueTrait::vtSkipZero | EValueTrait::vtFilterOdometer },
                { VEGA_SENSOR_N(CAN_FUEL_LEVEL_P,        "fuel_level"),                   EValueType::UI8,     EValueTrait::vtSkipZero | EValueTrait::vtPercentValue | vtSkipInvalidPercent },
                { VEGA_SENSOR_N(CAN_FUEL_DISTANCE_KM,    "fuel_distance"),                EValueType::Unknown, EValueTrait::vtSkipZero },
                { VEGA_SENSOR_N(CAN_ENGINE_RPM,          "engine_rpm") },
                { VEGA_SENSOR_N(CAN_ENGINE_TEMP,         "engine_temperature") },
                { VEGA_SENSOR_N(CAN_SPEED,               "speed_can") },
                { VEGA_SENSOR_N(CAN_AXLE_1,              "axle_1") },
                { VEGA_SENSOR_N(CAN_AXLE_2,              "axle_2") },
                { VEGA_SENSOR_N(CAN_AXLE_3,              "axle_3") },
                { VEGA_SENSOR_N(CAN_AXLE_4,              "axle_4") },
                { VEGA_SENSOR_N(CAN_AXLE_5,              "axle_5") },
                { VEGA_SENSOR_N(CAN_FUEL_FLOW,           "fuel_flow") },
                { VEGA_SENSOR_N(CAN_BRAKE,               "brake_pedal") },
                { VEGA_SENSOR_N(CAN_ACCELERATOR,         "accelerator_pedal") },
                { VEGA_SENSOR_N(CAN_STEERING_ACCELERATION, "steering_wheel_acceleration"),      EValueType::UI16},
                { VEGA_SENSOR_N(CAN_ODO_AFTER_SERVICE,   "odometer_after_service") },
                { VEGA_SENSOR_N(CAN_STEERING_WHEEL,      "steering_wheel") },
                { VEGA_SENSOR_N(CAN_CHECK_OIL,           "oil_check_on") },
                { VEGA_SENSOR_N(CAN_CHECK_COOLANT,       "check_coolant") },
                { VEGA_SENSOR_N(CAN_CHECK_HAND_BREAK,    "hand_brake_check_on") },
                { VEGA_SENSOR_N(CAN_BATTERY,             "battery_check_on") },
                { VEGA_SENSOR_N(CAN_AIRBAG,              "airbag_check_on") },
                { VEGA_SENSOR_N(CAN_CHECK_ENGINE,        "engine_check_on") },
                { VEGA_SENSOR_N(CAN_FAULT_LIGHTING,      "fault_lighting") },
                { VEGA_SENSOR_N(CAN_INFLATION_PRESSURE,  "inflation_pressure_check_on") },
                { VEGA_SENSOR_N(CAN_CHECK_BRAKE_PADS,    "brake_check_on") },
                { VEGA_SENSOR_N(CAN_FUEL_LEVEL,          "fuel_level_check_on") },
                { VEGA_SENSOR_N(SERVICE_MAINT,           "service_maintenance_check_on") },
                { VEGA_SENSOR_N(CAN_MARKER_LIGHTS,       "marker_lights") },
                { VEGA_SENSOR_N(CAN_DIPPED_BEAM,         "dipped_beam_on") },
                { VEGA_SENSOR_N(CAN_HIGH_BEAM,           "high_beam_on") },
                { VEGA_SENSOR_N(CAN_DRIVER_SAFE_BELT,    "driver_belt_fastened") },
                { VEGA_SENSOR_N(CAN_IGNITION,            "ignition_on") },
                { VEGA_SENSOR_N(CAN_PARKING,             "transmission_selector_park") },
                { VEGA_SENSOR_N(CAN_HAND_BREAK,          "hand_brake_on") },
                { VEGA_SENSOR_N(CAN_PEDAL_BREAK,         "pedal_brake_on") },
                { VEGA_SENSOR_N(CAN_ENGINE_IS_ON,        "engine_on"),                    EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_REVERSE_GEAR,        "transmission_selector_reverse") },
                { VEGA_SENSOR_N(CAN_DRIVER_DOOR,         "front_left_door_open"),         EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_PASS_DOOR,           "front_right_door_open"),        EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_L_REAR_DOOR,         "rear_left_door_open"),          EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_R_REAR_DOOR,         "rear_right_door_open"),         EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_HOOD,                "hood_open"),                    EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_TRUNK,               "trunk_open"),                   EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_IN_SLEEP,            "can_in_sleep") },
                { VEGA_SENSOR_N(CAN_TRANSMISSION_SELECTOR, "can_tr_selector"),            EValueType::UI8 },
                { VEGA_SENSOR_N(CAN_CUSTOM_FUEL_VOLUME,  "fuel_volume")},
                { VEGA_SENSOR_N(CAN_CLS_STATE,           "can_cls_state"),                EValueType::UI8 },
                { VEGA_SENSOR_N(BLE_FUEL_LEVEL,          "ble_fuel_level"),               EValueType::UI16 },

                // 3xxx
                { VEGA_SENSOR_A(VEGA_SETTING_SERVER_ADDR),    EValueType::Binary },
                { VEGA_SENSOR_A(VEGA_SETTING_APN),            EValueType::Binary },
                { VEGA_SENSOR_A(VEGA_SETTING_TRANSLATE_SENSORS), EValueType::Binary },
                { VEGA_SENSOR_A(VEGA_SETTING_USE_SERVER_PIN), EValueType::UI8 },
                { VEGA_SENSOR_A(VEGA_SETTING_SERVER_PIN),     EValueType::Binary },
                { VEGA_SENSOR_A(VEGA_SETTING_PIN_CODE),       EValueType::Binary },
                { VEGA_SENSOR_A(VEGA_SETTING_USE_PIN_CODE),   EValueType::UI8 },
                { VEGA_SENSOR_A(VEGA_SETTING_GPS_MOVE),       EValueType::UI16 },

                { VEGA_SENSOR_A(VEGA_SETTING_CAN_CUSTOM_SENSORS), EValueType::Binary},

                { VEGA_SENSOR_A(VEGA_SETTINGS_SCRIPT_SCRIPTS), EValueType::Binary},
                { VEGA_SENSOR_A(VEGA_SETTINGS_SCRIPT_COMMANDS), EValueType::Binary},
                { VEGA_SENSOR_A(VEGA_SETTINGS_SCRIPT_CHECKS), EValueType::Binary},
                { VEGA_SENSOR_A(VEGA_SETTINGS_SCRIPT_TRIGGERS), EValueType::Binary},
                { VEGA_SENSOR_A(VEGA_SETTINGS_SCRIPT_NOTIFIES), EValueType::Binary},

                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO1), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO2), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO3), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO4), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO5), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO6), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO7), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO8), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO9), EValueType::Binary },
                { VEGA_SENSOR_A(BLE_EXT_BOARD_BEACONS_INFO10), EValueType::Binary },

                // 5xxx
                { VEGA_SENSOR_N(VEGA_SETTING_CAN_CUSTOM_SENSORS_GROUP_NAME, "can_custom_group_name"), EValueType::Binary },
                { VEGA_SENSOR_N(VEGA_SETTING_CAN_CUSTOM_SENSORS,            "can_custom_sensors"),    EValueType::Binary },

                { VEGA_SENSOR_N(BLE_EXT_BOARD_COMMUNICATION_STATE,  "ble_communication_state") },
                { VEGA_SENSOR_N(BLE_EXT_BOARD_BOOT_VERS,            "ble_booloader_version") },
                { VEGA_SENSOR_N(BLE_EXT_BOARD_APP_VERS,             "ble_firmware_version") },
                { VEGA_SENSOR_N(BLE_EXT_BOARD_MAC,                  "ble_mac"),                 EValueType::Binary, EValueTrait::vtSkipZero },
                { VEGA_SENSOR_N(NDrive::NVega::BleSessionKey.Id,    "ble_session_key"),         EValueType::Binary },

                { VEGA_SENSOR_N(VEGA_SETTING_BLE_EXT_BOARD,         "ble_setting") },
                { VEGA_SENSOR_N(TSvrRawState::GetId(),              "svr_raw_state"),           EValueType::Binary },
                { VEGA_SENSOR_N(TSvrRawState::PollStateId,          "svr_poll_state"),          EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::SerialNumberId,       "svr_serial_number"),       EValueType::Unknown },
                { VEGA_SENSOR_N(TSvrRawState::EngineRpmId,          "svr_engine_rpm"),          EValueType::UI16 },
                { VEGA_SENSOR_N(TSvrRawState::VoltageId,            "svr_voltage"),             EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::MapId,                "svr_map"),                 EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::SypId,                "svr_syp"),                 EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::ReductorTemperatureId,"svr_reductor_temperature"),EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::GasTemperatureId,     "svr_gas_temperature"),     EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::PcbTemperatureId,     "svr_pcb_temperature"),     EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::PetrolLoadId,         "svr_pertol_load"),         EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::GasLoadId,            "svr_gas_load"),            EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::EngineLoadId,         "svr_engine_load"),         EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::EcuStateId,           "svr_ecu_state"),           EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::TankLevelAverageId,   "svr_tank_level_average"),  EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::DtcCountId,           "svr_dtc_count"),           EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::EcuResetCounterId,    "svr_ecu_reset_counter"),   EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::EmergencyStartCounterId,"svr_emergency_start_counter"),EValueType::UI8 },
                { VEGA_SENSOR_N(TSvrRawState::MaxPcbTemperatureId,  "svr_max_pcb_temperature"), EValueType::Float32 },
                { VEGA_SENSOR_N(TSvrRawState::WorkingOnPetrolId,    "svr_working_on_petrol"),   EValueType::UI32 },
                { VEGA_SENSOR_N(TSvrRawState::WorkingOnGasId,       "svr_working_on_gas"),      EValueType::UI32 },
                { VEGA_SENSOR_N(TSvrRawState::DistanceLeftId,       "svr_distance_left"),       EValueType::UI16 },
                { VEGA_SENSOR_N(TSvrDtcState::GetId(),              "svr_dtc_state"),           EValueType::Binary },
#undef VEGA_SENSOR_N
#undef VEGA_SENSOR_A
            };
            TMap<TStringBuf, const TSensor*> NameView;
        };

        THolder<NDrive::NProtocol::IPayload> CreatePayload(NDrive::NVega::EMessageType type, size_t packetSize = 0) {
            using namespace NDrive::NProtocol;

            switch (type) {
            case INCORRECT:
                return nullptr;
            case COMMAND_REQUEST:
                return IPayload::Create<TCommandRequest>();
            case COMMAND_RESPONSE:
                return IPayload::Create<TCommandResponse>();
            case COMMAND_REQUEST_SIGNED:
                return IPayload::Create<TCommandRequestSigned>();
            case COMMAND_RESPONSE_SIGNED:
                return IPayload::Create<TCommandResponseSigned>();
            case FILE_CHUNK_REQUEST:
                return IPayload::Create<TFileChunkRequest>();
            case FILE_CHUNK_RESPONSE:
                return IPayload::Create<TFileChunkResponse>();
            case AUTH_REQUEST:
                return IPayload::Create<TAuthorizationRequest>();
            case AUTH_RESPONSE:
                return IPayload::Create<TAuthorizationResponse>();
            case GET_FILE_REQUEST:
                return IPayload::Create<TGetFileRequest>();
            case GET_FILE_RESPONSE:
                return IPayload::Create<TGetFileResponse>();
            case PING_REQUEST:
                return IPayload::Create<TPingRequest>();
            case PING_RESPONSE:
                return IPayload::Create<TPingResponse>();
            case LIST_REQUEST:
                return IPayload::Create<TListRequest>();
            case LIST_RESPONSE:
                return IPayload::Create<TListResponse>();
            case CONNECT_REQUEST:
                return IPayload::Create<TConnectRequest>();
            case CONNECT_RESPONSE:
                return IPayload::Create<TConnectResponse>();
            case DISCONNECTED:
                return IPayload::Create<TDisconnectedNotification>();
            case BLACKBOX_RECORDS:
                return IPayload::Create<TBlackboxRecords>();
            case BLACKBOX_RECORDS_ACK:
                return IPayload::Create<TBlackboxRecordsAck>();
            case FILE_STREAM_REQUEST:
                return IPayload::Create<TFileStreamRequest>();
            case FILE_STREAM_RESPONSE:
                return IPayload::Create<TFileStreamResponse>();
            case INTERFACE_REQUEST:
                return IPayload::Create<TInterfaceRequest>();
            case INTERFACE_RESPONSE:
                return IPayload::Create<TInterfaceResponse>();
            case CAN_REQUEST:
                return IPayload::Create<TCanRequest>();
            case CAN_RESPONSE:
                return IPayload::Create<TCanResponse>();
            case FAST_DATA_RECORDS:
                return IPayload::Create<TFastDataRecords>(packetSize);
            }
        }
    }
}

NDrive::TSensorId NDrive::NVega::GetSensorId(TStringBuf name) {
    return NDrive::NVega::TSensorTraits::GetId(name);
}

TStringBuf NDrive::NVega::GetSensorName(TSensorId sensorId) {
    return NDrive::NVega::TSensorTraits::GetName(sensorId);
}

TVector<TStringBuf> NDrive::NVega::GetSensorNames() {
    return NDrive::NVega::TSensorTraits::GetNames();
}

TVector<NDrive::TSensorId> NDrive::NVega::GetSensorIds() {
    return NDrive::NVega::TSensorTraits::GetIds();
}

NDrive::NVega::EValueType NDrive::NVega::GetValueType(TSensorId sensorId) {
    return std::get<NDrive::NVega::EValueType>(NDrive::NVega::TSensorTraits::GetValueInfo(sensorId));
}

NDrive::NVega::TValueTraits NDrive::NVega::GetValueTraits(TSensorId sensorId) {
    return std::get<NDrive::NVega::TValueTraits>(NDrive::NVega::TSensorTraits::GetValueInfo(sensorId));
}

NDrive::NVega::TMessage::TMessage(EMessageType type /*= INCORRECT*/) {
    Payload = CreatePayload(type);
}

NDrive::NVega::TMessage::TMessage(THolder<NProtocol::IPayload>&& payload) {
    Payload = std::move(payload);
}

void NDrive::NVega::TMessage::Load(IInputStream& input) {
    ui32 crc32 = 0;
    ui16 length = 0;
    ui8 type = 0;

    constexpr size_t headerSize = sizeof(length) + sizeof(type);
    constexpr size_t crc32Size = sizeof(crc32);

    TBuffer b(headerSize);
    TBufferInput buffer(b);

    b.Advance(input.Load(b.End(), headerSize));
    if (b.Empty()) {
        return;
    }

    try {
        SaveLoad(&buffer, length);
        SaveLoad(&buffer, type);

        auto messageType = static_cast<EMessageType>(type);
        Y_ENSURE(GetEnumNames<EMessageType>().contains(messageType), "bad type " << static_cast<ui32>(type));
        Payload = CreatePayload(messageType, length);
        Y_ENSURE(Payload);

        b.Reserve(b.Capacity() + length + crc32Size);
        b.Advance(input.Load(b.End(), length + crc32Size));

        if (b.Size() < headerSize + length + crc32Size) {
            throw TIncompleteDataException(headerSize + length + crc32Size, b.Size());
        }

        Payload->Load(buffer);
        SaveLoad(&buffer, crc32);
    } catch (const TIncompleteDataException&) {
        throw;
    } catch (const std::exception& e) {
        auto messageType = static_cast<ui32>(type);
        auto len = static_cast<ui32>(length);
        ythrow yexception() << "cannot deserialize "
            << messageType << " "
            << len << " "
            << HexEncode(b.Data(), b.Size()) << ": "
            << FormatExc(e);
    }

    ui32 crc32check = NDrive::VegaCrc32(
        buffer.Buffer().Begin(),
        std::min(buffer.Buffer().Begin() + headerSize + length, buffer.Buffer().End())
    );
    Y_ENSURE(crc32 == crc32check, DebugString() << " crc32 mismatch: " << crc32 << " " << crc32check);

    Payload->PostLoad(input);
}

void NDrive::NVega::TMessage::Save(IOutputStream& output) const {
    Y_ENSURE(Payload);
    const size_t payloadSize = Payload->GetSize();
    const ui16 length = static_cast<ui16>(payloadSize);
    const ui8 type = static_cast<ui8>(GetMessageType());
    Y_ENSURE(payloadSize == length);
    constexpr size_t headerSize = sizeof(length) + sizeof(type);
    constexpr size_t crc32Size = sizeof(ui32);

    TBufferOutput buffer(headerSize + payloadSize + crc32Size);
    SaveLoad(&buffer, length);
    SaveLoad(&buffer, type);
    Payload->Save(buffer);
    const ui32 crc32 = NDrive::VegaCrc32(buffer.Buffer().Begin(), buffer.Buffer().End());
    SaveLoad(&buffer, crc32);

    output.Write(buffer.Buffer().Data(), buffer.Buffer().Size());
    Payload->PostSave(output);
}

template <>
void NDrive::NVega::TValue::Set(ui8 value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::UI8);
    Value = GetByteArrayFromType<decltype(Value)>(value);
}

template <>
void NDrive::NVega::TValue::Set(ui16 value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::UI16);
    Value = GetByteArrayFromType<decltype(Value)>(value);
}

template <>
void NDrive::NVega::TValue::Set(ui32 value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::UI32);
    Value = GetByteArrayFromType<decltype(Value)>(value);
}

template <>
void NDrive::NVega::TValue::Set(ui64 value, ui16 id) {
    auto type = GetValueType(id);
    switch (type) {
    case NDrive::NVega::EValueType::UI8: {
        auto v = static_cast<ui8>(value);
        Y_ENSURE(v == value, TypeName(v) << " overflow: " << value);
        Set(v);
        break;
    }
    case NDrive::NVega::EValueType::Unknown:
    case NDrive::NVega::EValueType::UI16: {
        auto v = static_cast<ui16>(value);
        Y_ENSURE(v == value, TypeName(v) << " overflow: " << value);
        Set(v);
        break;
    }
    case NDrive::NVega::EValueType::UI32: {
        auto v = static_cast<ui32>(value);
        Y_ENSURE(v == value, TypeName(v) << " overflow: " << value);
        Set(v);
        break;
    }
    case NDrive::NVega::EValueType::Float32:
    case NDrive::NVega::EValueType::ASCII:
    case NDrive::NVega::EValueType::Binary:
        ythrow yexception() << "type mismatch for sensor " << static_cast<ui32>(id) << ": expected " << type;
    }
}

template <>
void NDrive::NVega::TValue::Set(float value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::Float32);
    Value = GetByteArrayFromType<decltype(Value)>(value);
}

template <>
void NDrive::NVega::TValue::Set(double value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::Float32);
    Value = GetByteArrayFromType<decltype(Value)>(static_cast<float>(value));
}

template <>
void NDrive::NVega::TValue::Set(TStringBuf value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::ASCII);
    Value = { value.begin(), value.end() };
}

template <>
void NDrive::NVega::TValue::Set(TString value, ui16 id) {
    Set<TStringBuf>(value, id);
}

template <>
void NDrive::NVega::TValue::Set(TConstArrayRef<char> value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::Binary);
    Value = { value.begin(), value.end() };
}

template <>
void NDrive::NVega::TValue::Set(TBuffer value, ui16 id) {
    TSensorTraits::CheckValueType(id, EValueType::Binary);
    Value = { value.Begin(), value.End() };
}

template <>
void NDrive::NVega::TValue::Set(TNull /*value*/, ui16 /*id*/) {
    Value.clear();
}

template <>
void NDrive::NVega::TValue::Set(NDrive::TSensorValue value, ui16 id) {
    std::visit([this, id](auto&& v) {
        Set(v, id);
    }, value);
}

NDrive::TSensorRef NDrive::NVega::TValue::Get(TSensorId id /*= 0*/) const {
    return Get(Value, id);
}

NDrive::TSensorRef NDrive::NVega::TValue::Get(TConstArrayRef<char> data, TSensorId id /*= 0*/) {
    if (data.empty()) {
        return TNull();
    }
    auto&& [type, traits] = TSensorTraits::GetValueInfo(id);
    switch (type) {
    case NDrive::NVega::EValueType::UI8: {
        ui64 value = GetIntegerFromByteArray<ui8>(data);
        return value;
    }
    case NDrive::NVega::EValueType::UI16: {
        ui64 value = GetIntegerFromByteArray<ui16>(data);
        return value;
    }
    case NDrive::NVega::EValueType::UI32: {
        ui64 value = GetIntegerFromByteArray<ui32>(data);
        return value;
    }
    case NDrive::NVega::EValueType::Float32: {
        double value = GetIntegerFromByteArray<float>(data);
        return value;
    }
    case NDrive::NVega::EValueType::ASCII:
        return (traits & vtLeftStrlen) ? GetDStringFromByteArray(data) : GetStringFromByteArray(data);
    case NDrive::NVega::EValueType::Binary:
        return data;
    case NDrive::NVega::EValueType::Unknown:
        break;
    }
    switch (data.size()) {
        case 0:
            return TNull();
        case 1: {
            ui64 value = GetIntegerFromByteArray<ui8>(data);
            return value;
        }
        case 2: {
            ui64 value = GetIntegerFromByteArray<ui16>(data);
            return value;
        }
        case 4: {
            double value = GetIntegerFromByteArray<float>(data);
            if (std::abs(value) > 1e-10 || std::abs(value) == 0) {
                return value;
            }
        }
        {
            ui64 value = GetIntegerFromByteArray<ui32>(data);
            return value;
        }
        default: {
            return GetStringFromByteArray(data);
        }
    }
}

TString NDrive::NVega::TArgument::DebugString() const {
    return HexEncode(Value.data(), Value.size());
}

TString NDrive::NVega::TCommandRequest::DebugString() const {
    return "CommandRequest " + ToString(Id) + " " + ToString(static_cast<ui32>(Code)) + " " + Argument.DebugString();
}

void NDrive::NVega::TCommandRequest::TFastDataConfig::SetDiscretization(TDuration value) {
    Y_ENSURE(value >= TDuration::MilliSeconds(10));
    Y_ENSURE(value <= TDuration::MilliSeconds(1000));
    Discretization = value.MilliSeconds() / 10;
}

void NDrive::NVega::TCommandRequest::TFastDataConfig::SetDuration(TDuration value) {
    Y_ENSURE(value >= TDuration::Seconds(1));
    Y_ENSURE(value <= TDuration::Seconds(255));
    Duration = value.Seconds();
}

TString NDrive::NVega::TCommandRequestSigned::DebugString() const {
    return TStringBuilder() << "CommandRequestSigned " << Id << ' ' << HexEncode(Nonce.data(), Nonce.size()) << ' ' << static_cast<ui32>(Code) << ' ' << Argument.DebugString();
}

TString NDrive::NVega::TCommandResponse::DebugString() const {
    return "CommandResponse " + ToString(Id) + " " + ToString(Result) + " " + Argument.DebugString();
}

bool NDrive::NVega::TCommandResponseSigned::ValidateMac(const TKey& key) const {
    TString serialized;
    TStringOutput so(serialized);
    ::Save(&so, Nonce);
    ::Save(&so, Id);
    ::Save(&so, Result);
    ::Save(&so, Argument);

    TMac mac = NPoly1305::Calc(key, serialized);
    return mac == Mac;
}

TString NDrive::NVega::TCommandResponseSigned::DebugString() const {
    return TStringBuilder() << "CommandResponseSigned " << Id << ' ' << HexEncode(Nonce.data(), Nonce.size()) << ' ' << HexEncode(Mac.data(), Mac.size()) << ' ' << Result << ' ' << Argument.DebugString();
}

TString NDrive::NVega::TAuthorizationRequest::DebugString() const {
    std::array<ui32, 5> parts = {{ Password[0], Password[1], Password[2], Password[3], EnableBlackbox }};
    return "AuthorizationRequest " + JoinSeq(" ", parts);
}

TString NDrive::NVega::THardAuthorizationRequest::DebugString() const {
    return "HardAuthorizationRequest " + ToString<ui32>(EnableBlackbox);
}

TString NDrive::NVega::TAuthorizationResponse::DebugString() const {
    return "AuthorizationResponse " + ToString(static_cast<ui32>(Status));
}

TString NDrive::NVega::TGetFileRequest::DebugString() const {
    return "GetFileRequest " + Name.Get() + ' ' + ToString(Offset) + ' ' + ToString<ui32>(Size);
}

size_t NDrive::NVega::TGetFileResponse::TContent::CalcSize() const {
    return sizeof(ui16) + sizeof(Result) + Data.Size();
}

void NDrive::NVega::TGetFileResponse::TContent::Save(IOutputStream* s) const {
    ui16 size = Data.Size();
    Y_ENSURE(size == Data.Size());
    ::Save(s, size);
    ::Save(s, Result);
    ::SaveRange(s, Data.Begin(), Data.End());
}

void NDrive::NVega::TGetFileResponse::TContent::Load(IInputStream* s) {
    ui16 size = 0;
    ::Load(s, size);
    ::Load(s, Result);
    Data.Resize(size);
    ::LoadRange(s, Data.Begin(), Data.End());
}

TString NDrive::NVega::TGetFileResponse::DebugString() const {
    return "GetFileResponse " + Name.Get() + ' ' + ToString<ui32>(Content.Result) + ' ' + ToString(TotalSize) + ' ' + ToString(Content.Data.Size());
}

TString NDrive::NVega::TPingResponse::DebugString() const {
    std::array<ui32, 5> parts = {{ DeviceType, Reserved[0], Reserved[1], Reserved[2], Reserved[3] }};
    return "PingResponse " + IMEI.Get() + ' ' + JoinSeq(" ", parts);
}

TString NDrive::NVega::TListResponse::DebugString() const {
    return "ListResponse " + ToString(Devices.size());
}

TString NDrive::NVega::TConnectRequest::DebugString() const {
    return "ConnectRequest " + Device.IMEI.Get() + " " + Device.Name.Get();
}

TString NDrive::NVega::TConnectResponse::DebugString() const {
    return "ConnectResponse " + ToString(static_cast<EResult>(Result));
}

template <class T>
const T& NDrive::NVega::TBlackboxRecords::TParameters::GetImpl() const {
    if (!std::holds_alternative<T>(Value) && IsEmpty()) {
        return Default<T>();
    }
    return std::get<T>(Value);
}

template <class T>
T& NDrive::NVega::TBlackboxRecords::TParameters::MutableImpl() {
    if (!std::holds_alternative<T>(Value) && IsEmpty()) {
        Value.emplace<T>();
    }
    return std::get<T>(Value);
}

const NDrive::NVega::TBlackboxRecords::TParameters::TValue& NDrive::NVega::TBlackboxRecords::TParameters::Get() const {
    return GetImpl<TValue>();
}

NDrive::NVega::TBlackboxRecords::TParameters::TValue& NDrive::NVega::TBlackboxRecords::TParameters::Mutable() {
    return MutableImpl<TValue>();
}

const NDrive::NVega::TBlackboxRecords::TParameters::TTypedValue& NDrive::NVega::TBlackboxRecords::TParameters::GetTyped() const {
    return GetImpl<TTypedValue>();
}

NDrive::NVega::TBlackboxRecords::TParameters::TTypedValue& NDrive::NVega::TBlackboxRecords::TParameters::MutableTyped() {
    return MutableImpl<TTypedValue>();
}

size_t NDrive::NVega::TBlackboxRecords::TParameters::CalcSize() const {
    ui16 arrayByteSize = DataSize();
    return sizeof(arrayByteSize) + arrayByteSize;
}

size_t NDrive::NVega::TBlackboxRecords::TParameters::DataSize() const {
    ui16 result = 0;
    std::visit([&result](auto&& value) {
        for (auto&& parameter : value) {
            result += NPrivate::SizeOf(parameter);
        }
    }, Value);
    return result;
}

bool NDrive::NVega::TBlackboxRecords::TParameters::IsEmpty() const {
    return std::visit([](auto&& value) {
        return value.empty();
    }, Value);
}

bool NDrive::NVega::TBlackboxRecords::TParameters::IsTyped() const {
    return std::holds_alternative<TTypedValue>(Value);
}

constexpr ui16 typenessMask = 1 << (sizeof(ui16) * 8 - 1);
constexpr ui16 sizeMask = typenessMask - 1;
static_assert(sizeMask + typenessMask == std::numeric_limits<ui16>::max());

void NDrive::NVega::TBlackboxRecords::TParameters::Load(IInputStream* s) {
    ui16 sizeAndTypeness = 0;
    ::Load(s, sizeAndTypeness);
    ui16 size = sizeAndTypeness & sizeMask;
    bool isTyped = sizeAndTypeness & typenessMask;
    if (isTyped) {
        Value.emplace<TTypedValue>();
    } else {
        Value.emplace<TValue>();
    }
    std::visit([s, size](auto&& value) {
        for (size_t read = 0; read < size; read += NPrivate::SizeOf(value.back())) {
            ::Load(s, value.emplace_back());
        }
    }, Value);
}

void NDrive::NVega::TBlackboxRecords::TParameters::Save(IOutputStream* s) const {
    ui16 sizeAndTypeness = DataSize();
    if (IsTyped()) {
        sizeAndTypeness |= typenessMask;
    }
    ::Save(s, sizeAndTypeness);
    std::visit([s](auto&& value) {
        ::SaveRange(s, value.begin(), value.end());
    }, Value);
}

TString NDrive::NVega::TBlackboxRecords::DebugString() const {
    TString result = "BlackboxRecords " + ToString(static_cast<ui32>(Id));
    for (auto&& record : Records) {
        result.append(' ');
        result.append(ToString(record.Timestamp));
    }
    return result;
}

TString NDrive::NVega::TBlackboxRecordsAck::DebugString() const {
    return "BlackboxRecordsAck " + ToString(static_cast<ui32>(Id));
}

NDrive::NVega::TFileChunkRequest::TChunk::TChunk(ui16 id, ui16 count, TBuffer&& data)
    : Data(std::move(data))
    , Id(id)
    , Count(count)
{
    Y_ENSURE(Id <= Count);
}

size_t NDrive::NVega::TFileChunkRequest::TChunk::CalcSize() const {
    ui16 size = DataSize();
    return sizeof(size) + sizeof(Id) + sizeof(Count) + size;
}

ui16 NDrive::NVega::TFileChunkRequest::TChunk::DataSize() const {
    Y_ENSURE(Data.size() <= Max<ui16>());
    return Data.size();
}

void NDrive::NVega::TFileChunkRequest::TChunk::Load(IInputStream* s) {
    ui16 size = 0;
    ::Load(s, size);
    ::Load(s, Id);
    ::Load(s, Count);
    Y_ENSURE(Id <= Count);
    Data.Resize(size);
    ::LoadRange(s, Data.Begin(), Data.End());
}

void NDrive::NVega::TFileChunkRequest::TChunk::Save(IOutputStream* s) const {
    ui16 size = DataSize();
    ::Save(s, size);
    ::Save(s, Id);
    ::Save(s, Count);
    ::SaveRange(s, Data.Begin(), Data.End());
}

TString NDrive::NVega::TFileChunkRequest::TChunk::DebugString() const {
    return ToString(Id) + '/' + ToString(Count) + '/' + ToString(Data.size());
}

TString NDrive::NVega::TFileChunkRequest::DebugString() const {
    return "FileChunkRequest " + Name.Get() + ' ' + Chunk.DebugString();
}

TString NDrive::NVega::TFileChunkResponse::DebugString() const {
    return "FileChunkResponse " + Name.Get() + ' ' + ToString<ui32>(Result);
}

ui32 NDrive::NVega::TFileStreamRequest::TContent::CalcCrc32() const {
    return NDrive::VegaCrc32(Data.Begin(), Data.End());
}

size_t NDrive::NVega::TFileStreamRequest::TContent::CalcSize() const {
    return sizeof(ui32) + sizeof(ui32);
}

void NDrive::NVega::TFileStreamRequest::TContent::Save(IOutputStream* s) const {
    Size = static_cast<ui32>(Data.Size());
    Y_ENSURE(Size == Data.Size());
    ::Save(s, Size);

    Crc32 = CalcCrc32();
    ::Save(s, Crc32);
}

void NDrive::NVega::TFileStreamRequest::TContent::PostSave(IOutputStream* s) const {
    ::SaveRange(s, Data.Begin(), Data.End());
}

void NDrive::NVega::TFileStreamRequest::TContent::Load(IInputStream* s) {
    ::Load(s, Size);
    ::Load(s, Crc32);
}

void NDrive::NVega::TFileStreamRequest::TContent::PostLoad(IInputStream* s) {
    Data.Resize(Size);
    ::LoadRange(s, Data.Begin(), Data.End());

    ui32 crc32check = CalcCrc32();
    Y_ENSURE(Crc32 == crc32check, "file content crc32 mismatch: " << Crc32 << " " << crc32check);
}

TString NDrive::NVega::TFileStreamRequest::DebugString() const {
    return "FileStreamRequest " + Name.Get() + ' ' + ToString(Content.Get().Size());
}

TString NDrive::NVega::TFileStreamResponse::DebugString() const {
    return "FileStreamResponse " + Name.Get() + ' ' + ToString(Result);
}

TString NDrive::NVega::TInterfaceRequest::DebugString() const {
    return "InterfaceRequest " + ToString<ui32>(Interface) + ' ' + ToString(Data.size()) + ": " + GetDebugString(*this);
}

TString NDrive::NVega::TInterfaceResponse::DebugString() const {
    return "InterfaceResponse " + ToString<ui32>(Interface) + ' ' + ToString(Data.size()) + ": " + GetDebugString(*this);
}

TString NDrive::NVega::TCanRequest::DebugString() const {
    TStringBuilder result;
    result << "CanRequest " << Frames.size();
    for (auto&& frame : Frames) {
        result << ' ';
        const TVega_tx_can_frame& data = frame.Data;
        result << data.id << ':';
        result << static_cast<ui32>(data.itf_idx) << ':';
        result << static_cast<ui32>(data.id_type) << ':';
        result << static_cast<ui32>(data.rtr) << ':';
        result << static_cast<ui32>(data.reserved) << ':';
        result << static_cast<ui32>(data.dlen);
        if (data.dlen) {
            auto dataSize = std::min<size_t>(data.dlen, sizeof(data.data));
            result << ':' << HexEncode(data.data, dataSize);
        }
    }
    return result;
}

TString NDrive::NVega::TCanResponse::DebugString() const {
    TStringBuilder result;
    result << "CanResponse " << Frames.size();
    for (auto&& frame : Frames) {
        result << ' ';
        const TVega_rx_can_frame& data = frame.Data;
        result << data.time << ':';
        result << data.id << ':';
        result << static_cast<ui32>(data.itf_idx) << ':';
        result << static_cast<ui32>(data.id_type) << ':';
        result << static_cast<ui32>(data.rtr) << ':';
        result << static_cast<ui32>(data.reserved) << ':';
        result << static_cast<ui32>(data.dlen);
        if (data.dlen) {
            auto dataSize = std::min<size_t>(data.dlen, sizeof(data.data));
            result << ':' << HexEncode(data.data, dataSize);
        }
    }
    return result;
}

size_t  NDrive::NVega::TFastDataRecords::GetSize() const {
    return ::NPrivate::SizeMany(Records);
}

void NDrive::NVega::TFastDataRecords::Save(IOutputStream& stream) const {
    ::SaveMany(&stream, Records);
}

void NDrive::NVega::TFastDataRecords::Load(IInputStream& stream) {
    ui8 count = 0;
    ::Load(&stream, count);

    ui8 currentVersion = GetVersion(count);

    for (ui16 i = 0; i < count; ++i) {
        TRecord record;
        record.Version = currentVersion;

        stream.Skip(sizeof(ui8)); // skip version in data
        ::Load(&stream, record);
        Records.push_back(record);
    }
}

ui8 NDrive::NVega::TFastDataRecords::GetVersion(size_t packetsCount) {
    if (!PacketSize || !packetsCount) {
        return 0;
    }

    PacketSize -= sizeof(ui8); //exclude packet length value

    TRecord record;
    size_t recordSize = PacketSize / packetsCount;
    size_t minVersion = 0;
    size_t maxVersion = 1;

    for (size_t version = minVersion; version <= maxVersion; ++version) {
        if (record.GetSize(version) == recordSize) {
            return version;
        }
    }

    return 0;
}

TString NDrive::NVega::TFastDataRecords::DebugString() const {
    TStringBuilder result;
    result << "FastDataRecords " << Records.size();
    for (auto&& record : Records) {
        result << ' ' << record.Timestamp << '.' << record.TickCounter;
    }
    return result;
}

void NDrive::NVega::TCommandRequest::TQueryFuelLevelArgs::SetIgnitionDuration(TDuration value) {
    Y_ENSURE(value.Seconds() <= 200, "the ignition duration should be less than 200 seconds; " << value.Seconds() << " > 200");
    IgnitionDuration = value.Seconds();
}

TDuration NDrive::NVega::TCommandRequest::TQueryFuelLevelArgs::GetIgnitionDuration() const {
    return TDuration::Seconds(IgnitionDuration);
}

void NDrive::NVega::TCommandRequest::TTankerFuelQuantity::SetFuelQuantity(float value) {
    Y_ENSURE(value <= 150, "the fuel quantityt should be less than 150 liters; " << value << " > 150");
}

float NDrive::NVega::TCommandRequest::TTankerFuelQuantity::GetFuelQuantity() const {
    return FuelQuantity;
}

void NDrive::NVega::TCommandRequest::TFuelUpdatePeriod::SetPeriod(uint16_t minutes) {
    Y_ENSURE(minutes > 0, "the period should be more than 0");
    Y_ENSURE(minutes <= 720, "the period should be less than 12 hours; " << minutes << " minutes > 12 hours");
    Period = minutes;
}

void NDrive::NVega::TCommandRequest::TFuelUpdatePeriod::SetPeriodForStop() {
    Period = 0;
}

uint16_t NDrive::NVega::TCommandRequest::TFuelUpdatePeriod::GetPeriod() const {
    return Period;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TArgument& object) {
    return object.DebugString();
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NVega::TArgument& result) try {
    TString decoded = HexDecode(value.GetStringSafe());
    result.SetData(decoded);
    return true;
} catch (const std::exception& e) {
    Cdbg << "cannot parse NDrive::NVega::TArgument: " << FormatExc(e) << Endl;
    return false;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand::EType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NVega::TCommandRequest::TAuxiallaryCarCommand4::EType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <class T>
NJson::TJsonValue PayloadToJson(const T& object, bool native = true) {
    if (native) {
        return NJson::ToJson(object.Tuple());
    } else {
        TString serialized;
        TStringOutput so(serialized);
        object.Save(so);
        return NJson::ToJson(NJson::HexString(serialized));
    }
}

template <class T>
bool TryPayloadFromJson(const NJson::TJsonValue& value, T& result) {
    if (NJson::TryFromJson(value, result.Tuple())) {
        return true;
    }

    TString serialized;
    if (!NJson::TryFromJson(value, NJson::HexString(serialized))) {
        return false;
    }

    TStringInput si(serialized);
    try {
        result.Load(si);
        return true;
    } catch (const std::exception& e) {
        Cdbg << "cannot parse NDrive::NVega::TMessage from " << value.GetStringRobust() << ": " << FormatExc(e) << Endl;
        return false;
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NProtocol::TStructure<TVega_rx_can_frame>& object) {
    const auto& data = object.Data;
    NJson::TJsonValue result;
    result["time"] = data.time;
    result["id"] = data.id;
    result["interface"] = data.itf_idx;
    result["type"] = data.id_type;
    result["rtr"] = data.rtr;
    result["data"] = NJson::RangeToJson(data.data + 0, data.data + data.dlen);
    result["hexdata"] = HexEncode(data.data, data.dlen);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NProtocol::TStructure<TVega_rx_can_frame>& result) {
    uint32_t time;
    uint32_t id;
    uint8_t itf_idx;
    uint8_t id_type;
    uint8_t rtr;
    TVector<uint8_t> data;
    bool parsed =
        NJson::TryFromJson(value["time"], time) &&
        NJson::TryFromJson(value["id"], id) &&
        NJson::TryFromJson(value["interface"], itf_idx) &&
        NJson::TryFromJson(value["type"], id_type) &&
        NJson::TryFromJson(value["rtr"], rtr) &&
        NJson::TryFromJson(value["data"], data);
    if (!parsed) {
        return false;
    }

    auto& frame = result.Data;
    if (data.size() > Y_ARRAY_SIZE(frame.data)) {
        return false;
    }
    frame.time = time;
    frame.id = id;
    frame.itf_idx = itf_idx;
    frame.id_type = id_type;
    frame.rtr = rtr;
    frame.dlen = data.size();
    std::memcpy(frame.data, data.data(), data.size());
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TCanResponse& object) {
    NJson::TJsonValue result;
    result["frames"] = NJson::ToJson(object.Frames);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TCommandRequestSigned& object) {
    return PayloadToJson(object);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NVega::TCommandRequestSigned& result) {
    return TryPayloadFromJson(value, result);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NVega::TCommandResponseSigned& object) {
    return PayloadToJson(object);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NVega::TCommandResponseSigned& result) {
    return TryPayloadFromJson(value, result);
}

size_t NDrive::NVega::TFastDataRecords::TRecord::CalcSize() const {
    return GetSize(Version);
}

size_t NDrive::NVega::TFastDataRecords::TRecord::GetSize(ui8 version) const {
    size_t result = 0;

    result += NPrivate::SizeOf(Version);
    result += NPrivate::SizeOf(DropCounter);
    result += NPrivate::SizeOf(TickCounter);
    result += NPrivate::SizeOf(AccelerometerX);
    result += NPrivate::SizeOf(AccelerometerY);
    result += NPrivate::SizeOf(AccelerometerZ);
    result += NPrivate::SizeOf(Timestamp);
    result += NPrivate::SizeOf(Longitude);
    result += NPrivate::SizeOf(Latitude);
    result += NPrivate::SizeOf(Valid);
    result += NPrivate::SizeOf(Speed);
    result += NPrivate::SizeOf(Course);
    result += NPrivate::SizeOf(Active);
    result += NPrivate::SizeOf(CanSpeed);
    result += NPrivate::SizeOf(Accelerator);
    result += NPrivate::SizeOf(Brake);
    result += NPrivate::SizeOf(SteeringWheel);
    result += NPrivate::SizeOf(Rpm);

    if (version >= 1) {
        result += NPrivate::SizeOf(SteeringAcceleration);
    }

    return result;
}

size_t NDrive::NVega::TAuthorizationResponse::GetSize() const {
    size_t result = 0;

    if (Status == Encrypt) {
        result += NPrivate::SizeOf(Encrypt);
    }
    result += NPrivate::SizeOf(Status);

    return result;
}

void NDrive::NVega::TFastDataRecords::TRecord::Save(IOutputStream* stream) const {
    ::Save(stream, Version);
    ::Save(stream, DropCounter);
    ::Save(stream, TickCounter);
    ::Save(stream, AccelerometerX);
    ::Save(stream, AccelerometerY);
    ::Save(stream, AccelerometerZ);
    ::Save(stream, Timestamp);
    ::Save(stream, Longitude);
    ::Save(stream, Latitude);
    ::Save(stream, Valid);
    ::Save(stream, Speed);
    ::Save(stream, Course);
    ::Save(stream, Active);
    ::Save(stream, CanSpeed);
    ::Save(stream, Accelerator);
    ::Save(stream, Brake);
    ::Save(stream, SteeringWheel);
    ::Save(stream, Rpm);

    if (Version >= 1) {
        ::Save(stream, SteeringAcceleration);
    }
}

void NDrive::NVega::TFastDataRecords::TRecord::Load(IInputStream* stream) {
    ::Load(stream, DropCounter);
    ::Load(stream, TickCounter);
    ::Load(stream, AccelerometerX);
    ::Load(stream, AccelerometerY);
    ::Load(stream, AccelerometerZ);
    ::Load(stream, Timestamp);
    ::Load(stream, Longitude);
    ::Load(stream, Latitude);
    ::Load(stream, Valid);
    ::Load(stream, Speed);
    ::Load(stream, Course);
    ::Load(stream, Active);
    ::Load(stream, CanSpeed);
    ::Load(stream, Accelerator);
    ::Load(stream, Brake);
    ::Load(stream, SteeringWheel);
    ::Load(stream, Rpm);

    if (Version >= 1) {
        ::Load(stream, SteeringAcceleration);
    }
}

void NDrive::NVega::TAuthorizationResponse::Load(IInputStream& stream) {
    ::Load(&stream, Status);
    if (Status == Encrypt) {
        ::Load(&stream, EncryptedData);
    }
}

void NDrive::NVega::TAuthorizationResponse::Save(IOutputStream& stream) const {
    ::Save(&stream, Status);
    if (Status == Encrypt) {
        ::Save(&stream, EncryptedData);
    }
}

NDrive::TMultiSensor NDrive::NVega::ToSensors(const TBlackboxRecords::TRecord& record) {
    NDrive::TMultiSensor sensors;

    auto timestamp = TInstant::Seconds(record.Timestamp);
    auto add = [&sensors, timestamp] (ui32 id, auto value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };
    add(VEGA_LAT, static_cast<double>(record.Lattitude));
    add(VEGA_LON, static_cast<double>(record.Longitude));
    add(VEGA_ALT, static_cast<double>(record.Height));
    add(VEGA_SPEED, static_cast<double>(record.Speed));
    add(VEGA_DIR, static_cast<double>(record.Course));
    add(VEGA_SAT_USED, static_cast<ui64>(record.Satelites));
    record.Parameters.Iterate([add](auto&& parameter) {
        add(parameter.Id, NDrive::SensorValueFromRef(parameter.GetValue()));
    });

    return sensors;
}
