#pragma once

#include "poly1305.h"
#include "protocol.h"
#include "sensor.h"
#include "types.h"

#include <drive/vega/lib/trunk/vega_libs/serv_protocols_lib/vega/vega_protocol.h>

#include <util/generic/buffer.h>
#include <util/generic/utility.h>
#include <util/stream/mem.h>
#include <util/system/mutex.h>

#include <array>

constexpr ui16 CAN_FUEL_DISTANCE_KM         = CAN_FUEL_LEVEL_L;
constexpr ui16 CAN_STEERING_WHEEL           = CAN_BATTERY_CHARGE_LEVEL;
constexpr ui16 CAN_TRANSMISSION_SELECTOR    = CAN_STAT_TRANSMISSION;
constexpr ui32 IDLE_TIME_AS_SENSOR          = 100001;
constexpr ui16 CAN_CUSTOM_FUEL_VOLUME       = 2803;
constexpr ui16 BLE_FUEL_LEVEL               = BLE_SENSOR1_U16_1;
constexpr ui16 CAN_CLS_STATE                = 2830;


namespace NDrive {
    namespace NProtocol {
        class IConnection;
        enum class EHandlerStatus;
    }

    namespace NVega {
        enum class EValueType {
            Unknown,
            UI8,
            UI16,
            UI32,
            Float32,
            ASCII,
            Binary,
        };
        enum EValueTrait : ui32 {
            vtNone         = 0,
            vtSkipZero     = 1 << 0,
            vtHashValue    = 1 << 1,
            vtPercentValue = 1 << 2,
            vtFilterOdometer = 1 << 3,
            vtLeftStrlen   = 1 << 4,
            vtSkipZeroIdle = 1 << 5,
            vtSkipInvalidPercent  = 1 << 6,
        };

        class TCommandRequest;
        using TCommandCode = ui8;
        enum ECommandCode : TCommandCode {
            UNKNOWN                = 0,
            RESTART                = EVega_command_code::RESTART,
            SET_PARAM              = EVega_command_code::SET_PARAM,
            GET_PARAM              = EVega_command_code::GET_PARAM,
            GET_EXT_PARAM          = EVega_command_code::GET_EXT_PARAM,
            TOFACTORY              = EVega_command_code::TOFACTORY,
            CLEAN_BB               = EVega_command_code::CLEAN_BB,
            SET_OUTPUT             = EVega_command_code::SET_OUTPUT,
            DISCONNECT             = EVega_command_code::TAKE_PHOTO,
            TAKE_PHOTO             = EVega_command_code::TAKE_PHOTO,
            BLINKER_FLASH          = EVega_command_code::BLINKER_FLASH,
            CLOSE_DOORS            = EVega_command_code::CLOSE_DOORS,
            OPEN_DOORS             = EVega_command_code::OPEN_DOORS,
            OPEN_DR_DOOR           = EVega_command_code::OPEN_DR_DOOR,
            HORN                   = EVega_command_code::HORN,
            TRUNK                  = EVega_command_code::TRUNK,
            STOP_ENGINE            = EVega_command_code::STOP_ENGINE,
            START_ENGINE           = EVega_command_code::START_ENGINE,
            START_WEBASTO          = EVega_command_code::START_WEBASTO,
            STOP_WEBASTO           = EVega_command_code::STOP_WEBASTO,
            EMULATE_DRIVER_DOOR    = EVega_command_code::EMULATE_DRIVER_DOOR,
            HORN_AND_BLINK         = EVega_command_code::HORN_AND_BLINK,
            WINDOWS_CLOSING_3S     = EVega_command_code::WINDOWS_CLOSING_3S,
            WINDOWS_CLOSING_7S     = EVega_command_code::WINDOWS_CLOSING_7S,
            WINDOWS_CLOSING_11S    = EVega_command_code::WINDOWS_CLOSING_11S,
            WINDOWS_CLOSING_15S    = EVega_command_code::WINDOWS_CLOSING_15S,
            WINDOWS_CLOSING_19S    = EVega_command_code::WINDOWS_CLOSING_19S,
            WINDOWS_CLOSING_23S    = EVega_command_code::WINDOWS_CLOSING_23S,
            WINDOWS_CLOSING_29S    = EVega_command_code::WINDOWS_CLOSING_29S,
            WINDOWS_OPENING_3S     = EVega_command_code::WINDOWS_OPENING_3S,
            WINDOWS_OPENING_7S     = EVega_command_code::WINDOWS_OPENING_7S,
            WINDOWS_OPENING_11S    = EVega_command_code::WINDOWS_OPENING_11S,
            WINDOWS_OPENING_15S    = EVega_command_code::WINDOWS_OPENING_15S,
            WINDOWS_OPENING_19S    = EVega_command_code::WINDOWS_OPENING_19S,
            WINDOWS_OPENING_23S    = EVega_command_code::WINDOWS_OPENING_23S,
            WINDOWS_OPENING_29S    = EVega_command_code::WINDOWS_OPENING_29S,
            START_PROD_TESTS       = EVega_command_code::START_PROD_TESTS,
            RESET_LORA_SENS_ALARMS = EVega_command_code::RESET_LORA_SENS_ALARMS,

            YADRIVE_WARMING        = EVega_command_code::YADRIVE_WARMING,
            YADRIVE_STOP_WARMING   = EVega_command_code::YADRIVE_STOP_WARMING,
            YADRIVE_START_OF_LEASE = EVega_command_code::YADRIVE_START_OF_LEASE,
            YADRIVE_END_OF_LEASE   = EVega_command_code::YADRIVE_END_OF_LEASE,
            YADRIVE_UNLOCK_HOOD    = EVega_command_code::YADRIVE_UNLOCK_HOOD,
            YADRIVE_LOCK_HOOD      = EVega_command_code::YADRIVE_LOCK_HOOD,
            YADRIVE_FORCED_END_OF_LEASE = EVega_command_code::YADRIVE_FORCED_END_OF_LEASE,
            YADRIVE_PANIC          = EVega_command_code::YADRIVE_PANIC,
            YADRIVE_AUDIO_COMMAND  = EVega_command_code::YADRIVE_AUDIO_COMMANDS,
            YADRIVE_DTC_CLEAR      = EVega_command_code::YADRIVE_DTC_CLEAR,
            YADRIVE_EMERGENCY_STOP = EVega_command_code::YADRIVE_EMERGENCY_STOP,

            ELECTRIC_CAR_COMMAND   = EVega_command_code::AUXILIARY_CAR_COMMANDS,
            AUXILIARY_CAR_COMMAND  = EVega_command_code::AUXILIARY_CAR_COMMANDS,

            NRF_ADD_RELAY          = EVega_command_code::INTERNAL_NRF_ADD_RELAY,
            NRF_REMOVE_RELAY       = EVega_command_code::INTERNAL_NRF_DEL_RELAY,
            NRF_ADD_MARK           = EVega_command_code::INTERNAL_NRF_ADD_MARK,
            NRF_REMOVE_MARK        = EVega_command_code::INTERNAL_NRF_DEL_MARK,

            OBD_FORWARD_CONFIG     = EVega_command_code::CAN_VEGA_FWD_CONFIG,
            OBD_COMMAND            = EVega_command_code::CAN_VEGA_OBD_COMMAND,
            CAN_SCRIPT             = EVega_command_code::CAN_VEGA_TX_SCRIPT_COMMAND,

            GSM_MODEM_COMMAND      = EVega_command_code::GSM_MODEM_COMMANDS,
            FAST_DATA_CONFIG       = EVega_command_code::FASTDATA_FWD_CONFIG,

            // scenario
            SCENARIO_STOP_WARMING_AND_OPEN_DOORS   = 0x80,
            SCENARIO_STOP_WARMING_AND_END_OF_LEASE = 0x81,
            SCENARIO_UNLOCK_DOORS_AND_HOOD         = 0x82,
            SCENARIO_LOCK_DOORS_AND_HOOD           = 0x83,
            SCENARIO_UNLOCK_HOOD_AND_START_OF_LEASE= 0x84,
            SCENARIO_STOP_WARMING_END_OF_LEASE_AND_LOCK_HOOD = 0x85,
            SCENARIO_ENABLE_LONGHORN               = 0x86,
            SCENARIO_DISABLE_LONGHORN              = 0x87,
            SCENARIO_POLITE_FORCED_END_OF_LEASE    = 0x88,
            SCENARIO_BLE_RESET                     = 0x89,
            SCENARIO_PREPARE                       = 0x8A,
            SCENARIO_RESET                         = 0x8B,
            SCENARIO_FORCED_RESET                  = 0x8C,
            SCENARIO_POLITE_RESTART                = 0x8D,
            SCENARIO_DIN2_SHUTDOWN                 = 0x8E,
            SCENARIO_POLITE_SET_PARAM              = 0x8F,
            SCENARIO_BLE_RESET_FORCED_END_OF_LEASE = 0x90,
            SCENARIO_QUERY_BEACONS                 = 0x91,
            SCENARIO_QUERY_BEACONS_NO_WAIT         = 0x92,
            SCENARIO_START_IGNITION                = 0x93,
            SCENARIO_STOP_IGNITION                 = 0x94,
            SCENARIO_QUERY_FUEL_LEVEL              = 0x95,
            SCENARIO_OBD_REQUEST                   = 0x96,
            SCENARIO_POLITE_WIRELESS_BLOCK         = 0x97,
            SCENARIO_WIRELESS_BLOCK                = 0x98,
            SCENARIO_WIRELESS_UNBLOCK              = 0x99,
            SCENARIO_OPEN_DOORS                    = 0x9A,
            SCENARIO_CLOSE_DOORS                   = 0x9B,
            SCENARIO_RAPID_OFF_EMERGENCY_LIGHTS    = 0x9C,
            SCENARIO_WIRED_BLOCK                   = 0x9D,
            SCENARIO_WIRED_UNBLOCK                 = 0x9E,
            SCENARIO_POLITE_WIRED_BLOCK            = 0x9F,
            SCENARIO_ALL_BLOCK                     = 0xA0,
            SCENARIO_ALL_UNBLOCK                   = 0xA1,
            SCENARIO_POLITE_ALL_BLOCK              = 0xA2,
            SCENARIO_FTANKER_SET_GASOLINE_LITERS   = 0xA3,
            SCENARIO_FTANKER_SET_DIESEL_LITERS     = 0xA4,
            SCENARIO_START_FUEL_AUTO_UPDATE        = 0xA5,
            SCENARIO_STOP_FUEL_AUTO_UPDATE         = 0xA6,
            // car control for TTelematicsTestClient
            MOVE_TO_COORD = 0xf0,
        };

        using TValueTraits = ui32;
        using TArgument = NProtocol::TArgument;
        using IConnection = NProtocol::IConnection;
        using EHandlerStatus = NProtocol::EHandlerStatus;

        TSensorId GetSensorId(TStringBuf name);
        TStringBuf GetSensorName(TSensorId id);
        TVector<TStringBuf> GetSensorNames();
        TVector<TSensorId> GetSensorIds();

        EValueType GetValueType(TSensorId id);
        TValueTraits GetValueTraits(TSensorId id);

        template <ui8 N>
        constexpr ui16 AuxFuelLevel() {
            static_assert(N > 0, "incorrect input index");
            return VEGA_DUT_1 + N - 1;
        }

        template <ui8 N>
        constexpr ui16 AuxTemperature() {
            static_assert(N > 0, "incorrect input index");
            return VEGA_DUT_TEMP_1 + N - 1;
        }

        template <ui8 N>
        constexpr ui16 DigitalInput() {
            static_assert(N > 0, "incorrect input index");
            return VEGA_DIGITAL_INPUT_1 + N - 1;
        }

        template <ui8 N>
        constexpr ui16 DigitalOutput() {
            static_assert(N > 0, "incorrect output index");
            return VEGA_DIGITAL_OUTPUT_1 + N - 1;
        }

        inline constexpr size_t ServerSettingsCount = 4;
        inline constexpr size_t APNSettingsCount = 2;
        inline constexpr size_t CanParametersCount = 128;
        inline constexpr size_t CanScriptCount = 128;

        inline constexpr size_t LogicScriptCommandsCount = 240;
        inline constexpr size_t LogicScriptChecksCount = 60;
        inline constexpr size_t LogicScriptScriptsCount = 40;
        inline constexpr size_t LogicScriptTriggersCount = 15;
        inline constexpr size_t LogicScriptNotifiesCount = 40;

        inline bool IsSettingId(ui16 id) {
            return VEGA_SETTING_START_ID <= id && id <= VEGA_SETTING_LAST_ID;
        }

        inline ui16 DigitalOutputId(ui16 index) {
            ui16 result = VEGA_DIGITAL_OUTPUT_1 + index - 1;

            if (result < VEGA_DIGITAL_OUTPUT_1) {
                result = VEGA_DIGITAL_OUTPUT_1;
            }
            if (result > VEGA_DIGITAL_OUTPUT_20) {
                result = VEGA_DIGITAL_OUTPUT_20;
            }
            return result;
        }

        constexpr TSensorId BleMac = { BLE_EXT_BOARD_MAC };
        constexpr TSensorId BlePasskey = { VEGA_SETTING_BLE_EXT_BOARD, 1 };
        constexpr TSensorId BleSessionKey = { 33007 };
        constexpr TSensorId SvrRawState = { 8350 };
        constexpr TSensorId VegaMcuFirmwareVersionRevision = { VEGA_MCU_FIRMWARE_VERSION, 1 };

        constexpr ui8 InvalidPercent = 255;

        enum EMessageType: ui8 {
            INCORRECT            = 0,
            COMMAND_REQUEST      = EVega_message_type::COMMAND,
            COMMAND_RESPONSE     = EVega_message_type::COMMAND_ANS,
            FILE_CHUNK_REQUEST   = EVega_message_type::FILE_CHUNK,
            FILE_CHUNK_RESPONSE  = EVega_message_type::FILE_CHUNK_ANS,
            AUTH_REQUEST         = EVega_message_type::AUTORIZATION,
            AUTH_RESPONSE        = EVega_message_type::AUTORIZATION_ANS,
            GET_FILE_REQUEST     = EVega_message_type::GET_FILE,
            GET_FILE_RESPONSE    = EVega_message_type::GET_FILE_ANS,
            PING_REQUEST         = EVega_message_type::GET_DEV_INF,
            PING_RESPONSE        = EVega_message_type::GET_DEV_INF_ANS,
            LIST_REQUEST         = EVega_message_type::GET_DEV_LIST,
            LIST_RESPONSE        = EVega_message_type::GET_DEV_LIST_ANS,
            CONNECT_REQUEST      = EVega_message_type::CONNECT_TO_DEV,
            CONNECT_RESPONSE     = EVega_message_type::CONNECT_TO_DEV_ANS,
            DISCONNECTED         = EVega_message_type::DISCONNECT_FROM_DEV,
            BLACKBOX_RECORDS     = EVega_message_type::BBOX_MESSAGES,
            BLACKBOX_RECORDS_ACK = EVega_message_type::BBOX_MESSAGES_ANS,
            FILE_STREAM_REQUEST  = EVega_message_type::FILE_WHOLLY,
            FILE_STREAM_RESPONSE = EVega_message_type::FILE_WHOLLY_ANS,
            INTERFACE_REQUEST    = EVega_message_type::TRANSP_DATA_TO_ITF,
            INTERFACE_RESPONSE   = EVega_message_type::TRANSP_DATA_FROM_ITF,
            CAN_REQUEST          = EVega_message_type::FWD_DATA_TO_CAN,
            CAN_RESPONSE         = EVega_message_type::FWD_DATA_FROM_CAN,
            FAST_DATA_RECORDS    = EVega_message_type::FAST_DATA_MESS,
            COMMAND_REQUEST_SIGNED  = EVega_message_type::COMMAND_W_NONCE,
            COMMAND_RESPONSE_SIGNED = EVega_message_type::COMMAND_ANS_W_SINGN,
        };

        class TMessage: public NProtocol::IMessage {
        public:
            using TBaseMessageType = NProtocol::TMessageType;
            using TMessageType = EMessageType;

            TMessage(EMessageType type = INCORRECT);
            TMessage(THolder<NProtocol::IPayload>&& payload);

            TMessage(TMessage&& other) noexcept = default;
            TMessage& operator=(TMessage&& other) noexcept = default;

            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
        };

        template<EMessageType Type>
        class TPayload: public NProtocol::IPayload {
        public:
            TPayload() {
                MessageType = Type;
                ProtocolType = NProtocol::PT_VEGA;
            }

            TString DebugString() const override {
                return ToString(Type);
            }

            NProtocol::TSequenceId GetSequenceId() const override {
                return 0;
            }
        };

        class TValue {
        private:
            NProtocol::TArray<char, ui8> Value;

        public:
            static TSensorRef Get(TConstArrayRef<char> value, TSensorId id = 0);

        public:
            DEFINE_FIELDS(Value);

            TSensorRef Get(TSensorId id = 0) const;

            template <class T>
            void Set(T value, ui16 id = 0);
        };

        class TIdValue {
        public:
            ui16 Id = 0;

            auto GetValue() const {
                return Value.Get(Id);
            }
            template <class T>
            void SetValue(T value) {
                Value.Set(value, Id);
            }

            DEFINE_FIELDS(
                Id,
                Value
            );

        private:
            TValue Value;
        };

        class TIdTypeValue {
        public:
            enum EType: ui8 {
                UNKNOWN,
            };

        public:
            ui16 Id = 0;
            ui8 Type = INCORRECT;

            auto GetValue() const {
                return Value.Get(Id);
            }
            template <class T>
            void SetValue(T value) {
                Value.Set(value, Id);
            }

            DEFINE_FIELDS(
                Id,
                Type,
                Value
            );

        private:
            TValue Value;
        };

        class TIdSubIdValue {
        public:
            ui16 Id = 0;
            ui16 SubId = 0;

            auto GetSensorId() const {
                return TSensorId(Id, SubId);
            }
            auto GetValue() const {
                return Value.Get(Id);
            }
            template <class T>
            void SetValue(T value) {
                Value.Set(value, Id);
            }

            DEFINE_FIELDS(
                Id,
                SubId,
                Value
            );

        private:
            TValue Value;
        };

        class TCommandRequest: public TPayload<COMMAND_REQUEST> {
        public:
            using TCommandCode = NVega::TCommandCode;
            using ECommandCode = NVega::ECommandCode;

            struct TAudioFile {
            public:
                ui16 FileId = 0;

                DEFINE_FIELDS(
                    Type,
                    FileId
                );

            private:
                ui8 Type = 0;
            };

            struct TAudioScenario {
            public:
                ui8 ScenarioId = 0;

                DEFINE_FIELDS(
                    Type,
                    ScenarioId
                );

            private:
                ui8 Type = 1;
            };

            struct TGetParameter {
                ui16 Id = 0;
                ui16 SubId = 0;

                DEFINE_FIELDS(
                    Id,
                    SubId
                );
            };

            struct TAuxiallaryCarCommand {
                enum EType : ui8 {
                    UNBLOCK_CHARGE_CONNECTOR = 2,
                    UNBLOCK_CHARGE_PORT = 3,
                    ENABLE_ECO          = 4,
                    DISABLE_ECO         = 5,
                    LOCK_FUEL_CAP       = 6,
                    UNLOCK_FUEL_CAP     = 7,
                    LOCK_TRUNK          = 8,
                    UNLOCK_TRUNK        = 9,
                };

                ui8 Type = 0;

                DEFINE_FIELDS(
                    Type
                );
            };
            using TElectricCarCommand = TAuxiallaryCarCommand;

            struct TAuxiallaryCarCommand4 {
                enum EType : ui8 {
                    CONTROL_HEAD_UNIT = 10,
                };

                ui8 Type = 0;
                ui8 TurnOn = 0;
                ui16 Duration = 15;

                DEFINE_FIELDS(
                    Type,
                    TurnOn,
                    Duration
                );
            };

            struct TCanScript {
                ui8 Id = 0;
                enum class EType : ui8 {
                    CLOSE_DOORS = 1,
                    OPEN_DOORS  = 2,
                };

                DEFINE_FIELDS(
                    Id
                );
            };

            struct TFastDataConfig {
            private:
                ui8 Type = 0;

            public:
                ui8 Discretization = 0;
                ui8 Duration = 0;

                DEFINE_FIELDS(
                    Type,
                    Discretization,
                    Duration
                );

                TDuration GetDiscretization() const {
                    return TDuration::MilliSeconds(Discretization * 10);
                }
                TDuration GetDuration() const {
                    return TDuration::Seconds(Duration);
                }

                void SetDiscretization(TDuration value);
                void SetDuration(TDuration value);
            };

            struct TMark {
                ui8 Id = 0;

                DEFINE_FIELDS(
                    Id
                );
            };

            struct TMoveToCoordParameter {
                float Lon = 0;
                float Lat = 0;
                float Speed = 90;

                DEFINE_FIELDS(
                    Lon,
                    Lat,
                    Speed
                );
            };

            struct TObdForwardConfig {
                typedef struct {
                    uint32_t id_type : 3;   // тип CAN фрейма. 0 - reserved, 1 - standart 11 bit frame, 2 - extended 29 bit frame.
                    uint32_t value : 29;    // ID CAN фрейма.
                    uint32_t mask : 29;     // Маска. 0x0-0x7FF, если standart frame, 0x0-0x1FFFFFFF если extended frame.
                    uint32_t is_enable : 1; // 0-disabled, 1-enabled
                } can_fwd_settings_t;

                using TSetting = NProtocol::TStructure<can_fwd_settings_t>;
                using TSettings = NProtocol::TArray<TSetting, ui8>;

                ui8 Type = 0;               // should remain this way
                TSettings Cans;
                ui8 Duration = 0;

                DEFINE_FIELDS(
                    Type,
                    Cans,
                    Duration
                );
            };

            struct TCustomObdForwardConfig {
                ui8 Duration = 0;
                ui32 Value = 0;
                ui32 Mask = 0;

                DEFINE_FIELDS(
                    Duration,
                    Value,
                    Mask
                );
            };

            struct TObdCommand {
                enum EType : ui8 {
                    NONE        = 0,
                    UPDATE_VIN  = 1,
                    UPDATE_DTC  = 2,
                    CLEAR_DTC   = 3,
                };

                ui8 Type = NONE;

                DEFINE_FIELDS(
                    Type
                );
            };

            struct TObdRequest {
                ui32 Id = 0;
                ui8 Index = 0;
                NProtocol::TArray<ui8, ui8> Data;

                ui32 ResponseId = 0;

                DEFINE_FIELDS(
                    Id,
                    Index,
                    Data,
                    ResponseId
                );
            };

            struct TPanic {
                enum EType : ui8 {
                    NONE            = 0,
                    HORN            = 1,
                    BLINK           = 2,
                    HORN_AND_BLINK  = 3,
                };

                ui8 Type = NONE;
                ui8 Time = 0;

                DEFINE_FIELDS(
                    Type,
                    Time
                );
            };

            class TQueryFuelLevelArgs {
            public:
                void SetIgnitionDuration(TDuration value);
                TDuration GetIgnitionDuration() const;

            public:
                DEFINE_FIELDS(
                    IgnitionDuration
                );

            private:
                ui8 IgnitionDuration;
            };

            class TTankerFuelQuantity {
            public:
                void SetFuelQuantity(float value);
                float GetFuelQuantity() const;

                static TSensorId GetGasolineSensorId() {
                    return TSensorId(VEGA_SETTING_PUMP, 1);
                }
                static TSensorId GetDieselSensorId() {
                    return TSensorId(VEGA_SETTING_PUMP, 2);
                }

            public:
                DEFINE_FIELDS(
                    FuelQuantity
                );

            private:
                float FuelQuantity;
            };

            class TFuelUpdatePeriod {
            public:
                void SetPeriod(uint16_t minutes);
                void SetPeriodForStop();
                uint16_t GetPeriod() const;

                static TSensorId GetSensorId() {
                    return VEGA_SETTING_FUEL_LEVEL_FORCE_READ;
                }

            public:
                DEFINE_FIELDS(
                    Period
                );

            private:
                uint16_t Period;
            };

            struct TSwitchSim {
            public:
                DEFINE_FIELDS(
                    Type
                );

            private:
                ui8 Type = 0;
            };

            struct TWarming {
                ui8 Time = 0;

                DEFINE_FIELDS(
                    Time
                );
            };

            struct TBlinkerFlash {
                ui16 Time = 0; // in seconds
                ui16 Sampling = 0; // in milliseconds

                DEFINE_FIELDS(
                    Time,
                    Sampling
                );
            };

            using TSetParameter = TIdSubIdValue;

        public:
            ui32 Id = 0;
            TCommandCode Code = UNKNOWN;
            TArgument Argument;

            DEFINE_MESSAGE_FIELDS(
                Id,
                Code,
                Argument
            );

        public:
            virtual TString DebugString() const override;

            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }
        };

        class TCommandRequestSigned: public TPayload<COMMAND_REQUEST_SIGNED> {
        public:
            using TNonce = std::array<ui8, 8>;

        public:
            TNonce Nonce = MakeFilledArray(0);
            ui32 Id = 0;
            TCommandRequest::TCommandCode Code = ECommandCode::UNKNOWN;
            TArgument Argument;

            DEFINE_MESSAGE_FIELDS(
                Nonce,
                Id,
                Code,
                Argument
            );

        public:
            virtual TString DebugString() const override;

            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }
        };

        class TCommandResponse: public TPayload<COMMAND_RESPONSE> {
        public:
            enum EResult : ui8 {
                UNKNOWN           = 0,
                PROCESSED         = EVega_command_ans_result::DONE,
                IN_PROGRESS       = EVega_command_ans_result::IN_PROCESS,
                READY_TO_CONTINUE = EVega_command_ans_result::READY_TO_CONTINUE,
                ERROR             = EVega_command_ans_result::COMM_ERROR,
                INCORRECT         = EVega_command_ans_result::UNKNOWN_PARAMETER,
                BUSY              = EVega_command_ans_result::BUSY,
            };

            using TGetParameter = TIdValue;
            using TInfo = NProtocol::TZeroTerminatedString;
            struct TDiagnostics {
                ui32 Result = 0;

                DEFINE_FIELDS(
                    Result
                );
            };

        public:
            ui32 Id = 0;
            ui8 Result = UNKNOWN;
            TArgument Argument;

            DEFINE_MESSAGE_FIELDS(
                Id,
                Result,
                Argument
            );

        public:
            virtual TString DebugString() const override;

            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }
        };

        class TCommandResponseSigned: public TPayload<COMMAND_RESPONSE_SIGNED> {
        public:
            using TKey = NPoly1305::TKey;
            using TMac = NPoly1305::TMac;
            using TNonce = TCommandRequestSigned::TNonce;

        public:
            TMac Mac = MakeFilledArray(0);
            TNonce Nonce = MakeFilledArray(0);
            ui32 Id = 0;
            ui8 Result = TCommandResponse::UNKNOWN;
            TArgument Argument;

            DEFINE_MESSAGE_FIELDS(
                Mac,
                Nonce,
                Id,
                Result,
                Argument
            );

        public:
            bool ValidateMac(const TKey& key) const;

            virtual TString DebugString() const override;

            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }
        };

        class TAuthorizationRequest: public TPayload<AUTH_REQUEST> {
        public:
            static constexpr ui8 BlackboxEnabled = 1;
            static constexpr ui8 BlackboxDisabled = 0;

        public:
            std::array<ui8, 4> Password = MakeFilledArray(0);
            ui8 EnableBlackbox = BlackboxDisabled;

            DEFINE_MESSAGE_FIELDS(
                Password,
                EnableBlackbox
            );

        public:
            virtual TString DebugString() const override;
        };

        class THardAuthorizationRequest: public TPayload<AUTH_REQUEST> {
        public:
            NProtocol::TFixedSizeZeroTerminatedString<VEGA_HARD_PASS_LEN> Password;
            ui8 EnableBlackbox = TAuthorizationRequest::BlackboxDisabled;

            DEFINE_MESSAGE_FIELDS(
                Password,
                EnableBlackbox
            );

        public:
            virtual TString DebugString() const override;
        };

        class TAuthorizationResponse: public TPayload<AUTH_RESPONSE> {
        public:
            static constexpr ui8 Failure = 0;
            static constexpr ui8 Success = 1;
            static constexpr ui8 Encrypt = 255;

        public:
            ui8 Status = Failure;
            ui32 EncryptedData = 0;

            virtual size_t GetSize() const override;
            virtual void Load(IInputStream& stream) override;
            virtual void Save(IOutputStream& stream) const override;

        public:
            virtual TString DebugString() const override;
        };

        class TGetFileRequest: public TPayload<GET_FILE_REQUEST> {
        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            ui32 Offset = 0;
            ui16 Size = 0;

            DEFINE_MESSAGE_FIELDS(
                Name,
                Offset,
                Size
            );

        public:
            virtual TString DebugString() const override;
        };

        class TGetFileResponse: public TPayload<GET_FILE_RESPONSE> {
        public:
            enum EResult {
                OK              = EVega_get_file_ans_result::GET_FILE_OK,
                BUSY            = EVega_get_file_ans_result::GET_FILE_BUSY,
                OUT_OF_DATA     = EVega_get_file_ans_result::GET_FILE_OUT_OF_DATA,
                MEMORY_ERROR    = EVega_get_file_ans_result::GET_FILE_MEMORY_ERROR,
                FILENAME_ERROR  = EVega_get_file_ans_result::GET_FILE_FILENAME_ERROR,
                DATA_SIZE_ERROR = EVega_get_file_ans_result::GET_FILE_DATA_SIZE_ERROR,

                UNKNOWN_ERROR   = 0xa0,
            };

            class TContent {
            public:
                ui8 Result = OK;
                TBuffer Data;

            public:
                size_t CalcSize() const;
                void Save(IOutputStream* s) const;
                void Load(IInputStream* s);
            };

        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            ui32 TotalSize = 0;
            TContent Content;

            DEFINE_MESSAGE_FIELDS(
                Name,
                TotalSize,
                Content
            );

        public:
            virtual TString DebugString() const override;
        };

        class TPingRequest: public TPayload<PING_REQUEST> {
        public:
            NO_MESSAGE_FIELDS();
        };

        class TPingResponse: public TPayload<PING_RESPONSE> {
        public:
            NProtocol::TFixedSizeZeroTerminatedString<16> IMEI;
            ui8 DeviceType = 0;
            std::array<ui8, 4> Reserved = MakeFilledArray(0);

            DEFINE_MESSAGE_FIELDS(
                IMEI,
                DeviceType,
                Reserved
            );

        public:
            virtual TString DebugString() const override;
        };

        class TListRequest: public TPayload<LIST_REQUEST> {
        public:
            NO_MESSAGE_FIELDS();
        };

        class TListResponse: public TPayload<LIST_RESPONSE> {
        public:
            struct TDevice {
                NProtocol::TFixedSizeZeroTerminatedString<16> IMEI;
                NProtocol::TFixedSizeZeroTerminatedString<41> Name;

                DEFINE_FIELDS(
                    IMEI,
                    Name
                );
            };

        public:
            NProtocol::TArray<TDevice, ui16> Devices;

            DEFINE_MESSAGE_FIELDS(Devices);

        public:
            virtual TString DebugString() const override;
        };

        class TConnectRequest: public TPayload<CONNECT_REQUEST> {
        public:
            TListResponse::TDevice Device;

            DEFINE_MESSAGE_FIELDS(Device);

        public:
            virtual TString DebugString() const override;
        };

        class TConnectResponse: public TPayload<CONNECT_RESPONSE> {
        public:
            enum EResult {
                OK        = EVega_connect_to_dev_res::CONNECT_OK,
                FAIL      = EVega_connect_to_dev_res::CONNECT_FAIL,
                NOT_FOUND = EVega_connect_to_dev_res::CONNECT_NOT_FOUND,
                USED      = EVega_connect_to_dev_res::CONNECT_USED,
                UNKNOWN   = EVega_connect_to_dev_res::CONNECT_UNKNOWN,
            };

        public:
            ui8 Result = UNKNOWN;

            DEFINE_MESSAGE_FIELDS(Result);

        public:
            virtual TString DebugString() const override;
        };

        class TDisconnectedNotification: public TPayload<DISCONNECTED> {
        public:
            NO_MESSAGE_FIELDS();
        };

        class TBlackboxRecords: public TPayload<BLACKBOX_RECORDS> {
        public:
            using TParameter = TIdValue;
            using TTypedParameter = TIdTypeValue;

            struct TParameters {
            public:
                using TValue = TVector<TParameter>;
                using TTypedValue = TVector<TTypedParameter>;

            public:
                std::variant<TValue, TTypedValue> Value;

            public:
                const TValue& Get() const;
                TValue& Mutable();

                const TTypedValue& GetTyped() const;
                TTypedValue& MutableTyped();

                template <class F>
                void Iterate(F&& f) const {
                    std::visit([f = std::forward<F>(f)](auto&& value) {
                        for (auto&& parameter : value) {
                            f(parameter);
                        }
                    }, Value);
                }

                size_t CalcSize() const;
                size_t DataSize() const;
                bool IsEmpty() const;
                bool IsTyped() const;
                void Save(IOutputStream* s) const;
                void Load(IInputStream* s);

            private:
                template <class T>
                const T& GetImpl() const;
                template <class T>
                T& MutableImpl();
            };

            struct TRecord {
                ui32 Timestamp = 0;
                float Lattitude = 0;
                float Longitude = 0;
                i16 Height = 0;
                ui16 Speed = 0;
                ui16 Course = 0;
                ui8 Satelites = 0;
                TParameters Parameters;

                DEFINE_FIELDS(
                    Timestamp,
                    Lattitude,
                    Longitude,
                    Height,
                    Speed,
                    Course,
                    Satelites,
                    Parameters
                );
            };
            using TRecords = NProtocol::TArray<TRecord, ui16>;

        public:
            ui16 Id = 0;
            TRecords Records;

            DEFINE_MESSAGE_FIELDS(
                Id,
                Records
            );

        public:
            virtual TString DebugString() const override;

            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }
        };

        class TBlackboxRecordsAck: public TPayload<BLACKBOX_RECORDS_ACK> {
        public:
            enum EResult {
                Received = 0x00,
                Error    = 0xff
            };

        public:
            ui16 Id = 0;
            ui8 Result = Error;

            DEFINE_MESSAGE_FIELDS(
                Id,
                Result
            );

        public:
            virtual NProtocol::TSequenceId GetSequenceId() const override {
                return Id;
            }

            virtual TString DebugString() const override;
        };

        class TFileChunkRequest: public TPayload<FILE_CHUNK_REQUEST> {
        public:
            class TChunk {
            public:
                TChunk() = default;
                TChunk(ui16 id, ui16 count, TBuffer&& data);

                const TBuffer& GetData() const {
                    return Data;
                }
                ui16 GetCount() const {
                    return Count;
                }
                ui16 GetId() const {
                    return Id;
                }

                size_t CalcSize() const;
                ui16 DataSize() const;
                void Save(IOutputStream* s) const;
                void Load(IInputStream* s);

                TString DebugString() const;

            private:
                TBuffer Data;

                ui16 Id = 0;
                ui16 Count = 0;
            };

        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            TChunk Chunk;

            DEFINE_MESSAGE_FIELDS(
                Name,
                Chunk
            );

        public:
            virtual TString DebugString() const override;
        };

        class TFileChunkResponse: public TPayload<FILE_CHUNK_RESPONSE> {
        public:
            enum EResult {
                OK              = EVega_file_chunk_ans_result::CHUNK_ANS_OK,
                MEMORY_ERROR    = EVega_file_chunk_ans_result::CHUNK_ANS_MEMORY_ERROR,
                SIGNATURE_ERROR = EVega_file_chunk_ans_result::CHUNK_ANS_SIGNATURE_ERROR,
                SEQUENCE_ERROR  = EVega_file_chunk_ans_result::CHUNK_ANS_SEQUENCE_ERROR,
                FILENAME_ERROR  = EVega_file_chunk_ans_result::CHUNK_ANS_FILENAME_ERROR,

                UNKNOWN_ERROR   = 0xa0,
                UNORDERED_ERROR = 0xa1,
                BUSY_ERROR      = 0xa2,
            };

        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            ui16 Chunk = 0;
            ui8 Result = OK;

            DEFINE_MESSAGE_FIELDS(
                Name,
                Chunk,
                Result
            );

        public:
            virtual TString DebugString() const override;
        };

        class TFileStreamRequest: public TPayload<FILE_STREAM_REQUEST> {
        public:
            class TContent {
            public:
                const TBuffer& Get() const {
                    return Data;
                }
                void Set(TBuffer&& data) {
                    Data = std::move(data);
                }

                ui32 CalcCrc32() const;
                size_t CalcSize() const;
                void Save(IOutputStream* s) const;
                void PostSave(IOutputStream* s) const;
                void Load(IInputStream* s);
                void PostLoad(IInputStream* s);

            private:
                TBuffer Data;

                mutable ui32 Size = 0;
                mutable ui32 Crc32 = 0;
            };

        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            TContent Content;

            DEFINE_MESSAGE_FIELDS(
                Name,
                Content
            );

        public:
            virtual void PostLoad(IInputStream& input) override {
                Content.PostLoad(&input);
            }
            virtual void PostSave(IOutputStream& output) const override {
                Content.PostSave(&output);
            }

            virtual TString DebugString() const override;
        };

        class TFileStreamResponse: public TPayload<FILE_STREAM_RESPONSE> {
        public:
            enum EResult {
                OK              = EVega_file_wholly_ans_result::FILE_WHOLLY_ANS_OK,
                MEMORY_ERROR    = EVega_file_wholly_ans_result::FILE_WHOLLY_MEMORY_ERROR,
                SIGNATURE_ERROR = EVega_file_wholly_ans_result::FILE_WHOLLY_SIGNATURE_ERROR,
                FILENAME_ERROR  = EVega_file_wholly_ans_result::FILE_WHOLLY_FILENAME_ERROR,
                CRC_ERROR       = EVega_file_wholly_ans_result::FILE_WHOLLY_CRC_ERROR,

                UNKNOWN_ERROR   = 0xa0,
            };

        public:
            NProtocol::TFixedSizeZeroTerminatedString<10> Name;
            ui8 Result = OK;

            DEFINE_MESSAGE_FIELDS(
                Name,
                Result
            );

        public:
            virtual TString DebugString() const override;
        };

        class TInterfaceData {
        public:
            enum EInterface {
                UNKNOWN = 0,
                RS232_1 = 1,
                RS485_1 = 2,
            };

        public:
            ui8 Interface = UNKNOWN;
            std::array<ui8, 7> Reserved = MakeFilledArray(0);
            NProtocol::TArray<char, ui16> Data;
        };

        class TInterfaceRequest
            : public TPayload<INTERFACE_REQUEST>
            , public TInterfaceData
        {
        public:
            DEFINE_MESSAGE_FIELDS(
                Interface,
                Reserved,
                Data
            );

        public:
            virtual TString DebugString() const override;
        };

        class TInterfaceResponse
            : public TPayload<INTERFACE_RESPONSE>
            , public TInterfaceData
        {
        public:
            DEFINE_MESSAGE_FIELDS(
                Interface,
                Reserved,
                Data
            );

        public:
            virtual TString DebugString() const override;
        };

        template <EMessageType M, class T>
        class TCanFrames: public TPayload<M> {
        public:
            using THeader = NProtocol::TStructure<TVega_can_data_header>;
            using TFrame = NProtocol::TStructure<T>;
            using TFrames = TVector<TFrame>;
            static_assert(sizeof(THeader) == sizeof(TVega_can_data_header));
            static_assert(sizeof(TFrame) == sizeof(T));

        public:
            virtual size_t GetSize() const override {
                return sizeof(Header) + Frames.size() * sizeof(TFrame);
            }
            virtual void Load(IInputStream& input) override {
                ::Load(&input, Header);
                Frames.resize(Header.Data.frame_count);
                ::LoadRange(&input, Frames.begin(), Frames.end());
            }
            virtual void Save(IOutputStream& output) const override {
                Header.Data.frame_count = Frames.size();
                ::Save(&output, Header);
                ::SaveRange(&output, Frames.begin(), Frames.end());
            }

        private:
            mutable THeader Header;

        public:
            TFrames Frames;
        };

        class TCanRequest: public TCanFrames<CAN_REQUEST, TVega_tx_can_frame> {
        public:
            virtual TString DebugString() const override;

            static_assert(sizeof(TVega_tx_can_frame) == 14);
        };

        class TCanResponse: public TCanFrames<CAN_RESPONSE, TVega_rx_can_frame> {
        public:
            virtual TString DebugString() const override;

            static_assert(sizeof(TVega_rx_can_frame) == 18);
        };

        class TFastDataRecords: public TPayload<FAST_DATA_RECORDS> {
        public:
            struct TRecord {
                ui8 Version                 = 0;
                ui32 DropCounter            = 0;
                ui32 TickCounter            = 0;

                i16 AccelerometerX          = 0;
                i16 AccelerometerY          = 0;
                i16 AccelerometerZ          = 0;

                // GNSS
                ui32 Timestamp              = 0;
                float Longitude             = 0;
                float Latitude              = 0;
                ui8 Valid                   = 0;
                ui8 Speed                   = 0;
                ui16 Course                 = 0;
                ui8 Active                  = 0;

                // CAN
                ui8 CanSpeed                = 0;
                ui8 Accelerator             = 0;
                ui8 Brake                   = 0;
                i16 SteeringWheel           = 0;
                ui16 Rpm                    = 0;
                ui16 SteeringAcceleration   = 0;

                static float ConvertAcceletometer(i16 raw) {
                    return (raw >> 6) / 256.0;
                }
                float GetAccelerometerX() const {
                    return ConvertAcceletometer(AccelerometerX);
                }
                float GetAccelerometerY() const {
                    return ConvertAcceletometer(AccelerometerY);
                }
                float GetAccelerometerZ() const {
                    return ConvertAcceletometer(AccelerometerZ);
                }
                float GetCourse() const {
                    return 0.1f * Course;
                }

                size_t CalcSize() const;
                size_t GetSize(ui8 version) const;
                void Save(IOutputStream* stream) const;
                void Load(IInputStream* stream);

                TInstant GetTimestamp() const {
                    return TInstant::Seconds(Timestamp) + TDuration::MilliSeconds(TickCounter);
                }
            };
            using TRecords = NProtocol::TArray<TRecord, ui8>;

        public:
            TFastDataRecords(size_t packetSize)
                : PacketSize(packetSize)
            {
            }

            TRecords Records;

            size_t GetSize() const override;
            void Save(IOutputStream& stream) const override;
            void Load(IInputStream& stream) override;

            virtual TString DebugString() const override;

        private:
            size_t PacketSize = 0;

        private:
            ui8 GetVersion(size_t packetsCount);
        };

        NDrive::TMultiSensor ToSensors(const TBlackboxRecords::TRecord& data);
    }
}
