#include "vega.h"
#include "nonce.h"

#include <rtline/library/json/cast.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/str.h>
#include <util/string/hex.h>

using namespace NDrive::NVega;
using namespace std::string_view_literals;

namespace {
    void Fill(TMessage& message) {
        switch (message.GetMessageType()) {
        case BLACKBOX_RECORDS: {
            auto& payload = message.As<TBlackboxRecords>();
            payload.Id = 10;
            auto& record = payload.Records.emplace_back();
            record.Longitude = 37.1;
            record.Lattitude = 55.5;
            record.Course = 5;
            record.Height = -1;
            record.Satelites = 14;
            record.Speed = 7;
            record.Timestamp = Seconds();
            auto& param = record.Parameters.Mutable().emplace_back();
            param.Id = 1;
            param.SetValue<ui16>(4242);
            break;
        }
        case FILE_STREAM_REQUEST: {
            TBuffer buffer;
            buffer.Fill('*', 1 << 20);
            auto& payload = message.As<TFileStreamRequest>();
            payload.Name.Set("FILE42");
            payload.Content.Set(std::move(buffer));
            break;
        }
        case FAST_DATA_RECORDS: {
            auto& payload = message.As<TFastDataRecords>();
            auto& record = payload.Records.emplace_back();
            record.Version = 1;
            record.DropCounter = 3;
            record.TickCounter = 65153;
            record.AccelerometerX = 1563;
            record.AccelerometerY = -245;
            record.AccelerometerZ = -2345;
            record.Timestamp = Now().Seconds();
            record.Longitude = 37.1;
            record.Latitude = 55.5;
            record.Valid = 1;
            record.Speed = 55;
            record.Course = 5;
            record.Active = 1;
            record.CanSpeed = 53;
            record.Accelerator = 234;
            record.Brake = 0;
            record.SteeringWheel = 38;
            record.Rpm = 1000;
            record.SteeringAcceleration = 18;
            break;
        }
        default:
            break;
        }
    }

    TMessage SerializeDeserialize(const TMessage& message) {
        UNIT_ASSERT(message.IsValid());

        TString buffer;
        TStringOutput so(buffer);
        message.Save(so);

        TMessage result;
        TStringInput si(buffer);
        result.Load(si);

        UNIT_ASSERT(result.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(result.GetMessageType(), message.GetMessageType());
        return result;
    }
}

Y_UNIT_TEST_SUITE(TelematicsSuite) {
    Y_UNIT_TEST(Serialization) {
        auto types = {
            COMMAND_REQUEST,
            COMMAND_RESPONSE,
            AUTH_REQUEST,
            AUTH_RESPONSE,
            PING_REQUEST,
            PING_RESPONSE,
            LIST_REQUEST,
            LIST_RESPONSE,
            CONNECT_REQUEST,
            CONNECT_RESPONSE,
            DISCONNECTED,
            BLACKBOX_RECORDS,
            BLACKBOX_RECORDS_ACK,
            FAST_DATA_RECORDS,
            FILE_STREAM_REQUEST,
            FILE_STREAM_RESPONSE,
            INTERFACE_REQUEST,
            INTERFACE_RESPONSE,
        };
        for (auto&& type : types) {
            TMessage message(type);
            Fill(message);

            TMessage copy = SerializeDeserialize(message);

            UNIT_ASSERT(message.IsValid());
            UNIT_ASSERT(copy.IsValid());
            UNIT_ASSERT_VALUES_EQUAL(message.GetMessageType(), type);
            UNIT_ASSERT_VALUES_EQUAL(copy.GetMessageType(), type);
            UNIT_ASSERT_VALUES_EQUAL(message.DebugString(), copy.DebugString());
        }
    }

    Y_UNIT_TEST(Blackbox) {
        const TVector<TString> packets = {
            "20001100000100D07EBF5A1DEF5E427C5B16428900000000000807003708040000000052458A36",
            "D1001100000100DF99BF5A7EE25E426A40164287000000650010B8000900044B3000000D00040000000011000101140001006A0004C3F5283F6B00043D0A973F6C0004EC51783F6D0001106E000109710004A06CAE447200010013040100DB040400006F3FDC0404000000BCDD040400008A3EE2040472F9DE41E404043E1A6741E5040461188640D10702FA00D207020200D30704141E0000D40704FEF60000D507011AD8070488B10000DB0704A70F0000DC070423050000DF0704BB0500003B0801243D0802AF023E080400007C423F080100D257BC7E",
        };
        for (auto&& packet : packets) {
            auto decoded = HexDecode(packet);
            TStringInput si(decoded);
            TMessage message(BLACKBOX_RECORDS);
            message.Load(si);

            const auto& payload = message.As<TBlackboxRecords>();
            UNIT_ASSERT_VALUES_EQUAL(payload.Records.size(), 1);
            const auto& record = payload.Records[0];
            UNIT_ASSERT(record.Timestamp > 1522494755);
            UNIT_ASSERT(record.Lattitude > 55);
            UNIT_ASSERT(record.Longitude > 37);
        }
    }

    Y_UNIT_TEST(BlackboxParameter) {
        {
            TBlackboxRecords::TParameter param;
            UNIT_ASSERT(std::holds_alternative<TNull>(param.GetValue()));
        }
        {
            TBlackboxRecords::TParameter param;
            param.Id = VEGA_SETTING_USE_SERVER_PIN;
            param.SetValue<ui8>(42);
            UNIT_ASSERT(std::holds_alternative<ui64>(param.GetValue()));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(param.GetValue()), 42);
        }
        {
            TBlackboxRecords::TParameter param;
            param.Id = VEGA_INPUT_BUFFER_SIZE;
            param.SetValue<ui16>(11050);
            UNIT_ASSERT(std::holds_alternative<ui64>(param.GetValue()));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(param.GetValue()), 11050);
        }
        {
            TBlackboxRecords::TParameter param;
            param.SetValue(TStringBuf("42\0"sv));
            UNIT_ASSERT(std::holds_alternative<TStringBuf>(param.GetValue()));
            UNIT_ASSERT_VALUES_EQUAL(std::get<TStringBuf>(param.GetValue()), "42");
        }
        {
            TBlackboxRecords::TParameter param;
            param.Id = VEGA_TRIP_COUNTER;
            param.SetValue<ui32>(256 * 256 * 256);
            UNIT_ASSERT(std::holds_alternative<ui64>(param.GetValue()));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(param.GetValue()), 256 * 256 * 256);
        }
    }

    Y_UNIT_TEST(BlackboxTyped) {
        const TVector<TString> packets = {
            "2B001100000100906F89603FF45E426D0616428700090099001112806A00040448E13A3F1900010102F90A070100A404CB84",
        };
        for (auto&& packet : packets) {
            auto decoded = HexDecode(packet);
            TStringInput si(decoded);
            TMessage message(BLACKBOX_RECORDS);
            message.Load(si);

            const auto& payload = message.As<TBlackboxRecords>();
            UNIT_ASSERT_VALUES_EQUAL(payload.Records.size(), 1);
            const auto& record = payload.Records[0];
            UNIT_ASSERT(record.Timestamp > 1522494755);
            UNIT_ASSERT(record.Lattitude > 55);
            UNIT_ASSERT(record.Longitude > 37);
            UNIT_ASSERT(record.Parameters.IsTyped());
            UNIT_ASSERT(!record.Parameters.IsEmpty());
        }
    }

    Y_UNIT_TEST(SetParameter) {
        TMessage message(COMMAND_REQUEST);
        TCommandRequest& request = message.As<TCommandRequest>();
        request.Code = ECommandCode::SET_PARAM;

        TCommandRequest::TSetParameter argument;
        argument.Id = 42;
        argument.SubId = 1;
        {
            UNIT_ASSERT(std::holds_alternative<TNull>(argument.GetValue()));
            ui8 value = 42;
            argument.SetValue(value);
            request.Argument.Set(argument);
            auto copy = SerializeDeserialize(message);
            auto& r = copy.As<TCommandRequest>();
            UNIT_ASSERT_VALUES_EQUAL(r.Code, request.Code);

            auto arg = r.Argument.Get<TCommandRequest::TSetParameter>();
            UNIT_ASSERT_VALUES_EQUAL(arg.Id, argument.Id);
            UNIT_ASSERT_VALUES_EQUAL(arg.SubId, argument.SubId);

            auto val = arg.GetValue();
            UNIT_ASSERT(std::holds_alternative<ui64>(val));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(val), value);
        }
        {
            ui16 value = 4242;
            argument.SetValue(value);
            request.Argument.Set(argument);
            auto copy = SerializeDeserialize(message);
            auto& r = copy.As<TCommandRequest>();
            auto arg = r.Argument.Get<TCommandRequest::TSetParameter>();
            auto val = arg.GetValue();
            UNIT_ASSERT(std::holds_alternative<ui64>(val));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(val), value);
        }
        {
            ui32 value = 42424242;
            argument.Id = VEGA_TRIP_COUNTER;
            argument.SetValue(value);
            request.Argument.Set(argument);
            auto copy = SerializeDeserialize(message);
            auto& r = copy.As<TCommandRequest>();
            auto arg = r.Argument.Get<TCommandRequest::TSetParameter>();
            auto val = arg.GetValue();
            UNIT_ASSERT(std::holds_alternative<ui64>(val));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(val), value);
        }
        {
            TStringBuf value = "it's a dreich day, waddayasay?";
            argument.Id = 42;
            argument.SetValue(value);
            request.Argument.Set(argument);
            auto copy = SerializeDeserialize(message);
            auto& r = copy.As<TCommandRequest>();
            auto arg = r.Argument.Get<TCommandRequest::TSetParameter>();
            auto val = arg.GetValue();
            UNIT_ASSERT(std::holds_alternative<TStringBuf>(val));
            UNIT_ASSERT_VALUES_EQUAL(std::get<TStringBuf>(val), value);
        }
        argument.Id = VEGA_LON;
        argument.SubId = 0;
        {
            double value = 35.55;
            argument.SetValue(value);
            request.Argument.Set(argument);
            auto copy = SerializeDeserialize(message);
            auto& r = copy.As<TCommandRequest>();
            auto arg = r.Argument.Get<TCommandRequest::TSetParameter>();
            auto val = arg.GetValue();
            UNIT_ASSERT(std::holds_alternative<double>(val));
            UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(val), value, 0.001);
        }
    }

    Y_UNIT_TEST(ZeroTerminatedString) {
        using namespace NDrive::NProtocol;
        TZeroTerminatedString s;
        s.Set("4224");
        UNIT_ASSERT_VALUES_EQUAL(s.Get(), "4224");

        TArgument argument;
        argument.Set(s);
        UNIT_ASSERT_VALUES_EQUAL(argument.CalcSize(), 6);

        TZeroTerminatedString rs = argument.Get<TZeroTerminatedString>();
        UNIT_ASSERT_VALUES_EQUAL(rs.Get(), s.Get());
        UNIT_ASSERT_VALUES_EQUAL(rs.Get(), "4224");
    }

    Y_UNIT_TEST(LeftStrlen) {
        TString source = "enjoy";
        source[2] = '\0';
        auto value = NDrive::NVega::TValue::Get(source, VEGA_GSM_FIRMWARE_VERSION);
        UNIT_ASSERT(std::holds_alternative<TStringBuf>(value));
        UNIT_ASSERT_VALUES_EQUAL(std::get<TStringBuf>(value), "en");
    }

    Y_UNIT_TEST(CanRequestResponse) {
        NDrive::NVega::TCanRequest request;
        for (size_t i = 0; i < 2; ++i) {
            NDrive::NVega::TCanRequest::TFrame frame;
            frame.Data.id = i;
            frame.Data.itf_idx = 2;
            frame.Data.id_type = 1;
            frame.Data.dlen = 1;
            frame.Data.data[0] = 42;
            request.Frames.push_back(frame);
        }
        {
            auto predictedSize = request.GetSize();
            TString serialized;
            TStringOutput so(serialized);
            request.Save(so);
            UNIT_ASSERT_VALUES_EQUAL(serialized.size(), predictedSize);
            Cout << HexEncode(serialized) << Endl;

            NDrive::NVega::TCanRequest::THeader header;
            NDrive::NVega::TCanRequest::TFrame frame0;
            NDrive::NVega::TCanRequest::TFrame frame1;
            TStringInput si(serialized);
            header.Load(&si);
            frame0.Load(&si);
            frame1.Load(&si);
            UNIT_ASSERT_VALUES_EQUAL(header.Data.frame_count, 2);
            UNIT_ASSERT_VALUES_EQUAL(frame0.Data.id, 0);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.id, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.itf_idx, 2);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.id_type, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.dlen, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.data[0], 42);
        }

        NDrive::NVega::TCanResponse response;
        for (size_t i = 0; i < 2; ++i) {
            NDrive::NVega::TCanResponse::TFrame frame;
            frame.Data.id = i;
            frame.Data.itf_idx = 2;
            frame.Data.id_type = 1;
            frame.Data.dlen = 1;
            frame.Data.data[0] = 42;
            response.Frames.push_back(frame);
        }
        {
            auto predictedSize = response.GetSize();
            TString serialized;
            TStringOutput so(serialized);
            response.Save(so);
            UNIT_ASSERT_VALUES_EQUAL(serialized.size(), predictedSize);
            Cout << HexEncode(serialized) << Endl;

            NDrive::NVega::TCanResponse::THeader header;
            NDrive::NVega::TCanResponse::TFrame frame0;
            NDrive::NVega::TCanResponse::TFrame frame1;
            TStringInput si(serialized);
            header.Load(&si);
            frame0.Load(&si);
            frame1.Load(&si);
            UNIT_ASSERT_VALUES_EQUAL(header.Data.frame_count, 2);
            UNIT_ASSERT_VALUES_EQUAL(frame0.Data.id, 0);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.id, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.itf_idx, 2);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.id_type, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.dlen, 1);
            UNIT_ASSERT_VALUES_EQUAL(frame1.Data.data[0], 42);
        }
    }

    Y_UNIT_TEST(ToString) {
        Cout << NDrive::NVega::ECommandCode::AUXILIARY_CAR_COMMAND << Endl;
        Cout << NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND << Endl;
        Cout << FromString<NDrive::NVega::ECommandCode>("AUXILIARY_CAR_COMMAND") << Endl;
        Cout << FromString<NDrive::NVega::ECommandCode>("ELECTRIC_CAR_COMMAND") << Endl;
    }

    template <size_t N>
    struct TTestArray {
        std::array<char, N> Value;

        DEFINE_FIELDS(Value);
    };

    Y_UNIT_TEST(ArgumentFromJson) {
        {
            auto serialized = TString("BC0B4508");
            auto value = NJson::TJsonValue(serialized);
            auto argument = NJson::FromJson<NDrive::NVega::TArgument>(value);
            auto data = argument.Get<TTestArray<4>>();
            UNIT_ASSERT_VALUES_EQUAL(HexEncode(data.Value.data(), data.Value.size()), serialized);
        }
        {
            TTestArray<7> original;
            original.Value = {{ 42, 43, 11, 10 }};
            NDrive::NVega::TArgument argument;
            argument.Set(original);

            auto serialized = NJson::ToJson(argument);
            auto deserialized = NJson::FromJson<NDrive::NVega::TArgument>(serialized);
            auto roundtripped = deserialized.Get<decltype(original)>();
            UNIT_ASSERT_VALUES_EQUAL(original.Value.size(), roundtripped.Value.size());
            for (size_t i = 0; i < original.Value.size(); ++i) {
                UNIT_ASSERT_VALUES_EQUAL(original.Value[i], roundtripped.Value[i]);
            }
        }
    }

    Y_UNIT_TEST(SignedCommand) {
        auto request = NJson::FromJson<NDrive::NVega::TCommandRequestSigned>(
            "9521d55ec26d68a7010000002800"
        );
        auto response = NJson::FromJson<NDrive::NVega::TCommandResponseSigned>(
            "df05ab2410dd5bb64cc02233aa222c0d9521d55ec26d68a7010000000112656e64206f66206c6561736520646f6e6500"
        );
        Cerr << NJson::ToJson(request).GetStringRobust() << Endl;
        Cerr << NJson::ToJson(response).GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(request.Code, NDrive::NVega::ECommandCode::YADRIVE_END_OF_LEASE);
        UNIT_ASSERT_VALUES_EQUAL(request.Id, response.Id);
        UNIT_ASSERT_VALUES_EQUAL(request.Nonce, response.Nonce);
        UNIT_ASSERT_VALUES_EQUAL(response.Result, NDrive::NVega::TCommandResponse::PROCESSED);
        /*
        auto key = NPoly1305::TryParseKey("");
        UNIT_ASSERT(key);
        UNIT_ASSERT(response.ValidateMac(*key));
        */
        auto compositeNonce = NDrive::Decode(request.Nonce);
        Cerr << compositeNonce.Timestamp.Seconds() << Endl;
        Cerr << compositeNonce.Hash << Endl;
    }

    Y_UNIT_TEST(FastDataVersion) {
        TMessage message(FAST_DATA_RECORDS);
        Fill(message);
        {
            auto& payload = message.As<TFastDataRecords>();
            UNIT_ASSERT(payload.Records.size());
            auto& record = payload.Records[0];

            record.Version = 0;
            record.SteeringAcceleration = 23;

            message = SerializeDeserialize(message);
        }

        {
            auto& payload = message.As<TFastDataRecords>();
            UNIT_ASSERT(payload.Records.size());
            auto& record = payload.Records[0];
            UNIT_ASSERT_EQUAL(record.Version, 0);
            UNIT_ASSERT_EQUAL(record.SteeringAcceleration, 0);
        }

        {
            auto& payload = message.As<TFastDataRecords>();
            auto& record = payload.Records[0];

            record.Version = 1;
            record.SteeringAcceleration = 23;

            message = SerializeDeserialize(message);
        }

        {
            auto& payload = message.As<TFastDataRecords>();
            auto& record = payload.Records[0];

            UNIT_ASSERT_EQUAL(record.Version, 1);
            UNIT_ASSERT_EQUAL(record.SteeringAcceleration, 23);
        }
    }

    Y_UNIT_TEST(FastDataIncorrectVersion) {
        // in this case Version == 0
        TString data = HexDecode(
            "A5001F040000000000D6713E35CFD516" \
            "E1F5DA58A64662CF9B164281FB5E4201" \
            "00BE0301000000000000000000000000" \
            "00003A723E3509D6C8E19DDB58A64662" \
            "CF9B164281FB5E420100BE0301000000" \
            "00000000000000000000009E723E35DA" \
            "D533E15ADB58A64662CF9B164281FB5E" \
            "420100BE030100000000000000000000" \
            "0000000002733E354CD6EEE02FDB58A6" \
            "4662CF9B164281FB5E420100BE030100" \
            "00000000000000003D5281AD"
        );
        TStringInput stream(data);
        auto message = MakeHolder<NDrive::NVega::TMessage>();
        UNIT_ASSERT_NO_EXCEPTION(message->Load(stream));

        auto& payload = message->As<NDrive::NVega::TFastDataRecords>();
        UNIT_ASSERT(!payload.Records.empty());

        const auto& record = payload.Records.front();
        UNIT_ASSERT_EQUAL(record.Version, 1);
    }
}
