#include "crc.h"
#include "vega.h"
#include "wialon.h"

#include <util/string/hex.h>
#include <util/string/split.h>
#include <util/generic/serialized_enum.h>
#include <utility>

namespace {
    TMap<TStringBuf, NDrive::NWialon::EMessageType> PrefixCastValue = {
        { "L", NDrive::NWialon::MT_LOGIN_REQUEST },
        { "AL", NDrive::NWialon::MT_LOGIN_ANSWER },
        { "D", NDrive::NWialon::MT_DATA_REQUEST },
        { "AD", NDrive::NWialon::MT_DATA_ANSWER },
        { "P", NDrive::NWialon::MT_PING_REQUEST },
        { "AP", NDrive::NWialon::MT_PING_ANSWER },
        { "SD", NDrive::NWialon::MT_SHORT_DATA_REQUEST },
        { "ASD", NDrive::NWialon::MT_SHORT_DATA_ANSWER },
        { "B", NDrive::NWialon::MT_BLACK_BOX_REQUEST },
        { "AB", NDrive::NWialon::MT_BLACK_BOX_ANSWER },
        { "UC", NDrive::NWialon::MT_SETTING_FILE_REQUEST },
    };

    NDrive::NWialon::EMessageType GetMessageTypeByPrefix(TStringBuf prefix) {
        if (PrefixCastValue.contains(prefix)) {
            return PrefixCastValue[prefix];
        }
        return NDrive::NWialon::EMessageType::MT_INCORRECT;
    }

    template<class K, class V>
    TMap<V, K> InvertMap(const TMap<K, V>& input) {
        TMap<V, K> result;

        for (const auto& pair : input) {
            result[pair.second] = pair.first;
        }

        return result;
    }

    struct TMessageTypeByPrefix {
        using TValue = TMap<NDrive::NWialon::EMessageType, TStringBuf>;

        TMessageTypeByPrefix()
            : Value(InvertMap(PrefixCastValue))
        {
        }

        TValue Value;
    };

    TStringBuf GetPrefixByMessageType(NDrive::NWialon::EMessageType type) {
        auto& instance = Singleton<TMessageTypeByPrefix>()->Value;

        if (instance.contains(type)) {
            return instance[type];
        }
        return "";
    }

    THolder<NDrive::NProtocol::IPayload> CreatePayload(NDrive::NWialon::EMessageType messageType) {
        using namespace NDrive::NProtocol;
        using namespace NDrive::NWialon;

        switch (messageType) {
        case NDrive::NWialon::MT_LOGIN_REQUEST:
            return IPayload::Create<TLoginRequest>();
        case NDrive::NWialon::MT_LOGIN_ANSWER:
            return IPayload::Create<TLoginAnswer>();
        case NDrive::NWialon::MT_PING_REQUEST:
            return IPayload::Create<TPingRequest>();
        case NDrive::NWialon::MT_PING_ANSWER:
            return IPayload::Create<TPingAnswer>();
        case NDrive::NWialon::MT_SHORT_DATA_REQUEST:
            return IPayload::Create<TShortDataRequest>();
        case NDrive::NWialon::MT_SHORT_DATA_ANSWER:
            return IPayload::Create<TShortDataAnswer>();
        case NDrive::NWialon::MT_DATA_REQUEST:
            return IPayload::Create<TDataRequest>();
        case NDrive::NWialon::MT_DATA_ANSWER:
            return IPayload::Create<TDataAnswer>();
        case NDrive::NWialon::MT_BLACK_BOX_REQUEST:
            return IPayload::Create<TBlackBoxRequest>();
        case NDrive::NWialon::MT_BLACK_BOX_ANSWER:
            return IPayload::Create<TBlackBoxAnswer>();
        case NDrive::NWialon::MT_INCORRECT:
        default:
            return nullptr;
        };
    }

    bool IsChecksumPresent(NDrive::NWialon::EMessageType type) {
        using namespace NDrive::NWialon;

        switch (type) {
            case MT_PING_REQUEST:
            case MT_PING_ANSWER:
            case MT_LOGIN_ANSWER:
            case MT_SHORT_DATA_ANSWER:
            case MT_DATA_ANSWER:
            case MT_BLACK_BOX_REQUEST:
            case MT_BLACK_BOX_ANSWER:
            case MT_SETTING_FILE_REQUEST:
                return false;
            default:
                return true;
        }
    }

    TString GenerateChecksum(const char* begin, const char* end) {
        TString result;

        ui16 checksum = NDrive::WialonCrc16(begin, end);
        TBuffer buffer(reinterpret_cast<const char*>(&checksum), sizeof(checksum));
        Reverse(buffer.Begin(), buffer.End());
        result = HexEncode(buffer.Data(), buffer.Size());

        return result;
    }
}

NDrive::NWialon::TMessage::TMessage(EMessageType type/* = INCORRECT*/) {
    Payload = CreatePayload(type);
}

NDrive::NWialon::TMessage::TMessage(THolder<NProtocol::IPayload>&& payload) {
    Payload = std::move(payload);
}

void NDrive::NWialon::TMessage::Load(IInputStream& input) {
    const size_t minimalParametersCount = 3;
    const int prefixIndex = 1;
    const int payloadIndex = 2;

    TString message = input.ReadLine();

    if (message.Empty()) {
        return;
    }

    if (!message.StartsWith("#")) {
        ythrow yexception() << "invalid start char (#)";
    }

    TVector<TString> messageSplit;
    StringSplitter(std::move(message)).Split('#').AddTo(&messageSplit);

    if (messageSplit.size() < minimalParametersCount) {
        ythrow yexception() << "invalid message format";
    }

    TString messagePrefix = std::move(messageSplit[prefixIndex]);
    TString messagePayload = std::move(messageSplit[payloadIndex]);

    auto messageType = GetMessageTypeByPrefix(messagePrefix);
    Y_ENSURE(messageType != MT_INCORRECT);
    auto localMessageType = static_cast<NWialon::EMessageType>(messageType);
    TStringInput payloadStream(messagePayload);

    try {
        Y_ENSURE(GetEnumNames<NWialon::EMessageType>().contains(localMessageType), "bad type " << messageType);
        Payload = CreatePayload(localMessageType);
        Y_ENSURE(Payload);
        Payload->Load(payloadStream);
    } catch (const std::exception& e) {
        ythrow yexception() << "cannot deserialize "
            << messagePrefix << " "
            << messagePayload << ": "
            << FormatExc(e);
    }

    if (IsChecksumPresent(localMessageType)) {
        ui16 checksum = 0;
        SaveLoad(&payloadStream, checksum);
    }

    Payload->PostLoad(input);
}

void NDrive::NWialon::TMessage::Save(IOutputStream& output) const {
    Y_ENSURE(Payload);
    auto localMessageType = static_cast<EMessageType>(Payload->GetMessageType());
    auto prefix = ToString(GetPrefixByMessageType(localMessageType));

    output.Write("#");
    output.Write(prefix);
    output.Write("#");

    TString payloadData;
    TStringOutput stream(payloadData);
    Payload->Save(stream);

    if (IsChecksumPresent(static_cast<NWialon::EMessageType>(GetMessageType()))) {
        stream.Write(GenerateChecksum(payloadData.begin(), payloadData.end()));
    }

    output.Write(payloadData);
    output.Write(EndOfPacket);
    Payload->PostSave(output);
}

void NDrive::NWialon::TLoginAnswer::TAnswer::Load(IInputStream *input) {
    Y_UNUSED(input);
}

void NDrive::NWialon::TLoginAnswer::TAnswer::Save(IOutputStream *output) const {
    switch(Value) {
        case Success:
            output->Write("1");
            break;
        case Reject:
            output->Write("0");
            break;
        case PasswordError:
            output->Write("01");
            break;
        case ChecksumError:
            output->Write("10");
            break;
    }
}

TString NDrive::NWialon::TShortData::DebugString() const {
    TString result;

    result += DateTime.Get() ? ToString(DateTime.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Lattitude.Get() ? ToString(Lattitude.Get().GetRef().GetValue()) : ToString(EmptyValue);
    result += " ";
    result += Lattitude2.Get() ? ToString(Lattitude2.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Longitude.Get() ? ToString(Longitude.Get().GetRef().GetValue()) : ToString(EmptyValue);
    result += " ";
    result += Longitude2.Get() ? ToString(Longitude2.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Speed.Get() ? ToString(Speed.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Course.Get() ? ToString(Course.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Height.Get() ? ToString(Height.Get().GetRef()) : ToString(EmptyValue);
    result += " ";
    result += Satellites.Get() ? ToString(Satellites.Get().GetRef()) : ToString(EmptyValue);

    return result;
}

TString NDrive::NWialon::TAdditionalData::DebugString() const {
    TString result;

    result += Hdop.Get() ? ToString(Hdop.Get().GetRef()) : EmptyValue;
    result += " ";
    result += Inputs.Get() ? ToString(Inputs.Get().GetRef()) : EmptyValue;
    result += " ";
    result += Outputs.Get() ? ToString(Outputs.Get().GetRef()) : EmptyValue;
    result += " ";

    for (auto&& item : Analogs.Get()) {
        result += ToString(item.Get().GetRef());
        result += " ";
    }

    for (auto&& param : Params.Get()) {
        result += "[" + ToString(param.first) + "]=";

        if (std::holds_alternative<ui64>(param.second)) {
            result += ToString(std::get<ui64>(param.second));
        } else if (std::holds_alternative<double>(param.second)) {
            result += ToString(std::get<double>(param.second));
        } else {
            result += std::get<TString>(param.second);
        }

        result += " ";
    }

    return result;
}

TString NDrive::NWialon::TDataRequest::DebugString() const {
    return "Data Request " + ShortData.DebugString() + " " + AdditionalData.DebugString();
}

void NDrive::NWialon::TAdditionalData::TParams::Load(IInputStream* input) {
    auto rawParams = input->ReadTo(';');

    if (rawParams.Empty() || rawParams == EmptyValue) {
        return;
    }

    TStringInput stream(rawParams);
    ParseParams(&stream);
}

void NDrive::NWialon::TAdditionalData::TParams::ParseParams(IInputStream* input) {
    TString rawParams = input->ReadAll();

    TVector<TStringBuf> params = StringSplitter(rawParams).Split(',');

    for (auto&& rawParam : params) {
        if (rawParam.Empty()) {
            continue;
        }

        const char localSeparator = ':';
        TVector<TStringBuf> paramValues = StringSplitter(rawParam).Split(localSeparator);

        if (paramValues.size() < 3) {
            continue;
        }

        TString id = ToString(paramValues[0]);
        auto type = FromString<ui8>(paramValues[1]);
        TString value = ToString(paramValues[2]);

        switch (static_cast<EType>(type)) {
            case Integer:
                Value[id] = FromString<ui64>(value);
                break;
            case Double:
                Value[id] = FromString<double>(value);
                break;
            case String:
                Value[id] = value;
                break;
            case Incorrect:
            default:
                break;
        }
    }
}

void NDrive::NWialon::TAdditionalData::TParams::Save(IOutputStream* output) const {
    // TODO implement save function
    Y_UNUSED(output);
}

void NDrive::NWialon::TBlackBoxRequest::TCollection::Load(IInputStream* input) {
    TString rawPackets = input->ReadAll();

    if (rawPackets.Empty()) {
        return;
    }

    TVector<TString> packets = StringSplitter(rawPackets).Split(GetSeparator());
    // remove unused checksum
    packets.pop_back();

    for (auto&& packet : packets) {
        TItem item;
        TStringInput stream(packet);
        item.Load(stream);
        Collection.push_back(item);
    }
}

void NDrive::NWialon::TBlackBoxRequest::TCollection::Save(IOutputStream* output) const {
    // TODO implement save function
    Y_UNUSED(output);
}

TString NDrive::NWialon::TBlackBoxRequest::TCollection::DebugString() const {
    TString result;

    for (auto&& item : Collection) {
        result += item.DebugString();
        result += " ";
    }

    return result;
}

void NDrive::NWialon::TSettingFileRequest::Load(IInputStream& input) {
    // TODO implement load function
    Y_UNUSED(input);
}

void NDrive::NWialon::TSettingFileRequest::Save(IOutputStream& output) const {
    auto data = ToString(Data.Size());
    output.Write(data);
    ::Save(&output, ';');
    output.Write(GenerateChecksum(Data.Begin(), Data.End()));
}

void NDrive::NWialon::TSettingFileRequest::PostSave(IOutputStream& output) const {
    ::SaveRange(&output, Data.Begin(), Data.End());
}

TString NDrive::NWialon::TSettingFileRequest::DebugString() const {
    return ToString(GetMessageTypeAs<EMessageType>()) + " " + HexEncode(Data.Data(), Data.Size());
}

namespace {
    template<typename T, class F>
    void Apply(ui32 id, const auto& value, const F& add) {
        auto v = value.Get();
        if (v) {
            add(id, static_cast<T>(*v));
        }
    }
}

NDrive::TMultiSensor NDrive::NWialon::ToSensors(const TShortData& data) {
    NDrive::TMultiSensor sensors;
    const auto& dataTimestamp = data.DateTime.Get();
    auto timestamp = dataTimestamp.GetOrElse(TInstant::Now());

    auto add = [&sensors, timestamp] (ui32 id, auto value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };

    Apply<double>(VEGA_LAT, data.Lattitude, add);
    Apply<double>(VEGA_LON, data.Longitude, add);
    Apply<double>(VEGA_ALT, data.Height, add);
    Apply<double>(VEGA_SPEED, data.Speed, add);
    Apply<double>(VEGA_DIR, data.Course, add);
    Apply<ui64>(VEGA_SAT_USED, data.Satellites, add);

    return sensors;
}

NDrive::TMultiSensor NDrive::NWialon::ToSensors(const TAdditionalData& data, TInstant timestamp/* = Now()*/) {
    NDrive::TMultiSensor sensors;

    auto add = [&sensors, timestamp] (ui32 id, auto value) {
        NDrive::TSensor sensor;
        sensor.Id = id;
        sensor.SubId = 0;
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        sensors.push_back(std::move(sensor));
    };

    const auto& params = data.Params.Get();
    add(VEGA_MNC, std::get<ui64>(params.Value("mnc", ui64(0))));
    add(VEGA_MCC, std::get<ui64>(params.Value("mcc", ui64(0))));
    add(VEGA_LAC, std::get<ui64>(params.Value("lac", ui64(0))));
    add(VEGA_CELLID, std::get<ui64>(params.Value("cell_id", ui64(0))));

    auto rawVoltageLevel = std::get<TString>(params.Value("bat", "0"));
    SubstGlobal(rawVoltageLevel, "%", "");
    double voltageLevel = FromString(rawVoltageLevel);
    add(CAN_BATTERY_CHARGE_LEVEL, voltageLevel);

    auto rawSignalLevel = std::get<TString>(params.Value("sig", "0/4"));
    ui64 signalLevel = 0;
    if (rawSignalLevel == "1/4") {
        signalLevel = 1;
    } else if (rawSignalLevel == "2/4") {
        signalLevel = 9;
    } else if (rawSignalLevel == "3/4") {
        signalLevel = 18;
    } else if (rawSignalLevel == "4/4") {
        signalLevel = 27;
    }

    add(VEGA_GSM_SIGNAL_LEVEL, signalLevel);

    double temperature = FromString(std::get<TString>(params.Value("temp", "0")));
    add(VEGA_INT_TEMP, temperature);

    auto mode = std::get<TString>(params.Value("mode", "NONE"));
    ui64 resultMode = 0;
    if (mode == "SUT") {
        resultMode = 1;
    } else if (mode == "TEST") {
        resultMode = 2;
    } else if (mode == "POISK") {
        resultMode = 3;
    }
    add(VEGA_OPERATION_MODE, resultMode);

    ui64 satellitesInview = FromString(std::get<TString>(params.Value("sat", "0")));
    add(VEGA_TOTAL_SAT_INVIEW, satellitesInview);

    return sensors;
}
