#pragma once

#include "common.h"
#include "protocol.h"
#include "sensor.h"
#include "types.h"
#include "wialon_private.h"

#include <util/datetime/base.h>
#include <util/datetime/parser.h>
#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/string/split.h>

namespace NDrive {
    namespace NWialon {
        enum EMessageType: ui8 {
            MT_INCORRECT = 0 /* "incorrect" */,
            MT_LOGIN_REQUEST /* "login request" */,
            MT_LOGIN_ANSWER /* "login answer" */,
            MT_DATA_REQUEST /* "data request" */,
            MT_DATA_ANSWER /* "data answer" */,
            MT_PING_REQUEST /* "ping request" */,
            MT_PING_ANSWER /* "ping answer" */,
            MT_SHORT_DATA_REQUEST /* "short data request" */,
            MT_SHORT_DATA_ANSWER /* "short data answer" */,
            MT_BLACK_BOX_REQUEST /* "black box request" */,
            MT_BLACK_BOX_ANSWER /* "black box answer" */,
            MT_SETTING_FILE_REQUEST /* "setting file request" */
        };

        constexpr TStringBuf EmptyValue = "NA";
        constexpr TStringBuf EndOfPacket = "\r\n";

        template<typename TEnum, typename TBase = i8>
        class TEnumValue {
        public:
            TEnumValue() = default;
            TEnumValue(TEnum value)
                : Value(value)
            {
            }

            void Load(IInputStream* input) {
                Y_UNUSED(input);
            }
            void Save(IOutputStream* output) const {
                output->Write(ToString(static_cast<TBase>(Value)));
            }

            void Set(TEnum value) {
                Value = value;
            }
            TEnum Get() const {
                return Value;
            }

        private:
            TEnum Value;
        };

        class TCoordinate {
        public:
            explicit TCoordinate(float value)
                : Value(value)
            {
            }

            float GetValue() const {
                return Value;
            }

            explicit operator float() const {
                return Value;
            }
            explicit operator double() const {
                return Value;
            }

        private:
            float Value;
        };

        template<class TValue, char Separator>
        class TSeparatedValue {
        public:
            inline TSeparatedValue() = default;
            inline TSeparatedValue(TValue&& value)
                : Value(std::move(value))
            {
            }

            inline static char GetSeparator() {
                return Separator;
            }

            void Load(IInputStream* input) {
                TString result = input->ReadTo(GetSeparator());
                if (result != EmptyValue && !result.Empty()) {
                    Value = FromString<TValue>(result);
                }
            }
            void Save(IOutputStream* output) const {
                if (Value) {
                    output->Write(Value.GetRef());
                } else {
                    output->Write(EmptyValue);
                }
            }

            const TMaybe<TValue>& Get() const {
                return Value;
            }
            void Set(TValue&& value) {
                Value = std::move(value);
            }

        private:
            TMaybe<TValue> Value;
        };

        template<class TValue, char Separator>
        class TSeparatedCollection {
        public:
            using TCollection = TVector<TValue>;

            inline TSeparatedCollection() = default;
            inline TSeparatedCollection(TCollection&& collection)
                : Collection(std::move(collection))
            {
            }

            char GetSeparator() const {
                return Separator;
            }

            void Load(IInputStream* input) {
                if (!input) {
                    ythrow yexception() << "receive null input stream";
                }

                TString rawPacket = input->ReadTo(GetSeparator());

                if (rawPacket.Empty()) {
                    return;
                }

                TVector<TString> packets = StringSplitter(rawPacket).Split(TValue::GetSeparator());
                Collection.reserve(packets.size());

                for (auto&& item : packets) {
                    TValue value;
                    TStringInput stream(item);
                    value.Load(&stream);
                    Collection.push_back(value);
                }
            }

            void Save(IOutputStream* output) const {
                Y_UNUSED(output);
            }

            const TCollection& Get() const {
                return Collection;
            }

        private:
            TCollection Collection;
        };

        template<char Separator>
        class TSeparatedValue<TInstant, Separator> {
        public:
            using TValue = TInstant;

            inline TSeparatedValue() = default;
            inline TSeparatedValue(TValue&& value)
                : Value(std::move(value))
            {
            }

            char GetSeparator() const {
                return Separator;
            }

            void Load(IInputStream* input) {
                auto date = input->ReadTo(GetSeparator());
                auto time = input->ReadTo(GetSeparator());
                TString dateTime;

                if (date == EmptyValue && time == EmptyValue) {
                    dateTime = EmptyValue;
                } else if (date == EmptyValue) {
                    dateTime = "010100" + time;
                } else if (time == EmptyValue) {
                    dateTime = date + "000000";
                } else {
                    dateTime = date + time;
                }

                if (dateTime != EmptyValue) {
                    Value = ParseDateTime(std::move(dateTime));
                }
            }
            void Save(IOutputStream* output) const {
                Y_UNUSED(output);
            }

            const TMaybe<TValue>& Get() const {
                return Value;
            }
            void Set(TValue&& value) {
                Value = std::move(value);
            }

        private:
            TMaybe<TValue> Value;
        private:
            TValue ParseDateTime(TString&& input) {
                const size_t minimalSize = 12;
                if (input.Size() < minimalSize) {
                    ythrow yexception() << "Invalid input date and time string" << input;
                }

                const char* data = input.Data();
                TDateTimeFields field;

                field.Day = FromString(data, 2);
                data += 2;
                field.Month = FromString(data, 2);
                data += 2;
                field.SetLooseYear(FromString(data, 2));
                data += 2;
                field.Hour = FromString(data, 2);
                data += 2;
                field.Minute = FromString(data, 2);
                data += 2;
                field.Second = FromString(data, 2);
                data += 2;

                Y_ENSURE(field.IsOk());
                return field.ToInstant(TInstant::Now());
            }
        };

        template<char Separator>
        class TSeparatedValue<TCoordinate, Separator> {
        public:
            using TValue = TCoordinate;

            inline TSeparatedValue() = default;
            inline TSeparatedValue(TValue&& value)
                : Value(std::move(value))
            {
            }

            char GetSeparator() const {
                return Separator;
            }

            void Load(IInputStream* input) {
                TString result = input->ReadTo(GetSeparator());
                if (result != EmptyValue && !result.Empty()) {
                    const char dot = '.';
                    if (result.Contains(dot)) {
                        auto index = result.find(dot);
                        TVector<TStringBuf> split = StringSplitter(result).Split(dot).SkipEmpty();

                        if (split.size() < 2) {
                            return;
                        }

                        result = TString::Join(split[0], split[1]);
                        result.insert(index - 2, 1, dot);
                    }

                    Value = TValue(FromString<float>(result));
                }
            }
            void Save(IOutputStream* output) const {
                if (Value) {
                    float coordinate = float(Value.GetRef());
                    output->Write(ToString(coordinate));
                    output->Write(GetSeparator());
                }
            }

            const TMaybe<TValue>& Get() const {
                return Value;
            }
            void Set(TValue&& value) {
                Value = std::move(value);
            }

        private:
            TMaybe<TValue> Value;
        };

        template<class TValue>
        using TSeparated = TSeparatedValue<TValue, ';'>;
        using TSeparatedString = NProtocol::TSeparatedString<';'>;

        class TMessage: public NProtocol::IMessage {
        public:
            using TBaseMessageType = NProtocol::TMessageType;
            using TMessageType = EMessageType;

            TMessage(EMessageType type = MT_INCORRECT);
            TMessage(THolder<NProtocol::IPayload>&& payload);

            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
        };

        template<EMessageType Type>
        class TPayload: public NProtocol::IPayload {
        public:
            TPayload() {
                MessageType = Type;
                ProtocolType = NDrive::NProtocol::PT_WIALON_IPS;
            }

            NProtocol::TSequenceId GetSequenceId() const override {
                return 0;
            }

            size_t GetSize() const override {
                return 0;
            }
        };

        class TLoginRequest: public TPayload<MT_LOGIN_REQUEST> {
        public:
            TSeparatedString Version;
            TSeparatedString IMEI;
            TSeparatedString Password;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Version,
                IMEI,
                Password
            );

            TString DebugString() const override {
                return "Login Request " +
                        Version.Get() + " " +
                        IMEI.Get();
            }
        };

        class TLoginAnswer: public TPayload<MT_LOGIN_ANSWER> {
        public:
            enum EAnswer {
                Success = 0 /* "success" */,
                Reject /* "reject" */,
                PasswordError /* "password error" */,
                ChecksumError /* "checksum error" */
            };

            class TAnswer {
            public:
                inline TAnswer() = default;
                inline TAnswer(EAnswer value)
                    : Value(value)
                {
                }

                void Load(IInputStream* input);
                void Save(IOutputStream* output) const;

                EAnswer Get() const {
                    return Value;
                }
                void Set(EAnswer value) {
                    Value = value;
                }

            private:
                EAnswer Value;
            };

            TAnswer Answer;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Answer
            );

            TString DebugString() const override {
                return "Login Answer " + ToString(Answer.Get());
            }
        };

        class TPingRequest: public TPayload<MT_PING_REQUEST> {
        public:
            DEFINE_WIALON_NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return "Ping Request";
            }
        };

        class TPingAnswer: public TPayload<MT_PING_ANSWER> {
        public:
            DEFINE_WIALON_NO_MESSAGE_FIELDS();

            TString DebugString() const override {
                return "Ping Answer";
            }
        };

        class TShortData {
        public:
            TSeparated<TInstant> DateTime;
            TSeparated<TCoordinate> Lattitude;
            TSeparated<TString> Lattitude2;
            TSeparated<TCoordinate> Longitude;
            TSeparated<TString> Longitude2;
            TSeparated<ui16> Speed;
            TSeparated<ui16> Course;
            TSeparated<i16> Height;
            TSeparated<ui8> Satellites;

            DEFINE_WIALON_PACKET_FIELDS(
                DateTime,
                Lattitude,
                Lattitude2,
                Longitude,
                Longitude2,
                Speed,
                Course,
                Height,
                Satellites
            );

            TString DebugString() const;
        };

        class TAdditionalData {
        public:
            using TAnalog = TSeparatedValue<float, ','>;

            class TParams {
            public:
                using TItem = std::variant<ui64, double, TString>;
                using TValue = TMap<TString, TItem>;

                static constexpr size_t ValidParamSize = 3;

                enum EType: ui8 {
                    Incorrect = 0 /* "incorrect" */,
                    Integer /* "integer" */,
                    Double /* "double" */,
                    String /* "string" */,
                };

            public:
                TParams() = default;

                void Load(IInputStream* input);
                void Save(IOutputStream* output) const;

                const TValue& Get() const {
                    return Value;
                }

            private:
                TValue Value;

            private:
                void ParseParams(IInputStream* intut);
            };

            TSeparated<float> Hdop;
            TSeparated<ui64> Inputs;
            TSeparated<ui64> Outputs;
            TSeparatedCollection<TAnalog, ';'> Analogs;
            TSeparated<TString> IButton;
            TParams Params;

            DEFINE_WIALON_PACKET_FIELDS(
                Hdop,
                Inputs,
                Outputs,
                Analogs,
                IButton,
                Params
            );

            TString DebugString() const;
        };

        class TShortDataRequest: public TPayload<MT_SHORT_DATA_REQUEST> {
        public:
            TShortData ShortData;

            DEFINE_WIALON_MESSAGE_FIELDS(
                ShortData
            );

            TString DebugString() const override {
                return "Short Data Request " + ShortData.DebugString();
            }
        };

        class TShortDataAnswer: public TPayload<MT_SHORT_DATA_ANSWER> {
        public:
            enum EAnswer: i8 {
                StructurePacketError = -1 /* "structure packet error" */,
                TimeIncorrect = 0 /* "time incorrect" */,
                Success = 1 /* "success" */,
                CoordsError = 10 /* "coords error" */,
                SpeedCourseHeigthError = 11 /* "speed course heigth error" */,
                SatelliteError = 12 /* "satillite error" */,
                ChecksumError = 13 /* "checksum error" */

            };

            using TAnswer = TEnumValue<EAnswer>;

            TAnswer Answer;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Answer
            );

            TString DebugString() const override {
                return "Short Data " + ToString(Answer.Get());
            }
        };

        class TDataRequest: public TPayload<MT_DATA_REQUEST> {
        public:
            TShortData ShortData;
            TAdditionalData AdditionalData;

            DEFINE_WIALON_MESSAGE_FIELDS(
                ShortData,
                AdditionalData
            );

            TString DebugString() const override;
        };

        class TDataAnswer: public TPayload<MT_DATA_ANSWER> {
        public:
            enum EAnswer: i8 {
                StructurePacketError = -1 /* "structure packet error" */,
                TimeIncorrect = 0 /* "time incorrect" */,
                Success = 1 /* "success" */,
                CoordsError = 10 /* "coords error" */,
                SpeedCourseHeigthError = 11 /* "speed course heigth error" */,
                SatelliteHdopError = 12 /* "satellite or hdop error" */,
                InputOutputError = 13 /* "inputs or outputs error" */,
                AnalogError = 14 /* "analogs error" */,
                ParamError = 15 /* "params error" */,
                ChecksumError = 16 /* "checksum error" */,
            };

            using TAnswer = TEnumValue<EAnswer>;

            TAnswer Answer;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Answer
            );

            TString DebugString() const override {
                return "Data Answer " + ToString(Answer.Get());
            }
        };

        class TBlackBoxRequest: public TPayload<MT_BLACK_BOX_REQUEST> {
        public:
            struct TItem {
                TShortData ShortData;
                TAdditionalData AdditionalData;

                DEFINE_WIALON_PACKET_FIELDS(
                    ShortData,
                    AdditionalData
                );

                static char GetSeparator() {
                    return ';';
                }

                TString DebugString() const {
                    return ShortData.DebugString() + " " + AdditionalData.DebugString();
                }
            };

            class TCollection {
            public:
                inline TCollection() = default;

                static char GetSeparator() {
                    return '|';
                }

                void Load(IInputStream* input);
                void Save(IOutputStream* output) const;
                TString DebugString() const;

                const TVector<TItem>& Get() const {
                    return Collection;
                }

            private:
                TVector<TItem> Collection;
            };

            TCollection Data;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Data
            );

            TString DebugString() const override {
                return "Black box Request";
            }
        };

        class TBlackBoxAnswer: public TPayload<MT_BLACK_BOX_ANSWER> {
        public:
            using TAnswer = TEnumValue<size_t, size_t>;

            // Answer is count of receive packets
            TAnswer Answer;

            DEFINE_WIALON_MESSAGE_FIELDS(
                Answer
            );

            TString DebugString() const override {
                return "Black box Answer " + ToString(Answer.Get());
            }
        };

        class TSettingFileRequest: public TPayload<MT_SETTING_FILE_REQUEST> {
        public:

            void Load(IInputStream& input) override;
            void Save(IOutputStream& output) const override;
            void PostSave(IOutputStream& output) const override;

            const TBuffer& GetData() const {
                return Data;
            }
            void SetData(const TBuffer& data) {
                Data = data;
            }
            void SetData(TBuffer&& data) {
                Data = std::move(data);
            }

            TString DebugString() const override;

        private:
            TBuffer Data;
        };

        NDrive::TMultiSensor ToSensors(const TShortData& data);
        NDrive::TMultiSensor ToSensors(const TAdditionalData& data, TInstant timestamp = Now());
    }
}
