#include "nonce.h"
#include "wialon.h"

#include <rtline/library/json/cast.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/str.h>
#include <util/string/hex.h>

namespace {
    template<class TAnswerType, class TPayloadType>
    TString GetAnswerPayload(TAnswerType answer) {
        auto payload = NDrive::NProtocol::IPayload::Create<TPayloadType>();
        payload->Answer = answer;
        NDrive::NWialon::TMessage message(std::move(payload));

        TString result;
        TStringOutput stream(result);
        message.Save(stream);

        return result;
    }

    template<class T>
    struct HelperData {
        T Answer;
        TStringBuf Expected;
    };
}

Y_UNIT_TEST_SUITE(TelematicsWialonProtocol) {
    Y_UNIT_TEST(Login) {
        {
            TString data = "#L#2.0;869286035183933;NA;9E33\r\n";
            TStringInput dataStream(data);

            NDrive::NWialon::TMessage message;
            message.Load(dataStream);

            UNIT_ASSERT(message.IsValid());
            {
                auto& payload = message.As<NDrive::NWialon::TLoginRequest>();

                UNIT_ASSERT_STRINGS_EQUAL(payload.Version.Get(), "2.0");
                UNIT_ASSERT_STRINGS_EQUAL(payload.IMEI.Get(), "869286035183933");
                UNIT_ASSERT_STRINGS_EQUAL(payload.Password.Get(), "NA");
            }
        }

        {
            using THelperData = HelperData<NDrive::NWialon::TLoginAnswer::EAnswer>;
            using TAnswerType = NDrive::NWialon::TLoginAnswer::EAnswer;
            using TPayloadType = NDrive::NWialon::TLoginAnswer;

            const TVector<THelperData> data = {
                { NDrive::NWialon::TLoginAnswer::Success, "#AL#1\r\n" },
                { NDrive::NWialon::TLoginAnswer::Reject, "#AL#0\r\n" },
                { NDrive::NWialon::TLoginAnswer::PasswordError, "#AL#01\r\n" },
                { NDrive::NWialon::TLoginAnswer::ChecksumError, "#AL#10\r\n" },
            };

            for (auto&& item : data) {
                TString res = GetAnswerPayload<TAnswerType, TPayloadType>(item.Answer);
                UNIT_ASSERT_STRINGS_EQUAL(res, item.Expected);
            }
        }
    }

    Y_UNIT_TEST(Ping) {
        {
            TString data = "#P#\r\n";
            TStringInput dataStream(data);

            NDrive::NWialon::TMessage message;
            message.Load(dataStream);
            UNIT_ASSERT(message.IsValid());
        }

        {
            using namespace NDrive::NWialon;
            using namespace NDrive::NProtocol;
            auto payload = IPayload::Create<TPingAnswer>();
            TMessage message(std::move(payload));

            TString result;
            TStringOutput stream(result);
            message.Save(stream);

            UNIT_ASSERT_STRINGS_EQUAL(result, "#AP#\r\n");
        }
    }

    Y_UNIT_TEST(ShortData) {
        {
            TDateTimeFields field;
            field.SetLooseYear(21);
            field.Month = 9;
            field.Day = 29;
            field.Hour = 8;
            field.Minute = 12;
            field.Second = 50;
            TInstant expectInstant = field.ToInstant(TInstant::Now());

            TString data = "#SD#290921;081250;5544.6025;N;03739.6834;E;54;34;120;13;12B6\r\n";
            TStringInput dataStream(data);

            NDrive::NWialon::TMessage message;
            message.Load(dataStream);

            UNIT_ASSERT(message.IsValid());
            UNIT_ASSERT_EQUAL(message.GetMessageType(), NDrive::NWialon::MT_SHORT_DATA_REQUEST);
            UNIT_ASSERT_EQUAL(message.GetProtocolType(), NDrive::NProtocol::PT_WIALON_IPS);

            auto& payload = message.As<NDrive::NWialon::TShortDataRequest>();

            float lattitude = payload.ShortData.Lattitude.Get().GetRef().GetValue();
            float expectLattitude = 55.446025f;
            float longitude = payload.ShortData.Longitude.Get().GetRef().GetValue();
            float expectLongitude = 37.396834f;

            UNIT_ASSERT(payload.ShortData.DateTime.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.DateTime.Get().GetRef(), expectInstant);
            UNIT_ASSERT(payload.ShortData.Lattitude.Get());
            UNIT_ASSERT_EQUAL(lattitude, expectLattitude);
            UNIT_ASSERT(payload.ShortData.Lattitude2.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.Lattitude2.Get(), "N");
            UNIT_ASSERT(payload.ShortData.Longitude.Get());
            UNIT_ASSERT_EQUAL(longitude, expectLongitude);
            UNIT_ASSERT(payload.ShortData.Longitude2.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.Longitude2.Get(), "E");
            UNIT_ASSERT(payload.ShortData.Speed.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.Speed.Get(), 54);
            UNIT_ASSERT(payload.ShortData.Course.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.Course.Get(), 34);
            UNIT_ASSERT(payload.ShortData.Satellites.Get());
            UNIT_ASSERT_EQUAL(payload.ShortData.Satellites.Get(), 13);
        }

        {
            using namespace NDrive::NWialon;
            using namespace NDrive::NProtocol;

            using THelperData = HelperData<NDrive::NWialon::TShortDataAnswer::TAnswer>;
            using TAnswerType = NDrive::NWialon::TShortDataAnswer::TAnswer;
            using TPayloadType = NDrive::NWialon::TShortDataAnswer;

            const TVector<THelperData> data = {
                { NDrive::NWialon::TShortDataAnswer::StructurePacketError, "#ASD#-1\r\n" },
                { NDrive::NWialon::TShortDataAnswer::TimeIncorrect, "#ASD#0\r\n" },
                { NDrive::NWialon::TShortDataAnswer::Success, "#ASD#1\r\n" },
                { NDrive::NWialon::TShortDataAnswer::CoordsError, "#ASD#10\r\n" },
                { NDrive::NWialon::TShortDataAnswer::SpeedCourseHeigthError, "#ASD#11\r\n" },
                { NDrive::NWialon::TShortDataAnswer::SatelliteError, "#ASD#12\r\n" },
                { NDrive::NWialon::TShortDataAnswer::ChecksumError, "#ASD#13\r\n" },
            };

            for (auto&& item : data) {
                auto res = GetAnswerPayload<TAnswerType, TPayloadType>(item.Answer);
                UNIT_ASSERT_STRINGS_EQUAL(res, item.Expected);
            }
        }
    }
    Y_UNIT_TEST(Data) {
        {
            TDateTimeFields field;
            field.SetLooseYear(21);
            field.Month = 10;
            field.Day = 4;
            field.Hour = 23;
            field.Minute = 01;
            field.Second = 37;
            TInstant expectInstant = field.ToInstant(TInstant::Now());

            TString test = "14.77,0.02,3.6";
            TStringInput testStream(test);

            TString rawData = "#D#041021;230137;5538.9819;N;03726.5091;E;" \
                              "12;125;NA;11;0.8100;NA;NA;;NA;SOS:1:0,ev:3:NO EVENT" \
                              ",sat:3:11,HDOP:2:0.810000,spd:3:0,az:3:0,temp:3:16," \
                              "bat:3:100,mnc:1:01,mcc:1:250,lac:1:700,cell_id:1:63120," \
                              "sig:3:3/4,mode:3:SUT,up:3:+79139121666,pin:3:1234," \
                              "N:3:0,58:3:0,24:3:06 00,p:3:7,fmt:3:LINK,12:3:12," \
                              "38:3:OFF,pr1:3:23 00 - 07 00,pr2:3:10 00 - 17 00," \
                              "axi:3:5,axa:3:20,axd:3:0,axdbl:3:1,bootv:3:5,appv:3:62;BCD7";

            TStringInput inputStream(rawData);
            NDrive::NWialon::TMessage message;
            message.Load(inputStream);

            auto& payload = message.As<NDrive::NWialon::TDataRequest>();

            UNIT_ASSERT(payload.ShortData.DateTime.Get());
            UNIT_ASSERT(payload.ShortData.Lattitude.Get());
            UNIT_ASSERT(payload.ShortData.Lattitude2.Get());
            UNIT_ASSERT(payload.ShortData.Longitude.Get());
            UNIT_ASSERT(payload.ShortData.Longitude2.Get());
            UNIT_ASSERT(payload.ShortData.Speed.Get());
            UNIT_ASSERT(payload.ShortData.Course.Get());
            UNIT_ASSERT(!payload.ShortData.Height.Get());
            UNIT_ASSERT(payload.ShortData.Satellites.Get());
            UNIT_ASSERT(payload.AdditionalData.Hdop.Get());
            UNIT_ASSERT(!payload.AdditionalData.Inputs.Get());
            UNIT_ASSERT(!payload.AdditionalData.Outputs.Get());
            UNIT_ASSERT(payload.AdditionalData.Analogs.Get().empty());
            UNIT_ASSERT(!payload.AdditionalData.IButton.Get());
            UNIT_ASSERT(!payload.AdditionalData.Params.Get().empty());

            float lattitude = payload.ShortData.Lattitude.Get().GetRef().GetValue();
            float expectLattitude = 55.389819f;
            float longitude = payload.ShortData.Longitude.Get().GetRef().GetValue();
            float expectLongitude = 37.265091f;

            UNIT_ASSERT_EQUAL(payload.ShortData.DateTime.Get(), expectInstant);
            UNIT_ASSERT_EQUAL(lattitude, expectLattitude);
            UNIT_ASSERT_EQUAL(payload.ShortData.Lattitude2.Get(), "N");
            UNIT_ASSERT_EQUAL(longitude, expectLongitude);
            UNIT_ASSERT_EQUAL(payload.ShortData.Longitude2.Get(), "E");
            UNIT_ASSERT_EQUAL(payload.ShortData.Speed.Get(), 12);
            UNIT_ASSERT_EQUAL(payload.ShortData.Course.Get(), 125);
            UNIT_ASSERT_EQUAL(payload.ShortData.Satellites.Get(), 11);
            UNIT_ASSERT_EQUAL(payload.AdditionalData.Hdop.Get(), 0.81f);

            // params check
            const auto& params = payload.AdditionalData.Params.Get();

            auto emptyValue = ToString(NDrive::NWialon::EmptyValue);

            UNIT_ASSERT_EQUAL(std::get<ui64>(params.Value("SOS", ui64(12))), 0);
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("ev", emptyValue)), "NO EVENT");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("sat", emptyValue)), "11");
            UNIT_ASSERT_EQUAL(std::get<double>(params.Value("HDOP", double(-1))), 0.81);
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("spd", emptyValue)), "0");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("az", emptyValue)), "0");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("temp", emptyValue)), "16");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("bat", emptyValue)), "100");
            UNIT_ASSERT_EQUAL(std::get<ui64>(params.Value("mnc", ui64(0))), 1);
            UNIT_ASSERT_EQUAL(std::get<ui64>(params.Value("mcc", ui64(0))), 250);
            UNIT_ASSERT_EQUAL(std::get<ui64>(params.Value("lac", ui64(0))), 700);
            UNIT_ASSERT_EQUAL(std::get<ui64>(params.Value("cell_id", ui64(0))), 63120);
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("sig", emptyValue)), "3/4");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("mode", emptyValue)), "SUT");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("up", emptyValue)), "+79139121666");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("pin", emptyValue)), "1234");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("N", emptyValue)), "0");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("58", emptyValue)), "0");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("24", emptyValue)), "06 00");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("p", emptyValue)), "7");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("fmt", emptyValue)), "LINK");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("12", emptyValue)), "12");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("38", emptyValue)), "OFF");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("pr1", emptyValue)), "23 00 - 07 00");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("pr2", emptyValue)), "10 00 - 17 00");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("axi", emptyValue)), "5");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("axa", emptyValue)), "20");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("axd", emptyValue)), "0");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("axdbl", emptyValue)), "1");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("bootv", emptyValue)), "5");
            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("appv", emptyValue)), "62");

            UNIT_ASSERT_EQUAL(std::get<TString>(params.Value("dummy", "NA")), "NA");
        }

        {
            TString rawData = "#D#NA;NA;NA;NA;NA;NA;NA;NA;NA;NA;NA;NA;NA;;NA;" \
                              "SOS:1:0,ev:3:NO EVENT,sat:3:0,HDOP:2:0,spd:3:0," \
                              "az:3:0,temp:3:24,bat:3:100%,mnc:1:02,mcc:1:250," \
                              "lac:1:9736,cell_id:1:13545,sig:3:4/4,mode:3:SUT," \
                              "up:3:+79139121666,pin:3:1234,N:3:0,58:3:50,24:3:06 00," \
                              "p:3:3,fmt:3:LINK,12:3:,38:3:OFF,pr1:3:23 00 - 07 00," \
                              "pr2:3:10 00 - 17 00,axi:3:5,axa:3:20,axd:3:0,axdbl:3:1" \
                              ",bootv:3:5,appv:3:50;AD0B";

            TStringInput inputStream(rawData);
            NDrive::NWialon::TMessage message;
            message.Load(inputStream);

            UNIT_ASSERT(message.IsValid());

            auto& payload = message.As<NDrive::NWialon::TDataRequest>();

            UNIT_ASSERT(!payload.ShortData.DateTime.Get());
            UNIT_ASSERT(!payload.ShortData.Lattitude.Get());
            UNIT_ASSERT(!payload.ShortData.Lattitude2.Get());
            UNIT_ASSERT(!payload.ShortData.Longitude.Get());
            UNIT_ASSERT(!payload.ShortData.Longitude2.Get());
            UNIT_ASSERT(!payload.ShortData.Speed.Get());
            UNIT_ASSERT(!payload.ShortData.Course.Get());
            UNIT_ASSERT(!payload.ShortData.Height.Get());
            UNIT_ASSERT(!payload.ShortData.Satellites.Get());

            UNIT_ASSERT_EQUAL(payload.AdditionalData.Params.Get().size(), 31);
        }

        {
            using namespace NDrive::NWialon;
            using namespace NDrive::NProtocol;

            using THelperData = HelperData<NDrive::NWialon::TDataAnswer::TAnswer>;
            using TAnswerType = NDrive::NWialon::TDataAnswer::TAnswer;
            using TPayloadType = NDrive::NWialon::TDataAnswer;

            const TVector<THelperData> data = {
                { NDrive::NWialon::TDataAnswer::StructurePacketError, "#AD#-1\r\n" },
                { NDrive::NWialon::TDataAnswer::TimeIncorrect, "#AD#0\r\n" },
                { NDrive::NWialon::TDataAnswer::Success, "#AD#1\r\n" },
                { NDrive::NWialon::TDataAnswer::CoordsError, "#AD#10\r\n" },
                { NDrive::NWialon::TDataAnswer::SpeedCourseHeigthError, "#AD#11\r\n" },
                { NDrive::NWialon::TDataAnswer::SatelliteHdopError, "#AD#12\r\n" },
                { NDrive::NWialon::TDataAnswer::InputOutputError, "#AD#13\r\n" },
                { NDrive::NWialon::TDataAnswer::AnalogError, "#AD#14\r\n" },
                { NDrive::NWialon::TDataAnswer::ParamError, "#AD#15\r\n" },
                { NDrive::NWialon::TDataAnswer::ChecksumError, "#AD#16\r\n" },
            };

            for (auto&& item : data) {
                auto res = GetAnswerPayload<TAnswerType, TPayloadType>(item.Answer);
                UNIT_ASSERT_STRINGS_EQUAL(res, item.Expected);
            }
        }
    }

    Y_UNIT_TEST(BlackBox) {
        {

            TString rawData = "#B#041021;230137;5538.9819;N;03726.5091;E;" \
                              "12;125;NA;11;0.8100;NA;NA;;NA;SOS:1:0,ev:3:NO EVENT" \
                              ",sat:3:11,HDOP:2:0.810000,spd:3:0,az:3:0,temp:3:16," \
                              "bat:3:100,mnc:1:01,mcc:1:250,lac:1:700,cell_id:1:63120," \
                              "sig:3:3/4,mode:3:SUT,up:3:+79139121666,pin:3:1234," \
                              "N:3:0,58:3:0,24:3:06 00,p:3:7,fmt:3:LINK,12:3:12," \
                              "38:3:OFF,pr1:3:23 00 - 07 00,pr2:3:10 00 - 17 00," \
                              "axi:3:5,axa:3:20,axd:3:0,axdbl:3:1,bootv:3:5,appv:3:62|" \
                              "041021;230137;5538.9819;N;03726.5091;E;" \
                              "12;125;NA;11;0.8100;NA;NA;;NA;SOS:1:0,ev:3:NO EVENT" \
                              ",sat:3:11,HDOP:2:0.810000,spd:3:0,az:3:0,temp:3:16," \
                              "bat:3:100,mnc:1:01,mcc:1:250,lac:1:700,cell_id:1:63120," \
                              "sig:3:3/4,mode:3:SUT,up:3:+79139121666,pin:3:1234," \
                              "N:3:0,58:3:0,24:3:06 00,p:3:7,fmt:3:LINK,12:3:12," \
                              "38:3:OFF,pr1:3:23 00 - 07 00,pr2:3:10 00 - 17 00," \
                              "axi:3:5,axa:3:20,axd:3:0,axdbl:3:1,bootv:3:5,appv:3:62|BCD7";

            TStringInput inputStream(rawData);
            NDrive::NWialon::TMessage message;
            message.Load(inputStream);

            UNIT_ASSERT(message.IsValid());

            auto& payload = message.As<NDrive::NWialon::TBlackBoxRequest>();
            const auto& data = payload.Data.Get();
            UNIT_ASSERT_EQUAL(data.size(), 2);
        }

        {
            using namespace NDrive::NWialon;
            using namespace NDrive::NProtocol;

            using THelperData = HelperData<NDrive::NWialon::TBlackBoxAnswer::TAnswer>;
            using TAnswerType = NDrive::NWialon::TBlackBoxAnswer::TAnswer;
            using TPayloadType = NDrive::NWialon::TBlackBoxAnswer;

            const TVector<THelperData> data = {
                { 0, "#AB#0\r\n" },
                { 3, "#AB#3\r\n" },
            };

            for (auto&& item : data) {
                auto res = GetAnswerPayload<TAnswerType, TPayloadType>(item.Answer);
                UNIT_ASSERT_STRINGS_EQUAL(res, item.Expected);
            }
        }
    }

    Y_UNIT_TEST(SettingFile) {
        auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NWialon::TSettingFileRequest>();
        UNIT_ASSERT(payload);

        {
            TBuffer data("TS", 2);
            payload->SetData(std::move(data));
            NDrive::NWialon::TMessage message(std::move(payload));

            TString result = "";
            TStringOutput stream(result);
            message.Save(stream);

            auto expected = ToString("#UC#2;FD7E\r\nTS");
            UNIT_ASSERT_EQUAL(result, expected);
        }
    }
}
