#include "pusher.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/pusher/common.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/network/neh.h>

#include <util/string/vector.h>

THolder<NDrive::IPusher> NDrive::TBackendPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    return MakeHolder<TBackendPusher>(*this, tvm);
}

TSet<NTvmAuth::TTvmId> NDrive::TBackendPusherOptions::GetDestinationClientIds() const {
  return {DestinationClientId};
}

void NDrive::TBackendPusherOptions::Init(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    Endpoint = directives.Value("Endpoint", Endpoint);
    DestinationClientId = directives.Value("DestinationClientId", DestinationClientId);
    StringSplitter(
        directives.Value("SensorIds", JoinStrings(SensorIds.begin(), SensorIds.end(), TStringBuf(" ")))
    ).SplitBySet(" ,").SkipEmpty().ParseInto(&SensorIds);
}

void NDrive::TBackendPusherOptions::Print(IOutputStream& os) const {
    os << "Endpoint: " << Endpoint << Endl;
    os << "DestinationClientId: " << DestinationClientId << Endl;
    os << "SensorIds: " << JoinStrings(SensorIds.begin(), SensorIds.end(), " ") << Endl;
}

NDrive::TBackendPusher::TBackendPusher(const NDrive::TBackendPusherOptions& options, const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm)
    : Client(MakeHolder<NNeh::THttpClient>(options.Endpoint))
    , TvmAuth(options.DestinationClientId ? MakeHolder<NDrive::TTvmAuth>(tvm, options.DestinationClientId) : nullptr)
    , Handler(NUrl::SplitUrlToHostAndPath(options.Endpoint).path)
    , SensorIds(options.SensorIds)
{
}

NDrive::TBackendPusher::~TBackendPusher() {
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TBackendPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    Y_UNUSED(deadline);
    Y_UNUSED(handler);
    Y_UNUSED(imei);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TBackendPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    Y_UNUSED(deadline);
    Y_UNUSED(heartbeat);
    Y_UNUSED(imei);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TBackendPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    Y_UNUSED(deadline);
    Y_UNUSED(imei);
    Y_UNUSED(location);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TBackendPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    if (!SensorIds.contains(sensor)) {
        return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
    }

    auto request = CreateRequest();
    {
        NJson::TJsonValue post;
        post["imei"] = imei;
        post["sensors"] = NJson::ToJson(NContainer::Scalar(sensor));
        request.SetPostData(post.GetStringRobust());
    }

    auto asyncResponse = Yensured(Client)->SendAsync(request, deadline);
    auto result = ProcessResponse(asyncResponse);
    return result;
}

NNeh::THttpRequest NDrive::TBackendPusher::CreateRequest() const {
    NNeh::THttpRequest request;
    request.SetUri(Handler);
    request.AddHeader("UserPermissionsCache", "true");
    if (TvmAuth) {
        TvmAuth->UpdateRequest(request);
    }
    return request;
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TBackendPusher::ProcessResponse(const NThreading::TFuture<NNeh::THttpReply>& reply) const {
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        if (r.HasException()) {
            NDrive::TTelematicsUnistatSignals::Get().BackendPusherException.Signal(1);
            return TPushResult{ /* Written= */ false, /* Message= */ NThreading::GetExceptionMessage(r) };
        }
        const auto& response = r.GetValue();
        if (response.IsSuccessReply()) {
            NDrive::TTelematicsUnistatSignals::Get().BackendPusherSuccess.Signal(1);
        } else {
            NDrive::TTelematicsUnistatSignals::Get().BackendPusherFailure.Signal(1);
        }
        if (!response.IsSuccessReply()) {
            return TPushResult{ /* Written= */ false, /* Message= */ response.Serialize().GetStringRobust()};
        }
        return TPushResult{ /* Written= */ response.IsSuccessReply(), /* Message= */ "" };
    });
}
