#include "pusher.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/string_utils/url/url.h>

THolder<NDrive::IPusher> NDrive::TTelematicsClickHousePusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    Y_UNUSED(tvm);
    return MakeHolder<TTelematicsClickHousePusher>(*this);
}

void NDrive::TTelematicsClickHousePusherOptions::Init(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    NClickHouse::TClientOptions option;
    option.DefaultDatabase = directives.Value("Database", option.DefaultDatabase);
    option.User = directives.Value("User", option.User);
    option.Password = directives.Value("Password", option.Password);
    option.SetUseSsl(true);

    auto endpoints = directives.Value<TString>("Endpoints");
    for (auto&& endpoint : StringSplitter(endpoints).SplitBySet(", ").SkipEmpty()) {
        TStringBuf scheme;
        TStringBuf host;
        ui16 port = 0;
        Y_ENSURE(TryGetSchemeHostAndPort(endpoint, scheme, host, port), "cannot TryGetSchemeHostAndPort from " << endpoint.Token());
        option.SetHost(ToString(host));
        option.SetPort(port);
        Options.push_back(option);
    }
}

void NDrive::TTelematicsClickHousePusherOptions::Print(IOutputStream& os) const {
    os << "Endpoints: ";
    for (auto&& option : Options) {
        os << option.Host << ":" << option.Port;
    }
    os << Endl;
    if (!Options.empty()) {
        const auto& option = Options.front();
        os << "Database: " << option.DefaultDatabase << Endl;
        os << "User: " << option.User << Endl;
    }
}

NDrive::TTelematicsClickHousePusher::TTelematicsClickHousePusher(const TTelematicsClickHousePusherOptions& options)
    : Pusher(options.GetOptions(), options.GetBalancingOptions())
{
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsClickHousePusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    Y_UNUSED(imei);
    Y_UNUSED(deadline);
    Y_UNUSED(handler);
    return NThreading::MakeFuture<TPushResult>();
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsClickHousePusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    Y_UNUSED(imei);
    Y_UNUSED(deadline);
    Y_UNUSED(heartbeat);
    return NThreading::MakeFuture<TPushResult>();
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsClickHousePusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    Y_UNUSED(deadline);
    TString objectId;
    NThreading::TFuture<void> waiter;
    {
        NDrive::TLocationClickHouseRecord record(location, imei, objectId);
        waiter = Pusher.Push(std::move(record));
    }
    return waiter.Apply([](const NThreading::TFuture<void>& w) {
        TPushResult result;
        if (w.HasValue()) {
            result.Written = true;
        } else {
            result.Written = false;
            result.Message = NThreading::GetExceptionMessage(w);
        }
        return result;
    });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsClickHousePusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    Y_UNUSED(deadline);
    TString objectId;
    NThreading::TFuture<void> waiter;
    {
        NDrive::TSensorClickHouseRecord record(sensor, imei, objectId);
        waiter = Pusher.Push(std::move(record));
    }
    return waiter.Apply([](const NThreading::TFuture<void>& w) {
        TPushResult result;
        if (w.HasValue()) {
            result.Written = true;
        } else {
            result.Written = false;
            result.Message = NThreading::GetExceptionMessage(w);
        }
        return result;
    });
}
