#include "blackbox.h"

#include <drive/telematics/server/data/proto/blackbox.pb.h>

#include <drive/telematics/protocol/vega.h>

#include <drive/library/cpp/compression/simple.h>
#include <drive/library/cpp/yt/node/cast.h>

TInstant NDrive::TTelematicsHistory::GetTimestamp() const {
    return std::max(
        Locations.GetTimestamp(),
        Sensors.GetTimestamp()
    );
}

TString NDrive::TTelematicsHistory::DebugString() const {
    return Serialize<NDrive::NProto::TTelematicsHistory>().DebugString();
}

void NDrive::TTelematicsHistory::Add(TBlackboxRecord&& record) {
    if (record.Location) {
        Locations.Add(*record.Location);
    }
    {
        Add(std::move(record.Sensors));
    }
}

void NDrive::TTelematicsHistory::Add(TBlackboxRecords&& records) {
    for (auto&& record : records.Values) {
        Add(std::move(record));
    }
}

void NDrive::TTelematicsHistory::Add(NDrive::TGpsLocation&& location) {
    Locations.Add(std::move(location));
}

void NDrive::TTelematicsHistory::Add(TMultiSensor&& sensors) {
    Sensors.Add(std::move(sensors));
}

void NDrive::TTelematicsHistory::Add(TSensor&& sensor) {
    Sensors.Add(std::move(sensor));
}

TMaybe<TInstant> NDrive::TTelematicsHistory::GetPushed(const TGpsLocation& /*location*/) const {
    auto g = Guard(PushedLock);
    return PushedLocation;
}

TMaybe<TInstant> NDrive::TTelematicsHistory::GetPushed(TSensorId sensorId) const {
    auto g = Guard(PushedLock);
    auto p = PushedSensors.find(sensorId);
    if (p != PushedSensors.end()) {
        return p->second;
    } else {
        return {};
    }
}

void NDrive::TTelematicsHistory::OnPush(const TGpsLocation& location) {
    auto g = Guard(PushedLock);
    if (PushedLocation) {
        PushedLocation = std::max(*PushedLocation, location.Timestamp.Get());
    } else {
        PushedLocation = location.Timestamp;
    }
}

void NDrive::TTelematicsHistory::OnPush(const TSensor& sensor) {
    auto g = Guard(PushedLock);
    auto& s = PushedSensors[sensor];
    s = std::max(s, sensor.Timestamp.Get());
}

void NDrive::TTelematicsHistory::Serialize(NDrive::NProto::TTelematicsHistory& proto) const {
    if (auto locations = proto.MutableLocations()) {
        Locations.Serialize(*locations);
    }
    if (auto sensors = proto.MutableSensors()) {
        Sensors.Serialize(*sensors);
    }
}

void NDrive::TTelematicsHistory::Serialize(TString& value) const {
    NDrive::NProto::TTelematicsHistory proto;
    Serialize(proto);
    value = NDrive::Compress(proto.SerializeAsString()).ExtractValue();
}

bool NDrive::TTelematicsHistory::Deserialize(const NDrive::NProto::TTelematicsHistory& proto) {
    if (proto.HasLocations()) {
        if (!Locations.Deserialize(proto.GetLocations())) {
            return false;
        }
    }
    if (proto.HasSensors()) {
        if (!Sensors.Deserialize(proto.GetSensors())) {
            return false;
        }
    }
    return true;
}

bool NDrive::TTelematicsHistory::Deserialize(const TString& value) {
    auto decompressed = NDrive::Decompress(value);
    if (!decompressed) {
        return false;
    }
    NDrive::NProto::TTelematicsHistory proto;
    if (!proto.ParseFromString(*decompressed)) {
        return false;
    }
    return Deserialize(proto);
}

template <>
NDrive::TBlackboxRecord NYT::FromNode<NDrive::TBlackboxRecord>(const TNode& node) {
    NDrive::TBlackboxRecord result;
    auto timestamp = TInstant::Seconds(node["timestamp"].ConvertTo<ui64>());
    for (auto&& subrecord : node["subrecords"].AsList()) {
        auto type = subrecord["type"].ConvertTo<TString>();
        if (type == "position_data") {
            NDrive::TGpsLocation location;
            location.Course = subrecord["course"].ConvertTo<ui64>();
            location.Latitude = subrecord["lat"].ConvertTo<double>();
            location.Longitude = subrecord["lon"].ConvertTo<double>();
            location.Speed = subrecord["speed"].ConvertTo<ui32>();
            location.Timestamp = timestamp;
            location.Since = timestamp;
            result.Location = location;

            auto addSensor = [&] (ui16 id, auto value) {
                NDrive::TSensor sensor(id);
                sensor.Value = value;
                sensor.Timestamp = timestamp;
                sensor.Since = timestamp;
                result.Sensors.push_back(std::move(sensor));
            };
            auto altitude = subrecord["height"].ConvertTo<double>();
            auto satelites = subrecord["sats"].ConvertTo<ui64>();
            auto speed = static_cast<double>(location.Speed);
            addSensor(VEGA_ALT, altitude);
            addSensor(VEGA_SAT_USED, satelites);
            addSensor(VEGA_SPEED, speed);
        }
        if (type == "custom_parameters") {
            for (auto&& [param, valueNode] : subrecord["params"].AsMap()) {
                auto id = FromString<NDrive::TSensorId>(param);
                auto valueString = valueNode.ConvertTo<TString>();
                auto value = NDrive::SensorValueFromString(valueString, id);
                NDrive::TSensor sensor(id);
                sensor.Value = std::move(value);
                sensor.Timestamp = timestamp;
                sensor.Since = timestamp;
                result.Sensors.push_back(std::move(sensor));
            }
        }
    }
    return result;
}

template <>
NDrive::TBlackboxRecords NYT::FromNode<NDrive::TBlackboxRecords>(const TNode& node) {
    NDrive::TBlackboxRecords result;
    result.Id = node["id"].ConvertTo<ui64>();
    for (auto&& record : node["records"].AsList()) {
        result.Values.push_back(FromNode<NDrive::TBlackboxRecord>(record));
    }
    return result;
}
