#include "client.h"

#include "block.h"

#include <drive/library/cpp/clickhouse/client.h>

#include <library/cpp/clickhouse/client/client.h>
#include <library/cpp/logger/global/global.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/exception.h>
#include <rtline/library/storage/sql/query.h>

NDrive::TClickHouseClient::TClickHouseClient(const NClickHouse::TAsyncClientOptions& options)
    : Client(MakeHolder<NClickHouse::TAsyncClient>(options))
{
}

NDrive::TClickHouseClient::~TClickHouseClient() {
}

NThreading::TFuture<TVector<TString>> NDrive::TClickHouseClient::SelectIds(TStringBuf query, TInstant deadline) const {
    auto block = Yensured(Client)->Execute(query, deadline);
    auto ids = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        Y_ENSURE(block.GetColumnCount() == 1, "invalid column count: " << block.GetColumnCount());
        const auto column = block[0];
        Y_ENSURE(column);

        TVector<TString> result(column->Size());
        for (size_t i = 0; i < column->Size(); ++i) {
            result[i] = NClickHouse::GetColumnValue<TString>(column, i);
        }
        return result;
    });
    return ids;
}

template <class T>
NSQL::TQueryOptions CreateQueryOptions(const T& query) {
    NSQL::TQueryOptions queryOptions;
    if (const auto& objectIds = query.GetObjectIds(); !objectIds.empty()) {
        queryOptions.SetGenericCondition("object_id", objectIds);
    }
    if (const auto& imeis = query.GetImeis(); !imeis.empty()) {
        queryOptions.SetGenericCondition("imei", imeis);
    }
    if (const auto& timestamp = query.GetTimestamp()) {
        TRange<ui64> seconds;
        seconds.From = timestamp.From ? MakeMaybe<ui32>(timestamp.From->Seconds()) : Nothing();
        seconds.To = timestamp.To ? MakeMaybe<ui32>(timestamp.To->Seconds()) : Nothing();
        queryOptions.SetGenericCondition("timestamp", seconds);
    }
    if (query.HasLimit()) {
        queryOptions.SetLimit(query.GetLimitRef());
    }
    queryOptions.SetOrderBy({
        "timestamp DESC"
    });
    return queryOptions;
}

void FillQueryOptions(NSQL::TQueryOptions& queryOptions, const TVector<NDrive::TSensorId>& sensors) {
    if (!sensors.empty()) {
        TSet<ui64> ids;
        TSet<ui64> subids;
        for (auto&& sensor : sensors) {
            ids.insert(sensor.Id);
            subids.insert(sensor.SubId);
        }
        queryOptions.SetGenericCondition("id", ids);
        queryOptions.SetGenericCondition("subid", subids);
    }
}

NThreading::TFuture<NDrive::TClickHouseClient::TLocationHistoryResult> NDrive::TClickHouseClient::Get(const TLocationHistoryQuery& query, TInstant deadline) const {
    auto queryOptions = CreateQueryOptions(query);
    auto block = Yensured(Client)->Select(NDrive::GetLocationHistoryClickHouseTable(), queryOptions, deadline);
    auto locations = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TLocationHistoryClickHouseRecords records;
        Extract(block, records);
        TLocationHistoryResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
            if (record.Imei) {
                result[record.Imei].push_back(std::move(record));
            }
        }
        return result;
    });
    return locations;
}

NThreading::TFuture<NDrive::TClickHouseClient::TSensorHistoryResult> NDrive::TClickHouseClient::Get(const TSensorHistoryQuery& query, TInstant deadline) const {
    auto queryOptions = CreateQueryOptions(query);
    FillQueryOptions(queryOptions, query.GetSensors());

    auto block = Yensured(Client)->Select(NDrive::GetSensorHistoryClickHouseTable(), queryOptions, deadline);
    auto sensors = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TSensorHistoryClickHouseRecords records;
        Extract(block, records);
        TSensorHistoryResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
            if (record.Imei) {
                result[record.Imei].push_back(std::move(record));
            }
        }
        return result;
    });
    return sensors;
}

NThreading::TFuture<NDrive::TClickHouseClient::TLocationResult> NDrive::TClickHouseClient::Get(const TLocationQuery& query, TInstant deadline) const {
    NSQL::TQueryOptions queryOptions;
    if (const auto& imeis = query.GetImeis(); !imeis.empty()) {
        queryOptions.SetGenericCondition("imei", imeis);
    }
    if (const auto& names = query.GetNames(); !names.empty()) {
        queryOptions.SetGenericCondition("name", names);
    }

    auto block = Yensured(Client)->Select(NDrive::GetSensorClickHouseTable(), queryOptions, deadline);
    auto locations = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TLocationClickHouseRecords records;
        Extract(block, records);
        TLocationResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
            if (record.Imei) {
                result[record.Imei].push_back(std::move(record));
            }
        }
        return result;
    });
    return locations;
}

NThreading::TFuture<NDrive::TClickHouseClient::TSensorResult> NDrive::TClickHouseClient::Get(const TSensorQuery& query, TInstant deadline) const {
    NSQL::TQueryOptions queryOptions;
    if (const auto& imeis = query.GetImeis(); !imeis.empty()) {
        queryOptions.SetGenericCondition("imei", imeis);
    }
    FillQueryOptions(queryOptions, query.GetSensors());

    auto block = Yensured(Client)->Select(NDrive::GetSensorClickHouseTable(), queryOptions, deadline);
    auto sensors = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TSensorClickHouseRecords records;
        Extract(block, records);
        TSensorResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
            if (record.Imei) {
                result[record.Imei].push_back(std::move(record));
            }
        }
        return result;
    });
    return sensors;
}

NDrive::TClickHousePusher::TClickHousePusher(TConstArrayRef<NClickHouse::TClientOptions> options, const TBalancingOptions& balancingOptions)
    : BalancingOptions(balancingOptions)
    , ClientOptions(MakeVector<NClickHouse::TClientOptions>(options))
    , CurrentReplicaIndex(0)
    , QueueActive(true)
{
    Y_ENSURE(!ClientOptions.empty());
    if (ClientOptions.size() == 1) {
        InitializeClient();
    }
    InitializeProcessing();
}

NDrive::TClickHousePusher::TClickHousePusher(const NClickHouse::TClientOptions& options, const TBalancingOptions& balancingOptions)
    : TClickHousePusher(NContainer::Scalar(options), balancingOptions)
{
}

const NClickHouse::TClientOptions& NDrive::TClickHousePusher::GetCurrentClientOptions() const {
    const auto index = CurrentReplicaIndex.load() % ClientOptions.size();
    return ClientOptions[index];
}

void NDrive::TClickHousePusher::InitializeClient() {
    const auto& options = GetCurrentClientOptions();
    Client.Destroy();
    Client = MakeHolder<NClickHouse::TClient>(options);
}

void NDrive::TClickHousePusher::InitializeProcessing() {
    Processing = SystemThreadFactory()->Run([this] {
        TQueue local;
        auto waitTimeout = BalancingOptions.GetSuccessBackoff();
        bool resetConnection = false;
        while (QueueActive.test()) try {
            if (!Client || resetConnection) {
                InitializeClient();
                resetConnection = false;
            }
            while (Queue.size() > 0 || QueueReady.WaitT(waitTimeout)) {
                if (local.empty()) {
                    auto guard = Guard(QueueLock);
                    local = std::move(Queue);
                }
                NClickHouse::TSimpleBlock locationHistoryBlock;
                NClickHouse::TSimpleBlock locationBlock;
                NClickHouse::TSimpleBlock sensorHistoryBlock;
                NClickHouse::TSimpleBlock sensorBlock;
                for (auto&& element : local) {
                    if (element.Sent) {
                        continue;
                    }
                    try {
                        if (std::holds_alternative<TLocationHistoryClickHouseRecord>(element.Record)) {
                            Append(locationHistoryBlock, std::get<TLocationHistoryClickHouseRecord>(element.Record));
                        }
                        if (std::holds_alternative<TLocationClickHouseRecord>(element.Record)) {
                            Append(locationBlock, std::get<TLocationClickHouseRecord>(element.Record));
                        }
                        if (std::holds_alternative<TSensorHistoryClickHouseRecord>(element.Record)) {
                            Append(sensorHistoryBlock, std::get<TSensorHistoryClickHouseRecord>(element.Record));
                        }
                        if (std::holds_alternative<TSensorClickHouseRecord>(element.Record)) {
                            Append(sensorBlock, std::get<TSensorClickHouseRecord>(element.Record));
                        }
                    } catch (const std::exception& e) {
                        auto exception = std::current_exception();
                        NJson::TJsonValue info = NJson::TMapBuilder
                            ("record", NJson::VariantToJson(element.Record))
                            ("exception", NJson::GetExceptionInfo(exception, /*forceBacktrace=*/true))
                        ;
                        ERROR_LOG << "ClickHousePusherElement exception: " << info.GetStringRobust() << Endl;
                        if (element.Promise.Initialized() && !element.Promise.HasValue() && !element.Promise.HasException()) {
                            element.Promise.SetException(exception);
                        }
                        element.Processed = true;
                    }
                }
                if (NClickHouse::TBlock block = locationHistoryBlock; block.GetRowCount() > 0) {
                    INFO_LOG << "ClickHousePusher " << GetCurrentClientOptions().Host << ": sending " << block.GetRowCount() << " locations" << Endl;
                    Y_ENSURE(Client);
                    Client->Insert(NDrive::GetLocationHistoryClickHouseTable(), block);
                }
                if (NClickHouse::TBlock block = sensorHistoryBlock; block.GetRowCount() > 0) {
                    INFO_LOG << "ClickHousePusher " << GetCurrentClientOptions().Host << ": sending " << block.GetRowCount() << " sensors" << Endl;
                    Y_ENSURE(Client);
                    Client->Insert(NDrive::GetSensorHistoryClickHouseTable(), block);
                }
                if (NClickHouse::TBlock block = sensorBlock; block.GetRowCount() > 0) {
                    INFO_LOG << "ClickHousePusher " << GetCurrentClientOptions().Host << ": sending " << block.GetRowCount() << " sensors" << Endl;
                    Y_ENSURE(Client);
                    Client->Insert(NDrive::GetSensorClickHouseTable(), block);
                }
                for (auto&& element : local) {
                    element.Sent = true;
                }
                for (auto&& element : local) {
                    if (element.Processed) {
                        continue;
                    }
                    if (element.Promise.Initialized() && !element.Promise.HasValue() && !element.Promise.HasException()) {
                        element.Promise.SetValue();
                    }
                    element.Processed = true;
                }
                local.clear();
                Sleep(waitTimeout);
            }
        } catch (const NClickHouse::TServerException& e) {
            ERROR_LOG << "ClickHousePusher " << GetCurrentClientOptions().Host << ": an exception occurred: " << NJson::ToJson(e.GetException()).GetStringRobust() << Endl;
            resetConnection = true;
            ++CurrentReplicaIndex;
            Sleep(BalancingOptions.GetFailureBackoff());
        } catch (const std::exception& e) {
            ERROR_LOG << "ClickHousePusher " << GetCurrentClientOptions().Host << ": an exception occurred: " << CurrentExceptionInfo(/*forceBacktrace=*/true).GetStringRobust() << Endl;
            resetConnection = true;
            ++CurrentReplicaIndex;
            Sleep(BalancingOptions.GetFailureBackoff());
        }
    });
}

NDrive::TClickHousePusher::~TClickHousePusher() {
    QueueActive.clear();
    QueueReady.Signal();
    if (Processing) {
        Processing->Join();
    }
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(const TString& objectId, const TString& imei, const TGpsLocation& location, TInstant received) const {
    NDrive::TLocationHistoryClickHouseRecord record(location, imei, objectId, received);
    return Push(std::move(record));
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(const TString& objectId, const TString& imei, const TSensor& sensor, TInstant groupstamp, TInstant received) const {
    NDrive::TSensorHistoryClickHouseRecord record(sensor, imei, objectId, groupstamp, received);
    return Push(std::move(record));
}

template <class T>
NThreading::TFuture<void> NDrive::TClickHousePusher::PushImpl(T&& record) const {
    TQueueItem item;
    item.Record = std::forward<T>(record);
    item.Promise = NThreading::NewPromise<void>();
    auto result = item.Promise.GetFuture();
    {
        auto guard = Guard(QueueLock);
        Queue.push_back(std::move(item));
    }
    QueueReady.Signal();
    return result;
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(TLocationHistoryClickHouseRecord&& record) const {
    return PushImpl(std::move(record));
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(TLocationClickHouseRecord&& record) const {
    return PushImpl(std::move(record));
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(TSensorHistoryClickHouseRecord&& record) const {
    return PushImpl(std::move(record));
}

NThreading::TFuture<void> NDrive::TClickHousePusher::Push(TSensorClickHouseRecord&& record) const {
    return PushImpl(std::move(record));
}
