#pragma once

#include "location.h"
#include "sensor.h"

#include <drive/library/cpp/clickhouse/options.h>

#include <library/cpp/threading/future/future.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/range.h>

#include <util/system/mutex.h>
#include <util/thread/factory.h>

#include <deque>

namespace NClickHouse {
    class TAsyncClient;
    class TClient;
    struct TAsyncClientOptions;
    struct TClientOptions;
}

namespace NDrive {
    class TClickHouseClient {
    public:
        struct TLocationHistoryQuery {
            R_FIELD(TVector<TString>, ObjectIds);
            R_FIELD(TVector<TString>, Imeis);
            R_FIELD(TRange<TInstant>, Timestamp);
            R_OPTIONAL(ui64, Limit);
        };
        struct TSensorHistoryQuery {
            R_FIELD(TVector<TString>, ObjectIds);
            R_FIELD(TVector<TString>, Imeis);
            R_FIELD(TVector<TSensorId>, Sensors);
            R_FIELD(TRange<TInstant>, Timestamp);
            R_OPTIONAL(ui64, Limit);
        };
        struct TLocationQuery {
            R_FIELD(TVector<TString>, Imeis);
            R_FIELD(TVector<TString>, Names);
        };
        struct TSensorQuery {
            R_FIELD(TVector<TString>, Imeis);
            R_FIELD(TVector<TSensorId>, Sensors);
        };

        using TLocationHistoryResult = TMap<TString, TGpsLocations>;
        using TSensorHistoryResult = TMap<TString, TMultiSensor>;
        using TLocationResult = TMap<TString, TLocations>;
        using TSensorResult = TMap<TString, TMultiSensor>;

    public:
        TClickHouseClient(const NClickHouse::TAsyncClientOptions& options);
        ~TClickHouseClient();

        NThreading::TFuture<TVector<TString>> SelectIds(TStringBuf query, TInstant deadline = TInstant::Zero()) const;

        NThreading::TFuture<TLocationHistoryResult> Get(const TLocationHistoryQuery& query, TInstant deadline = TInstant::Zero()) const;
        NThreading::TFuture<TSensorHistoryResult> Get(const TSensorHistoryQuery& query, TInstant deadline = TInstant::Zero()) const;
        NThreading::TFuture<TLocationResult> Get(const TLocationQuery& query, TInstant deadline = TInstant::Zero()) const;
        NThreading::TFuture<TSensorResult> Get(const TSensorQuery& query, TInstant deadline = TInstant::Zero()) const;

    private:
        THolder<NClickHouse::TAsyncClient> Client;
    };

    class TClickHousePusher {
    public:
        struct TBalancingOptions {
            R_FIELD(TDuration, SuccessBackoff, TDuration::Seconds(3));
            R_FIELD(TDuration, FailureBackoff, TDuration::Seconds(5));
        };

    public:
        TClickHousePusher(TConstArrayRef<NClickHouse::TClientOptions> options, const TBalancingOptions& balancingOptions);
        TClickHousePusher(const NClickHouse::TClientOptions& options, const TBalancingOptions& balancingOptions);
        ~TClickHousePusher();

        NThreading::TFuture<void> Push(const TString& objectId, const TString& imei, const TGpsLocation& location, TInstant received) const;
        NThreading::TFuture<void> Push(const TString& objectId, const TString& imei, const TSensor& sensor, TInstant groupstamp, TInstant received) const;
        NThreading::TFuture<void> Push(TLocationHistoryClickHouseRecord&& record) const;
        NThreading::TFuture<void> Push(TLocationClickHouseRecord&& record) const;
        NThreading::TFuture<void> Push(TSensorHistoryClickHouseRecord&& record) const;
        NThreading::TFuture<void> Push(TSensorClickHouseRecord&& record) const;

    private:
        struct TQueueItem {
            std::variant<
                std::monostate,
                TLocationClickHouseRecord,
                TLocationHistoryClickHouseRecord,
                TSensorClickHouseRecord,
                TSensorHistoryClickHouseRecord
            > Record;
            NThreading::TPromise<void> Promise;
            bool Processed = false;
            bool Sent = false;
        };
        using TQueue = std::deque<TQueueItem>;

    private:
        const NClickHouse::TClientOptions& GetCurrentClientOptions() const;
        void InitializeClient();
        void InitializeProcessing();

        template <class T>
        NThreading::TFuture<void> PushImpl(T&& record) const;

    private:
        TBalancingOptions BalancingOptions;
        TVector<NClickHouse::TClientOptions> ClientOptions;
        THolder<NClickHouse::TClient> Client;
        THolder<IThreadFactory::IThread> Processing;

        std::atomic<size_t> CurrentReplicaIndex;
        std::atomic_flag QueueActive;
        TMutex QueueLock;
        mutable TAutoEvent QueueReady;
        mutable TQueue Queue;
    };
}
