#pragma once

#include <library/cpp/clickhouse/client/block.h>
#include <library/cpp/clickhouse/client/columns/date.h>
#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/columns/string.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>

#include <string_view>
#include <tuple>

namespace NDrive {
    namespace NPrivate {
        template <size_t N>
        struct TStringLiteral {
            constexpr TStringLiteral(const char (&value)[N]) {
                static_assert(N > 0);
                std::copy_n(value, N, Value);
            }
            constexpr operator std::string_view() const {
                return { Value, N - 1 };
            }

            char Value[N];
        };
    }

    template <class T, size_t Index, NPrivate::TStringLiteral Name>
    struct TClickHouseColumn {
    public:
        using TType = T;

    public:
        explicit operator bool() const {
            return Value != nullptr;
        }
        inline auto operator->() const {
            return Yensured(Value.Get());
        }
        inline auto operator->() {
            return Yensured(Value.Get());
        }

        TIntrusivePtr<T> Create() const {
            return T::Create();
        }
        template <class TBlock>
        TIntrusivePtr<T> Extract(const TBlock& block) const {
            if (Index >= block.GetColumnCount()) {
                return nullptr;
            }
            return block[Index]->template As<T>();
        }
        constexpr size_t GetIndex() const {
            return Index;
        }
        constexpr std::string_view GetName() const {
            return Name;
        }

    public:
        TIntrusivePtr<T> Value;
    };

    struct TLocationClickHouseCtx {
        TClickHouseColumn<NClickHouse::TColumnString,   0,  "imei"> ImeiColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   1,  "name"> NameColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  2,  "latitude"> LatitudeColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  3,  "longitude"> LongitudeColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  4,  "course"> CourseColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 5,  "since"> SinceColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 6,  "timestamp"> TimestampColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   7,  "object_id"> ObjectIdColumn;

        auto Columns() {
            return std::tie(
                ImeiColumn,
                NameColumn,
                LatitudeColumn,
                LongitudeColumn,
                CourseColumn,
                SinceColumn,
                TimestampColumn,
                ObjectIdColumn
            );
        }
    };

    struct TLocationHistoryClickHouseCtx {
        TClickHouseColumn<NClickHouse::TColumnString,   0,  "object_id"> ObjectIdColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   1,  "imei"> ImeiColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 2,  "timestamp"> TimestampColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 3,  "received"> ReceivedColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  4,  "latitude"> LatitudeColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  5,  "longitude"> LongitudeColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  6,  "course"> CourseColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt32,   7,  "speed"> SpeedColumn;

        auto Columns() {
            return std::tie(
                ObjectIdColumn,
                ImeiColumn,
                TimestampColumn,
                ReceivedColumn,
                LatitudeColumn,
                LongitudeColumn,
                CourseColumn,
                SpeedColumn
            );
        }
    };

    struct TSensorClickHouseCtx {
        TClickHouseColumn<NClickHouse::TColumnString,   0,  "imei"> ImeiColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt16,   1,  "id"> IdColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt16,   2,  "subid"> SubIdColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt64,   3,  "value_int"> ValueIntColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  4,  "value_float"> ValueFloatColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   5,  "value_string"> ValueStringColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 6,  "since"> SinceColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 7,  "timestamp"> TimestampColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   8,  "object_id"> ObjectIdColumn;

        auto Columns() {
            return std::tie(
                ImeiColumn,
                IdColumn,
                SubIdColumn,
                ValueIntColumn,
                ValueFloatColumn,
                ValueStringColumn,
                SinceColumn,
                TimestampColumn,
                ObjectIdColumn
            );
        }
    };

    struct TSensorHistoryClickHouseCtx {
        TClickHouseColumn<NClickHouse::TColumnString,   0,  "object_id"> ObjectIdColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   1,  "imei"> ImeiColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt16,   2,  "id"> IdColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt16,   3,  "subid"> SubIdColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 4,  "timestamp"> TimestampColumn;
        TClickHouseColumn<NClickHouse::TColumnUInt64,   5,  "value_int"> ValueIntColumn;
        TClickHouseColumn<NClickHouse::TColumnFloat64,  6,  "value_float"> ValueFloatColumn;
        TClickHouseColumn<NClickHouse::TColumnString,   7,  "value_string"> ValueStringColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 8,  "groupstamp"> GroupstampColumn;
        TClickHouseColumn<NClickHouse::TColumnDateTime, 9,  "received"> ReceivedColumn;

        auto Columns() {
            return std::tie(
                ObjectIdColumn,
                ImeiColumn,
                IdColumn,
                SubIdColumn,
                TimestampColumn,
                ValueIntColumn,
                ValueFloatColumn,
                ValueStringColumn,
                GroupstampColumn,
                ReceivedColumn
            );
        }
    };

    template <class TCtx, class TBlock>
    void CreateColumns(TCtx&& ctx, TBlock& block) {
        auto columns = ctx.Columns();
        ForEach(columns, [&](auto&& column) {
            if (block.GetColumnCount() <= column.GetIndex()) {
                block.AppendColumn(ToString(column.GetName()), column.Create());
            }
        });
    }

    template <class TCtx, class TBlock>
    void FillColumns(TCtx&& ctx, const TBlock& block) {
        auto columns = ctx.Columns();
        ForEach(columns, [&](auto&& column) {
            if (!column) {
                column.Value = column.Extract(block);
            }
        });
    }

    template <class TCtx, class TBlock>
    void Initialize(TCtx&& ctx, TBlock& block) {
        CreateColumns(ctx, block);
        FillColumns(ctx, block);
    }
}
